package com.xebialabs.deployit.itest.cloudhost;

import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.xebialabs.overcast.host.CloudHost;
import com.xebialabs.overcast.host.CloudHostFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;

public class ItestHostLauncher {
    private static Logger LOG = LoggerFactory.getLogger(ItestHostLauncher.class);
    public static final String MDC_KEY = "ProvisionedVm";

    protected final ListeningExecutorService vmExecutor;

    protected final Map<String, ListenableFuture<CloudHost>> hostRegistry = newHashMap();
    protected List<CloudHostFutureTransformer> transformers = newArrayList();

    private static ItestHostLauncher singleton = null;

    public static synchronized ItestHostLauncher getInstance() {
        if (singleton != null) {
            return singleton;
        }
        singleton = new ItestHostLauncher();
        return singleton;
    }

    protected ItestHostLauncher() {
        vmExecutor = MoreExecutors.listeningDecorator(Executors.newCachedThreadPool());
        addVirtualMachineShutdownHook();
    }

    public void addTransformer(CloudHostFutureTransformer transformer) {
        transformers.add(transformer);
    }

    public synchronized ListenableFuture<CloudHost> launch(final String host) {
        if (hostRegistry.containsKey(host)) {
            LOG.debug("Returning existing cloudhost for {}", host);
            return hostRegistry.get(host);
        }

        LOG.debug("launcher: submit future {}", host);
        ListenableFuture<CloudHost> future = vmExecutor.submit(() -> {
            try {
                MDC.put(MDC_KEY, host);

                LOG.info("itest host '{}' starting", host);
                CloudHost cloudHost = CloudHostFactory.getCloudHost(host);
                LOG.debug("itest host '{}' setup", host);
                cloudHost.setup();
                LOG.info("itest host '{}' started", host);
                return cloudHost;
            } finally {
                MDC.remove(MDC_KEY);
            }
        });
        LOG.debug("launcher: adding transformers {}", host);
        for (CloudHostFutureTransformer t : transformers) {
            future = Futures.transform(future, t.transform(future));
        }
        LOG.debug("launcher: registering {}", host);
        hostRegistry.put(host, future);
        return future;
    }

    public synchronized ListenableFuture<CloudHost> getCloudHostFuture(String host) {
        LOG.debug("Looking up future for host {}", host);
        if (!hostRegistry.containsKey(host)) {
            throw new NoSuchElementException(String.format("No such host '%s' launch it first", host));
        }
        return hostRegistry.get(host);
    }

    public synchronized void shutdown() {
        LOG.info("Shutting down ItestHostLauncher");
        vmExecutor.shutdown();
        shutdownHosts();
    }

    public void addVirtualMachineShutdownHook() {
        Runtime.getRuntime().addShutdownHook(new Thread() {
            @Override
            public void run() {
                LOG.debug("Shutdown hook: shutting down virtual machines");
                shutdown();
                LOG.debug("Shutdown hook: completed.");
            }
        });
    }

    private void shutdownHosts() {
        for (ListenableFuture<CloudHost> h : hostRegistry.values()) {
            CloudHost ch;
            try {
                ch = h.get();
                LOG.debug("Shutting down itest host '{}'", ch.getHostName());
                ch.teardown();
            } catch (InterruptedException | ExecutionException e) {
                LOG.debug("Shutting down Executor.");
            }
        }
    }
}
