package com.xebialabs.xlplatform.endpoints.routes

import java.net.URL

import akka.http.scaladsl.server.{PathMatcher0, PathMatchers}
import com.xebialabs.xlplatform.utils.ClassLoaderUtils.classLoader
import grizzled.slf4j.Logging

import scala.xml.Elem

trait EndpointDefinition {
  def path: String

  def getPathMatcher: PathMatcher0 = PathMatchers.separateOnSlashes(cleanPath)

  def cleanPath: String = {
    path.split('/').filterNot(_.isEmpty).mkString("/")
  }
}
case class ScalaEndpoint(path: String, className: String) extends EndpointDefinition
case class ScriptEndpoint(path: String, method: String, script: URL) extends EndpointDefinition

object ScalaEndpoint extends Logging {
  def apply(path: String, elem: Elem): ScalaEndpoint = {
    val route: String = (elem \ "@route").text
    debug(s"Found configured extension route: $route")
    ScalaEndpoint(path, route)
  }
}

object ScriptEndpoint extends Logging {

  def apply(path: String, elem: Elem): ScriptEndpoint = {
    ScriptEndpoint(path, (elem \ "@method").text, findFile((elem \ "@script").text))
  }

  private def findFile(p: String): URL = {
    Option(classLoader.getResource(p)) match {
      case Some(url) => url
      case None => throw new IllegalArgumentException(s"Can not find script [$p]")
    }
  }
}

object ScriptEndpoints extends Logging {
  import com.xebialabs.xlplatform.utils.ResourceUtils._
  def apply(config: String): Seq[ScriptEndpoint] = {
    (loadXmlResources(config).values.toSeq \ "endpoint").collect({case e: Elem if e.attribute("script").isDefined =>
      val path: String = (e \ "@path").text
      debug(s"Found dynamic script route mounted on path [$path]")
      ScriptEndpoint(path, e)
    })
  }
}


object ScalaEndpoints extends Logging {
  import com.xebialabs.xlplatform.utils.ResourceUtils._
  def apply(config: String): Seq[ScalaEndpoint] = {
    (loadXmlResources(config).values.toSeq \ "route").collect({case e: Elem if e.attribute("route").isDefined =>
      val path: String = (e \ "@path").text
      debug(s"Found dynamic route mounted on path [$path]")
      ScalaEndpoint(path, e)
    })
  }
}
