package com.xebialabs.xlplatform.endpoints.routes

import akka.actor.ActorSystem
import akka.http.scaladsl.server.{Directives, RequestContext}
import com.xebialabs.xlplatform.endpoints.{AuthenticatedRoute, _}
import com.xebialabs.xlplatform.settings.XlPlatformSettings
import grizzled.slf4j.Logging

import scala.util.{Failure, Try}

trait ScalaExtensionRoute extends Directives with Logging {
  def settings: XlPlatformSettings

  private[this] var routeSources: Option[Seq[ExtensionRoutes]] = None

  def codedExtensionRoutes(file: String)(implicit system: ActorSystem): Seq[AuthenticatedRoute] = {
    if (routeSources.isEmpty) {
      val endpoints: Seq[ScalaEndpoint] = ScalaEndpoints(file)
      debug(s"Going to register ${endpoints.size} scala endpoints")
      routeSources = Option(endpoints.map { case ep =>
        debug(s"Using class ${ep.className} to generate extension endpoints function for ${ep.path}.")
        try {
          Thread.currentThread().getContextClassLoader.loadClass(ep.className).newInstance().asInstanceOf[ExtensionRoutes]
        } catch {
          case _: InstantiationException =>
            throw new IllegalArgumentException(s"Unable to create a new instance of ${ep.className}. Please make sure that it has a default non-arg constructor.")
          case _: ClassCastException =>
            throw new IllegalArgumentException(s"Class ${ep.className} cannot be used to collect extension routes. Please make sure that it extends ${classOf[ExtensionRoutes].getCanonicalName}")
        }
      })
    }

    val routesByAuth = routeSources.get.flatMap { rs =>
      Try(rs.route(system)).recoverWith {
        case e =>
          logger.error(s"Could not create route ${rs.getClass.getName}", e)
          Failure(e)
      }.toOption
    }
    if (routesByAuth.nonEmpty) {
      Seq((auth: AuthenticatedData) => (ctx: RequestContext) =>
        routesByAuth.map(_.apply(auth)).reduceLeft(_ ~ _) (ctx)
      )
    } else {
      Seq.empty
    }
  }

  def codedExtensionRoutes(implicit system: ActorSystem): Seq[AuthenticatedRoute] = {
    codedExtensionRoutes(settings.ServerExtension.file)
  }
}
