package com.xebialabs.xlplatform.endpoints.routes

import akka.actor.{ActorRef, ActorRefFactory}
import akka.http.scaladsl.model._
import akka.http.scaladsl.model.headers.RawHeader
import akka.http.scaladsl.server._
import akka.pattern.{AskTimeoutException, ask}
import com.xebialabs.platform.script.jython.ScriptSource
import com.xebialabs.xlplatform.endpoints.{AuthenticatedRoute, _}
import com.xebialabs.xlplatform.endpoints.actors.{JythonScriptExecutorActor, RunScript, ScriptDone}
import com.xebialabs.xlplatform.endpoints.json.{ScriptRequest, ScriptResponse, ScriptResponseProtocol}
import com.xebialabs.xlplatform.settings.XlPlatformSettings
import grizzled.slf4j.Logging
import spray.json.{JsNull, JsValue}

import scala.language.postfixOps
import scala.util.{Failure, Success}

trait ScriptExtensionRoute extends Directives with ScriptResponseProtocol with Logging {

  implicit def actorRefFactory: ActorRefFactory

  def settings: XlPlatformSettings

  implicit val JsValueUnmarshaller = rootFormat(JsValueFormat)

  private def jythonActor(): ActorRef = actorRefFactory.actorOf(JythonScriptExecutorActor.props())

  private def createEndpointRoute(ep: ScriptEndpoint, auth: AuthenticatedData, body: JsValue = JsNull): Route = {
    debug(s"Using script ${ep.script} to generate extension ${ep.method} endpoint ${ep.path}.")
    rawPathPrefix(segments(settings.ServerExtension.scriptsPathPrefix)) {
      path(ep.getPathMatcher) {
        parameterMultiMap { params =>
          implicit val timeout = settings.ServerExtension.timeout
          onComplete(jythonActor ? RunScript(ScriptSource.byUrl(ep.script), ScriptRequest(params, body), auth)) {
            case Success(d@ScriptDone(_, _, _, statusCode, headers, None)) =>
              val returnCode = statusCode.map(StatusCode.int2StatusCode(_)).getOrElse(StatusCodes.OK)
              respondWithHeaders(headers.map(p => RawHeader(p._1, p._2)).toList) {
                complete(returnCode -> ScriptResponse(d))
              }
            case Success(d@ScriptDone(_, _, _, _, _, Some(e))) => complete(StatusCodes.InternalServerError, ScriptResponse(d))
            case Failure(e) if e.isInstanceOf[AskTimeoutException] => complete(StatusCodes.RequestTimeout, e)
            case Failure(e) => complete(StatusCodes.InternalServerError, e)
          }
        }
      }
    }
  }

  def optionalJsEndpoint(endpoint: ScriptEndpoint, auth: AuthenticatedData): Route = {
    extractRequest { req =>
      if (req.entity.contentLengthOption.contains(0L)) {
        createEndpointRoute(endpoint, auth, JsNull)
      } else {
        entity(as[JsValue])(createEndpointRoute(endpoint, auth, _))
      }
    }
  }

  def customScriptEndpoint(endpoint: ScriptEndpoint)(auth: AuthenticatedData): Route = {
    endpoint.method match {
      case "GET" => get {
        createEndpointRoute(endpoint, auth)
      }
      case "POST" => post {
        optionalJsEndpoint(endpoint, auth)
      }
      case "PUT" => put {
        optionalJsEndpoint(endpoint, auth)
      }
      case "DELETE" => delete {
        createEndpointRoute(endpoint, auth)
      }
    }
  }

  def customScriptEndpoints(endpoints: Seq[ScriptEndpoint]): Seq[AuthenticatedRoute] = {
    debug(s"Going to register ${endpoints.size} script endpoints")
    endpoints.map(customScriptEndpoint)
  }

  def customScriptEndpoints: Seq[AuthenticatedRoute] = customScriptEndpoints(ScriptEndpoints(settings.ServerExtension.file))

}
