package com.xebialabs.xlplatform.endpoints.json

import com.xebialabs.deployit.core._
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind._
import com.xebialabs.deployit.plugin.api.reflect.{Descriptor, PropertyDescriptor}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.deployit.util.PasswordEncrypter
import spray.json._

import java.util.{Collection => JCollection, Date => JDate, Map => JMap}
import scala.collection.immutable.ListMap
import scala.jdk.CollectionConverters._

trait PasswordEncrypterProvider {
  def passwordEncrypter: PasswordEncrypter

  def encrypt(s: String): String = passwordEncrypter.ensureEncrypted(s)

  def isEncrypted(s: String): Boolean = passwordEncrypter.isEncodedAndDecryptable(s)
}

trait CiJsonProtocol extends DefaultJsonProtocol {
  self: PasswordEncrypterProvider =>

  implicit object StringValueFormat extends JsonFormat[StringValue] {
    override def write(obj: StringValue): JsValue = obj match {
      case e: EncryptedStringValue => JsString(encrypt(e.toString))
      case v: StringValue => JsString(v.toString)
    }

    override def read(json: JsValue): StringValue = json match {
      case JsString(v) if passwordEncrypter.isEncodedAndDecryptable(v) => new EncryptedStringValue(passwordEncrypter.decrypt(v))
      case JsString(v) => new StringValue(v)
      case _ => deserializationError("Expected a (possibly encrypted) StringValue as JsString, but got " + json)
    }
  }

  def getOption(ci: ConfigurationItem, pd: PropertyDescriptor): Option[AnyRef] = Option(pd.get(ci))

  class ConfigurationItemFormat(depth: Int = 1) extends RootJsonFormat[ConfigurationItem] {
    override def read(json: JsValue): BaseConfigurationItem = ???

    override def write(ci: ConfigurationItem): JsValue = {
      writeCi(ci, depth)
    }

    def writeCi(ci: ConfigurationItem, depth: Int, seen: Set[String] = Set()): JsValue = {
      val descriptor: Descriptor = ci.getType.getDescriptor

      def writeBasics: Map[String, Option[JsValue]] = ListMap(
        "id" -> Option(ci.getId).map(JsString(_)),
        "type" -> Option(ci.getType).map(t => JsString(t.toString)))

      def writeAll: Map[String, Option[JsValue]] = {
        def getTokenOption: Option[JsValue] = ci match {
          case bci: BaseConfigurationItem => Option(bci.get$token()).map(JsString(_))
          case _ => None
        }

        def asList[A](c: Any): List[A] = c match {
          case jc: JCollection[_] => jc.asInstanceOf[JCollection[A]].asScala.toList
          case s: Iterable[_] => s.asInstanceOf[Iterable[A]].toList
          case _ => throw new IllegalArgumentException(s"Do not know how to convert [$c] to a List[_]")
        }

        def asVector[A](c: Any): Vector[A] = asList[A](c).toVector

        def writeSub(sub: ConfigurationItem, depth: Int) = writeCi(sub, depth, seen + ci.getId)

        def asListStringValue(c: Any): List[StringValue] = c match {
          case losv: ListOfStringView => losv.getWrapped.asScala.toList
          case sosv: SetOfStringView => sosv.getWrapped.asScala.toList
          case _ => asList[String](c).map(new StringValue(_))
        }


        def asMapStringStringValue(c: Any): Map[String, StringValue] = c match {
          case mssv: MapStringStringView => mssv.getWrapped.asScala.toMap
          case jm: JMap[_, _] => jm.asInstanceOf[JMap[String, String]].asScala.toMap[String, String].view.mapValues(new StringValue(_)).toMap
          case sm: Map[_, _] => sm.asInstanceOf[Map[String, String]].view.mapValues(new StringValue(_)).toMap
          case _ => throw new IllegalArgumentException(s"Do not know how to convert [$c] to a Map[String, StringValue]")
        }

        val jsonProperties: Iterable[(String, Option[JsValue])] = descriptor.getPropertyDescriptors.asScala.map { pd =>
          (pd.getName, pd.getKind match {
            case BOOLEAN => getOption(ci, pd).map(b => JsBoolean(b.asInstanceOf[Boolean]))

            case INTEGER => getOption(ci, pd).map(i => JsNumber(i.asInstanceOf[Integer]))

            case STRING if pd.isPassword => getOption(ci, pd).map(s => JsString(encrypt(s.toString)))

            case STRING => getOption(ci, pd).map(s => JsString(s.toString))

            case ENUM => getOption(ci, pd).map(e => JsString(e.toString))

            case CI => getOption(ci, pd).map(ci => writeSub(ci.asInstanceOf[ConfigurationItem], 0))

            case LIST_OF_STRING | SET_OF_STRING => getOption(ci, pd).map(asListStringValue).map(_.toJson)

            case LIST_OF_CI | SET_OF_CI if pd.isAsContainment => getOption(ci, pd).map(loc => JsArray(asVector[ConfigurationItem](loc).map(writeSub(_, depth - 1))))

            case LIST_OF_CI | SET_OF_CI => getOption(ci, pd).map(loc => JsArray(asVector[ConfigurationItem](loc).map(writeSub(_, 0))))

            case MAP_STRING_STRING  => Option(getOption(ci, pd).map(asMapStringStringValue).toJson)

            case DATE => getOption(ci, pd).map(d => JsNumber(d.asInstanceOf[JDate].getTime))
          })
        }

        writeBasics.+(("$token", getTokenOption)).++(jsonProperties).filter(_._2.isDefined)
      }

      depth match {
        case 0 => writeBasics.toJson
        case _ if seen.contains(ci.getId) => writeBasics.toJson
        case _ => writeAll.toJson
      }
    }
  }

}
