package com.xebialabs.license;

import java.util.*;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

public class LicensePropertyMap {
    public static final DateTimeFormatter LICENSE_DATE_FORMAT = DateTimeFormat.forPattern("yyyy-MM-dd");

    private Map<LicenseProperty, Object> licenseProperties = new HashMap<>();

    public void set(LicenseProperty prop, String value) throws LicenseParseException {
        if (value == null || value.trim().isEmpty()) {
            return;
        }

        switch (prop.getType()) {
            case STRING:
                licenseProperties.put(prop, value);
                break;
            case DATE:
                try {
                    licenseProperties.put(prop, LocalDate.parse(value, LICENSE_DATE_FORMAT));
                } catch (Exception e) {
                    throw new LicenseParseException(String.format("Unable to parse '%s' '%s'", prop.getName().toLowerCase(), value));

                }
                break;
            case MAP_STRING_INTEGER:
                Map<String, Integer> msi = get(prop);
                try {
                    String[] pairStringInt = value.trim().split("=");
                    Integer intValue = Integer.parseInt(pairStringInt[1].trim());
                    String stringValue = pairStringInt[0].trim();
                    msi.put(stringValue, intValue);
                } catch (Exception e) {
                    throw new LicenseParseException(String.format("Unable to parse '%s' '%s'", prop.getName().toLowerCase(), value));
                }
                break;
            case LIST_OF_STRINGS:
                List<String> set = get(prop);
                set.add(value);
                break;
            default:
                throw new IllegalArgumentException("Unknown License property type " + prop.getType());
        }
    }

    @SuppressWarnings("unchecked")
    public <T> T get(LicenseProperty prop) {
        switch (prop.getType()) {
            case STRING:
            case DATE:
                return (T) licenseProperties.get(prop);
            case MAP_STRING_INTEGER:
                if (!licenseProperties.containsKey(prop)) {
                    licenseProperties.put(prop, new LinkedHashMap<String, Integer>());
                }
                return (T) licenseProperties.get(prop);
            case LIST_OF_STRINGS:
                if (!licenseProperties.containsKey(prop)) {
                    licenseProperties.put(prop, new ArrayList<String>());
                }
                return (T) licenseProperties.get(prop);
            default:
                throw new IllegalArgumentException("Unknown License property type " + prop.getType());
        }
    }

    public String getAsString(LicenseProperty prop) {
        switch (prop.getType()) {
            case STRING:
                return (String) get(prop);
            case DATE:
                LocalDate propValue = get(prop);
                if (propValue != null) {
                    return LICENSE_DATE_FORMAT.print(propValue);
                }
                return null;
            case MAP_STRING_INTEGER:
                Map<String, Integer> msi = get(prop);
                if (msi.isEmpty()) {
                    return null;
                }
                StringBuilder b = new StringBuilder();
                for (Map.Entry<String, Integer> entry : msi.entrySet()) {
                    if (b.length() != 0) {
                        b.append(",");
                    }
                    b.append(entry.getKey()).append("=").append(entry.getValue());
                }
                return b.toString();
            case LIST_OF_STRINGS:
                Iterable<String> list = get(prop);
                return String.join(",", list);
            default:
                throw new IllegalArgumentException("Unknown License property type " + prop.getType());
        }
    }

    boolean containsKey(LicenseProperty prop) {
        return licenseProperties.containsKey(prop);
    }

    void format(StringBuilder collector, String template, LicenseProperty prop) {
        if (containsKey(prop)) {
            switch (prop.getType()) {
                case STRING:
                case DATE:
                    collector.append(String.format(template, prop.getName(), licenseProperties.get(prop)));
                    break;
                case MAP_STRING_INTEGER:
                    Map<String, Integer> msi = get(prop);
                    for (String s : msi.keySet()) {
                        collector.append(String.format(template, prop.getName(), s + "=" + msi.get(s)));
                    }
                    break;
                case LIST_OF_STRINGS:
                    List<String> ss = get(prop);
                    for (String s : ss) {
                        collector.append(String.format(template, prop.getName(), s));
                    }
                    break;
            }
        }
    }

    @Override
    public String toString() {
        // used in logging in license generator do not remove nor change without double checking
        // license generator logging is still to spec.
        return licenseProperties.toString();
    }
}
