package com.xebialabs.plugin.manager

import java.nio.file.{Path, Paths}
import akka.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot
import com.xebialabs.plugin.manager.config.ConfigWrapper
import com.xebialabs.plugin.manager.metadata.{ArtifactId, Version, VersionExpr}
import grizzled.slf4j.Logging
import spray.json._

import scala.util.Try
import scala.util.matching.Regex

sealed trait PluginId

object PluginId extends Logging {
  val localGroupId: String = LocalPluginManager.name
  lazy val pattern: Regex = ("""(.+)-(\d.*)\.""" + ConfigWrapper.extension).r
  lazy val patternNoVersion: Regex = ("""(.+).""" + ConfigWrapper.extension).r
  lazy val patternVersionWithNightly: Regex = ("""(.+)-(\d.*-\d.*)\.""" + ConfigWrapper.extension).r

  case class LocalFile private(basename: String, version: Option[Version]) extends PluginId
  case class Artifact private(repository: String,
                              groupId: String,
                              artifactId: String,
                              version: Version,
                              packaging: String,
                              classifier: Option[String]) extends PluginId {
    requireProperFilename(repository, "repository")
    require(repository != LocalPluginManager.name, s"repository ${LocalPluginManager.name} is reserved for manually installed plugins")
    requireProperFilename(groupId, "groupId")
    require(groupId != localGroupId, s"$localGroupId is reserved for manually installed plugins")
    requireProperFilename(artifactId, "artifactId")
    val classifier1: Option[String] = classifier.filter(_.nonEmpty)
    require(classifier1.fold(true)(c => {
      require(!c.matches("\\s+"), "classifier cannot contain spaces")
      true
    }))
  }

  object LocalFile {
    lazy val idPattern: Regex = s"""${LocalPluginManager.name}:$localGroupId:${pattern.toString}""".r
    lazy val idPatternNoVersion: Regex = s"""${LocalPluginManager.name}:$localGroupId:${patternNoVersion.toString}""".r
    lazy val idPatternVersionWithNightly: Regex = s"""${LocalPluginManager.name}:$localGroupId:${patternVersionWithNightly.toString}""".r

    def apply(filename: String): LocalFile = {
      requireProperFilename(filename, "filename")
      require(filename.endsWith(ConfigWrapper.extension), s"filename must end with '.${ConfigWrapper.extension}'")

      filename match {
        case patternVersionWithNightly(basename, versionWithNightly) =>
          new LocalFile(basename, Option(versionWithNightly).flatMap(Version.fromString))
        case pattern(basename, versionString) =>
          new LocalFile(basename, Option(versionString).flatMap(Version.fromString))
        case patternNoVersion(basename) =>
          new LocalFile(basename, None)
        case _ =>
          new LocalFile(filename.stripSuffix(ConfigWrapper.extension), None)
      }
    }

    def apply(basename: String, version: String): LocalFile =
      parsed(basename, Version.fromString(version))

    def parsed(basename: String, version: Option[Version]): LocalFile = {
      requireProperFilename(basename, "basename")
      new LocalFile(basename, version)
    }

    def fromIdString(id: String): Option[LocalFile] = id match {
      case idPatternVersionWithNightly(base, ver) =>
        for {
          basename <- Option(base)
          version = Option(ver).getOrElse("")
          parsed <- Try(LocalFile(s"$basename-$version.${ConfigWrapper.extension}")).toOption
        } yield parsed
      case idPattern(base, ver) =>
        for {
          basename <- Option(base)
          version = Option(ver).getOrElse("")
          parsed <- Try(LocalFile(s"$basename-$version.${ConfigWrapper.extension}")).toOption
        } yield parsed
      case idPatternNoVersion(base) =>
        for {
          basename <- Option(base)
          parsed <- Try(LocalFile.apply(basename, None)).toOption
        } yield parsed
      case _ =>
        None
    }

    trait Protocol extends SprayJsonSupport with DefaultJsonProtocol
      with Version.Protocol {
      val pluginIdLocalFileWriter: RootJsonWriter[LocalFile] = {
        case id@LocalFile(base, version) => JsObject(
          "id" -> id.id.toJson,
          "filename" -> id.filename.toJson,
          "repository" -> LocalPluginManager.name.toJson,
          "basename" -> base.toJson,
          "version" -> (version: Option[Version]).toJson
        )
      }
      val pluginIdLocalFileReader: RootJsonReader[LocalFile] = { jsValue =>
        val maybeLocalFile = jsValue match {
          case obj: JsObject =>
            obj.getFields("id") match {
              case Seq(JsString(id)) => PluginId.LocalFile.fromIdString(id)
              case _ => None
            }
          case JsString(id) => PluginId.LocalFile.fromIdString(id)
          case _ => None
        }
        maybeLocalFile.getOrElse {
          deserializationError(s"Cannot parse PluginId.LocalFile from '$jsValue'")
        }
      }
      implicit val pluginIdLocalFormat: RootJsonFormat[LocalFile] = rootJsonFormat(pluginIdLocalFileReader, pluginIdLocalFileWriter)
    }

  }

  object Artifact {
    lazy val idPattern: Regex = s"""(.+):(.+):$pattern""".r
    lazy val idPatternWithNightly: Regex = s"""(.+):(.+):$patternVersionWithNightly""".r

    def apply(repository: String, groupId: String, artifactId: String, version: String, classifier: Option[String] = None): Artifact = {
      requireProperFilename(version, "version")
      val parsedVersion = Version.fromString(version)
      require(parsedVersion.nonEmpty, s"Invalid version format: $version")
      parsed(repository, groupId, artifactId, parsedVersion.get, classifier)
    }

    def parsed(repository: String, groupId: String, artifactId: String, version: Version, classifier: Option[String] = None): Artifact = {
      new Artifact(repository, groupId, artifactId, version, ConfigWrapper.extension, classifier.filter(_.nonEmpty))
    }

    def fromIdString(id: String): Option[Artifact] = id match {
      case idPatternWithNightly(repo, groupId, artifactId, version) =>
        Some(Artifact.apply(repo, groupId, artifactId, version, None))
      case idPattern(repo, groupId, artifactId, version) =>
        Some(Artifact.apply(repo, groupId, artifactId, version, None))
      case _ => None
    }

    trait Protocol extends SprayJsonSupport with DefaultJsonProtocol
      with Version.Protocol {
      val pluginIdArtifactReader: RootJsonReader[Artifact] = { jsValue =>
        val maybeArtifactId = jsValue match {
          case obj: JsObject =>
            obj.getFields("id") match {
              case Seq(JsString(id)) => PluginId.Artifact.fromIdString(id)
              case _ => None
            }
          case JsString(id) => PluginId.Artifact.fromIdString(id)
          case _ => None
        }
        maybeArtifactId.getOrElse {
          deserializationError(s"Cannot parse PluginId.Artifact from '$jsValue'")
        }
      }
      val pluginIdArtifactWriter: RootJsonWriter[Artifact] = {
        case id@Artifact(repo, groupId, artifactId, version, packaging, _) => JsObject(
          "id" -> id.id.toJson,
          "filename" -> id.filename.toJson,
          "repository" -> repo.toJson,
          "groupId" -> groupId.toJson,
          "artifactId" -> artifactId.toJson,
          "packaging" -> packaging.toJson,
          "version" -> version.toJson
        )
      }
      implicit val pluginIdArtifactFormat: RootJsonFormat[Artifact] = rootJsonFormat(pluginIdArtifactReader, pluginIdArtifactWriter)
    }

    implicit class ArtifactOps(val artifact: Artifact) extends AnyVal {
      def toArtifactId: ArtifactId = ArtifactId(artifact.groupId, artifact.artifactId, artifact.version, Some(artifact.repository))
    }

  }

  def fromIdString(id: String): Option[PluginId] = {
    if (id.startsWith(LocalPluginManager.name + ":")) {
      LocalFile.fromIdString(id)
    } else {
      Artifact.fromIdString(id)
    }
  }

  implicit class PluginIdOps(val pluginId: PluginId) extends AnyVal {

    def fold[B](ifLocal: LocalFile => B, ifArtifact: Artifact => B): B = pluginId match {
      case local: LocalFile => ifLocal(local)
      case artifact: Artifact => ifArtifact(artifact)
    }

    def toArtifactId: ArtifactId = fold(
      l => ArtifactId(PluginId.localGroupId, l.basename, l.version.getOrElse(Version.zero)),
      a => new Artifact.ArtifactOps(a).toArtifactId
    )

    def filename: String = fold(
      l => s"${l.basename}${l.version.fold("")("-" + _.id)}." + ConfigWrapper.extension,
      a => s"${a.artifactId}-${a.version.id}${a.classifier.map("-" + _).getOrElse("")}.${a.packaging}"
    )

    def pluginName: String = fold(
      l => s"${l.basename}",
      a => s"${a.artifactId}"
    )

    def pluginSource: String = fold(
      _ => localGroupId,
      a => s"${a.repository}"
    )

    def groupId: String = fold(
      _ => localGroupId,
      a => a.groupId
    )

    def path: Path = Paths.get(pluginId.pluginRepository)

    def pluginVersion: Option[Version] = fold(
      _.version,
      a => Some(a.version)
    )

    def pluginRepository: String = fold(
      _ => LocalPluginManager.name,
      _.repository
    )

    def id: String = fold(
      _ => s"${LocalPluginManager.name}:$localGroupId:",
      a => s"${a.repository}:${a.groupId}:"
    ) + filename

    def matching(other: PluginId): Boolean =
      fold(
        l => other.fold(_.basename == l.basename, _ => false),
        a => other.fold(_ => false, x => x.groupId == a.groupId && x.artifactId == a.artifactId)
      )

  }

  trait Protocol extends SprayJsonSupport with DefaultJsonProtocol
    with VersionExpr.Protocol
    with LocalFile.Protocol
    with Artifact.Protocol {
    val pluginIdWriter: RootJsonWriter[PluginId] = {
      case l: LocalFile => pluginIdLocalFileWriter.write(l)
      case a: Artifact => pluginIdArtifactWriter.write(a)
    }
    val pluginIdReader: RootJsonReader[PluginId] = { jsValue =>
      val maybePluginId = jsValue match {
        case obj: JsObject =>
          obj.getFields("id") match {
            case Seq(JsString(id)) => PluginId.fromIdString(id)
            case _ => None
          }
        case JsString(id) => PluginId.fromIdString(id)
        case _ => None
      }
      maybePluginId.getOrElse {
        deserializationError(s"Cannot parse PluginId from '$jsValue'")
      }
    }
    implicit val pluginIdFormat: RootJsonFormat[PluginId] = rootJsonFormat(pluginIdReader, pluginIdWriter)
  }

  protected def parseVersionAndPackaging(str: String): Option[(String, String)] = str.split('.').toList match {
    case version :: packaging :: Nil => Some(version -> packaging)
    case _ => None
  }

}


