package com.xebialabs.plugin.manager.metadata

import akka.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import spray.json._

import scala.util.matching.Regex


case class VersionExpr(op: VersionExpr.Op, version: Version)

object VersionExpr {

  sealed trait Op

  object Op {
    case object GEQ extends Op
    case object GT extends Op
    case object LT extends Op
    case object LEQ extends Op
    case object EQ extends Op
  }

  trait Protocol extends SprayJsonSupport with DefaultJsonProtocol
    with Version.Protocol {
    import Op._

    val opString: Op => String = {
      case GEQ => ">="
      case GT => ">"
      case LT => "<"
      case LEQ => "<="
      case EQ => "=="
    }
    val opWriter: JsonWriter[Op] = o => opString(o).toJson
    val opReader: JsonReader[Op] = jsonReader[Op] {
      case JsString(">=") => GEQ
      case JsString(">") => GT
      case JsString("<") => LT
      case JsString("<=") => LEQ
      case JsString("==") => EQ
      case JsString(other) =>
        deserializationError(s"Unknown operator: '$other'")
      case unknown =>
        deserializationError(s"Cannot parse operator: '$unknown'")
    }
    implicit val opFormat: JsonFormat[Op] = jsonFormat(opReader, opWriter)

    val versionExprWriter: JsonWriter[VersionExpr] = jsonWriter[VersionExpr] {
      case VersionExpr(op, version) => (opString(op) ++ " " ++ versionString(version)).toJson
    }
    val versionExprReader: JsonReader[VersionExpr] = jsonReader[VersionExpr] {
      case JsString(str) if str.split(' ').length == 2 =>
        val split = str.split(' ')
        val op = JsString(split(0).trim).convertTo[Op]
        val version = JsString(split(1).trim).convertTo[Version]
        VersionExpr(op, version)
      case v@JsString(str) if str.split(' ').length == 1 =>
        val version = v.convertTo[Version]
        VersionExpr(EQ, version)
      case unknown =>
        deserializationError(s"Cannot parse version expression: '$unknown'")
    }
    implicit val versionExprFormat: JsonFormat[VersionExpr] = jsonFormat(versionExprReader, versionExprWriter)
  }
}

