package com.xebialabs.plugin.manager.repository.storage

import java.io.{File, InputStream}
import java.nio.file.{Path, Paths}

import com.xebialabs.plugin.manager.metadata.{ArtifactId, ExtendedMetadata, PluginsMetadata, Version}
import com.xebialabs.plugin.manager.{LocalPluginManager, PluginId}
import com.xebialabs.xlplatform.io.ZipUtils
import grizzled.slf4j.Logging
import org.apache.commons.io.FileUtils

import scala.collection.mutable


// Note: not purely "MemoryStorage" anymore, since we persist the content of the zip file...
class PluginMetadataMemoryStorage(val repositoryName: String) extends PluginMetadataStorage with Logging {
  protected val cacheDir: Path = Paths.get("cache").resolve("plugins")

  protected val memory: mutable.Map[ArtifactId, PluginsMetadata] = mutable.Map.empty

  override def clear(): Unit = this.synchronized {
    memory.clear()
  }

  // TODO: validate content of zipfile!
  override def storeLogos(data: InputStream): Unit = {
    val dir = cacheDir.resolve(repositoryName).toFile
    FileUtils.deleteDirectory(dir)
    ZipUtils.extract(data, dir)
  }

  override def store(data: Seq[PluginId.Artifact], metadata: Map[ArtifactId, ExtendedMetadata]): Unit = this.synchronized {
    memory.clear()
    memory ++= merge(data, metadata)
  }

  override def get(id: ArtifactId): Option[PluginsMetadata] =
    memory.get(id)

  override def getLogo(id: ArtifactId): Option[File] = {
    val dir = cacheDir.resolve(repositoryName)
    logger.debug(s"search logo data for ${id.artifactId} in $dir")
    if (dir.toFile.isDirectory) {
      LocalPluginManager
        .listDir(dir)
        .find(_.getFileName.toString.startsWith(s"${id.artifactId}."))
        .map(_.toFile)
    } else {
      logger.debug(s"not a directory: ${dir.toString}")
      None
    }
  }

  override def search(query: Option[String]): Map[ArtifactId, PluginsMetadata] = {
    query match {
      case None =>
        readonly

      case Some(name) if name.nonEmpty =>
        for {
          (artifactId, pm@PluginsMetadata(_, versions, _)) <- readonly
          subset = versionsByFilename(name)(artifactId, versions)
          if subset.nonEmpty
        } yield {
          artifactId -> pm.copy(versions = subset)
        }
    }
  }

  protected def versionsByFilename(query: String)(id: ArtifactId, versions: Set[Option[Version.Constant]]): Set[Option[Version.Constant]] =
    for {
      v <- versions
      pluginId <- id.toPluginId(v)
      if pluginId.filename contains query
    } yield v

  protected def merge(data: Seq[PluginId.Artifact],
                      metadata: Map[ArtifactId, ExtendedMetadata]): Seq[(ArtifactId, PluginsMetadata)] = {
    for {
      (artifactId, group) <- data.groupBy(_.toArtifactId)
      versions = group.map(_.version).map(Option.apply).toSet
      if versions.nonEmpty
      meta = metadata.get(artifactId)
      hasLogo = getLogo(artifactId).isDefined
    } yield {
      artifactId -> PluginsMetadata(meta, versions, hasLogo)
    }
  }.toSeq

  protected def readonly: Map[ArtifactId, PluginsMetadata] = memory.toMap

}