package com.xebialabs.plugin.classloader

import java.io.{File, FilenameFilter, InputStream}
import java.net.URL
import java.util

import com.xebialabs.plugin.protocol.xlp.PluginURL
import com.xebialabs.plugin.zip.PluginScanner
import com.xebialabs.xlplatform.utils.PerformanceLogging
import de.schlichtherle.truezip.file.TFile
import grizzled.slf4j.Logger

import scala.util.Try

object PluginClassLoader {
  def apply(pluginExtension: String, pluginDirectory: File, parentClassLoader: ClassLoader) =
    new PluginClassLoader(pluginExtension, Seq(pluginDirectory), Seq(), parentClassLoader)

  lazy val hotfixLogger = Logger.apply("hotfix")
}

private[plugin] class PluginClassLoader(pluginExtension: String, pluginDirectories: Iterable[File], explodedDirectories: Iterable[File], parentClassLoader: ClassLoader)
  extends ClassLoader(parentClassLoader) with PluginScanner with PerformanceLogging {

  import scala.collection.convert.ImplicitConversionsToJava._

  override def findClass(name: String): Class[_] = logWithTime(s"Loading class $name") {
    logger.trace(s"Loading class $name")
    val classOption = findResourceUrl(convertClassName(name)).map(loadClassFromUrl(name, _))
    classOption.getOrElse(
      throw new ClassNotFoundException(
        s"""A plugin could not be loaded due to a missing class ($name). Please remove the offending plugin to successfully start the server.
         |Classes related to JCR were removed from the server because of the migration from JCR to SQL.
         |If the plugin depends on these classes and its functionality is required, please contact support to fix your configuration.
         |$name not found""".stripMargin.replaceAll("\n", ""))
    )
  }

  private def loadClassFromUrl(className: String, resourceUrl: URL): Class[_] = {
    logger.trace(s"Loading class from url $resourceUrl")
    import com.xebialabs.xlplatform.utils.ResourceManagement._
    using(resourceUrl.openStream()) { classInputStream =>
      val bytes: Array[Byte] = readFully(classInputStream)
      if (bytes.isEmpty) {
        throw new ClassFormatError("Could not load class. Empty stream returned")
      }
      definePackageIfNeeded(className)
      val clazz = defineClass(className, bytes, 0, bytes.length)
      resolveClass(clazz)
      clazz
    }
  }

  private def definePackageIfNeeded(className: String): Unit = {
    val packageName: String = className.split('.').init.mkString(".")
    Option(getPackage(packageName)).getOrElse(definePackage(packageName, null, null, null, null, null, null, null))
  }

  def logHotfix(url: URL): URL = {
    if (url != null && url.toString.contains("hotfix")) {
      PluginClassLoader.hotfixLogger.warn(s"Loading class/resource from hotfix: $url")
    }
    url
  }

  override def findResource(name: String): URL = logWithTime(s"Loading resource $name")(logHotfix(findResourceUrl(name).orNull))

  override def findResources(name: String): util.Enumeration[URL] = logWithTime(s"Loading resources $name")(resourcesByName(name).map(tf => logHotfix(PluginURL(tf))).toIterator)

  private def findResourceUrl(name: String): Option[URL] = resourceByName(name)

  private def convertClassName(className: String) = className.replaceAll("\\.", "/").concat(".class")

  private def listAllContent(root: TFile): Stream[TFile] = {
    if (root.isFile && !root.isArchive) {
      Seq(root).toStream
    } else {
      val fileStream = {
        for {
          r <- Option(root)
          children <- Option(r.listFiles())
        } yield children.toStream
      }.getOrElse(Stream.empty)
      fileStream ++ fileStream.flatMap(listAllContent)
    }
  }

  private def resourceByName(resourcePath: String): Option[URL] = {
    import collection.JavaConverters._
    Option(Try(ResourceFinder.findResourceInDirs(resourcePath, classPathRoots.asJava)).recover {
      case e:RuntimeException =>
        logger.warn(s"Could not load resource at $resourcePath, delegating to parent classloader")
        null
    }.getOrElse(null))
  }

  private def resourcesByName(resourcePath: String): Stream[TFile] = {
    val r = for {
      root <- classPathRoots
      allRes = listAllContent(root)
      resource <- allRes if filterByResourceName(root, resource, resourcePath)
    } yield resource

    r.toSet.toStream
  }

  private def filterByResourceName(root: TFile, resource: TFile, resourcePath: String): Boolean = {
    val relativePath = resource.getAbsolutePath.drop(root.getAbsolutePath.length + 1)
    relativePath.replaceAllLiterally(File.separator, "/") == resourcePath
  }

  private def readFully(is: InputStream) = Stream.continually(is.read).takeWhile(-1 != _).map(_.toByte).toArray

  private val classPathRoots: Iterable[TFile] = {
    val archives = pluginDirectories.flatMap { pluginDirectory =>
      val jarPlugins = findAllPluginFiles(pluginDirectory, "jar").map(new TFile(_))
      val extensionPlugins = findAllPluginFiles(pluginDirectory, pluginExtension).map(new TFile(_)).flatMap(_.listFiles(jarFilter))
      jarPlugins ++ extensionPlugins
    }

    archives foreach (_.listFiles()) //To mount every jar file to avoid it mounting in recursion

    explodedDirectories.map(new TFile(_)) ++ archives
  }

  private lazy val jarFilter = new FilenameFilter {
    override def accept(dir: File, name: String): Boolean = name.endsWith(".jar")
  }
}
