package com.xebialabs.xlplatform.security.sql

import java.util

import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.{PermissionEditor, Permissions, Role}
import com.xebialabs.xlplatform.security.sql.db.Ids._
import com.xebialabs.xlplatform.security.sql.db.Tables
import com.xebialabs.xlplatform.security.sql.db.Tables.RolePermission
import com.xebialabs.xlplatform.repository.sql.Database
import grizzled.slf4j.Logging
import slick.dbio.DBIOAction.seq
import slick.jdbc.JdbcProfile

import scala.collection.JavaConverters._

class SqlPermissionEditor(securityDatabase: Database, roleService: SqlRoleService) extends PermissionEditor with Logging {

  import securityDatabase._

  val profile: JdbcProfile = config.databaseType.profile

  import profile.api._

  override def editPermissions(onConfigurationItem: String, permissions: util.Map[Role, util.Set[Permission]]): Unit = {
    logger.debug(s"Writing permissions $permissions to CI [$onConfigurationItem]")

    val notApplicableTo = Permissions.isApplicableTo(permissions.asScala.flatMap(_._2.asScala).toList.asJava, onConfigurationItem, false)
    checkArgument(notApplicableTo.isEmpty, "The permissions %s are not applicable to [%s]", notApplicableTo, onConfigurationItem)

    runAwait {
      seq(
        Tables.rolePermissions.filter(_.isOnConfigurationItem(Option(onConfigurationItem))).delete,
        Tables.rolePermissions ++= permissions.asScala.flatMap {
          case (role, rolePermissions) => rolePermissions.asScala.map(permission => RolePermission(role.getId, permission.getPermissionName, toDbId(Option(onConfigurationItem))))
        }
      )
    }
  }

  override def readPermissions(onConfigurationItem: String): util.Map[Role, util.Set[Permission]] = {
    logger.debug(s"Reading permissions from [$onConfigurationItem]")

    val permissions = runAwait {
      Tables.roles
        .join(Tables.rolePermissions).on(_.id === _.roleId)
        .filter { case (_, rolePermission) => rolePermission.isOnConfigurationItem(Option(onConfigurationItem)) }
        .result
    }.map { case (role, rolePermission) => (new Role(role.id, role.name), Permission.find(rolePermission.permissionName)) }
      .groupBy(_._1)
      .mapValues { values => values.map(_._2).toSet.asJavaMutable() }

    logger.debug(s"Read from [$onConfigurationItem] permissions $permissions")
    permissions.asJavaMutable()
  }
}
