package com.xebialabs.deployit.security;

import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.security.permission.PlatformPermissions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.stereotype.Component;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import static com.xebialabs.deployit.security.Permissions.authenticationToPrincipals;
import static com.xebialabs.deployit.security.Permissions.getAuthentication;

@Component
public class PermissionEnforcer {

    public static final String ROLE_ADMIN = "ROLE_ADMIN";

    private PermissionChecker checker;
    private final RoleService roleService;

    @Autowired
    public PermissionEnforcer(PermissionChecker checker, RoleService roleService) {
        this.checker = checker;
        this.roleService = roleService;
    }

    public boolean hasPermission(Authentication authentication, final Permission... permissions) {
        return hasPermission(authentication, Arrays.asList(permissions), "");
    }

    public boolean hasLoggedInUserPermission(final Permission... permissions) {
        Authentication authentication = getAuthentication();
        return hasPermission(authentication, Arrays.asList(permissions), "");
    }

    public boolean hasLoggedInUserPermission(Permission permission, String onConfigurationItem) {
        return hasLoggedInUserPermission(Collections.singletonList(permission), onConfigurationItem);
    }

    public boolean hasLoggedInUserPermission(final List<Permission> permissions, String onConfigurationItem) {
        return hasPermission(getAuthentication(), permissions, onConfigurationItem);
    }

    private boolean hasPermission(Authentication auth, List<Permission> permissions, String onConfigurationItem) {
        final Collection<String> allPrincipals = authenticationToPrincipals(auth);
        List<Role> allRoles = roleService.getRolesFor(auth);
        return isAdmin(allPrincipals, allRoles) || checker.checkPermission(permissions, onConfigurationItem, allRoles);
    }

    boolean isAdmin(Collection<String> allPrincipals, List<Role> allRoles, Authentication auth) {
        Boolean isAdmin = allPrincipals.contains(ROLE_ADMIN) || checker.checkPermission(Collections.singletonList(PlatformPermissions.ADMIN), "", allRoles, auth);
        logger.trace("Admin privileges [{}] granted to {}", isAdmin ? "are" : "are not", allPrincipals);
        return isAdmin;
    }

    boolean isAdmin(Collection<String> allPrincipals, List<Role> allRoles) {
        return isAdmin(allPrincipals, allRoles, Permissions.getAuthentication());
    }

    private static final Logger logger = LoggerFactory.getLogger(PermissionEnforcer.class);

    public boolean isCurrentUserAdmin() {
        return isAdmin(getAuthentication());
    }

    public boolean isAdmin(Authentication auth) {
        Collection<String> principals = authenticationToPrincipals(auth);
        List<Role> allRoles = roleService.getRolesFor(auth);
        return isAdmin(principals, allRoles, auth);
    }
}
