package com.xebialabs.xlplatform.sugar

import java.io.{File, IOException}
import java.nio.charset.Charset
import java.nio.charset.StandardCharsets.UTF_8
import java.nio.file.{FileVisitOption, Files, Path}
import java.util.Comparator.reverseOrder
import scala.jdk.StreamConverters._
import scala.language.implicitConversions

object PathSugar {

  implicit def path2File(path: Path): File = path.toFile

  implicit def file2Path(file: File): Path = file.toPath

  implicit class PathUtils(realPath: Path) {

    def createFile(createParents: Boolean = true, failIfExists: Boolean = false): Path = {
      if (createParents) {
        realPath.getParent.createDirectory(failIfExists = false)
      }
      Files.createFile(realPath)
    }

    def createDirectory(createParents: Boolean = true, failIfExists: Boolean = true): Path = {
      if (!failIfExists && Files.exists(realPath)) {
        realPath
      } else {
        if (createParents) {
          Files.createDirectories(realPath)
        } else {
          Files.createDirectory(realPath)
        }
      }
    }

    def write(content: String, charset: Charset = UTF_8): Unit = {
      Files.write(realPath, content.getBytes(charset))
    }

    def deleteRecursively(ignoreErrors: Boolean = false) = {
        try {
          Files.walk(realPath, FileVisitOption.FOLLOW_LINKS)
            .sorted(reverseOrder()).toScala(Seq)
            .map(_.toFile())
            .foreach((f: File) => f.delete())
        } catch {
          case e: IOException => if (!ignoreErrors) throw e
        }
    }

    def /(rest: String): Path = realPath.resolve(rest)

    def jfile(): File = realPath.toFile

    def path: String = realPath.toFile.getPath

    def children(): Seq[Path] = Files.walk(realPath, 1).toScala(Seq) diff Seq(realPath)

    def size(): Long = Files.size(realPath)

    def chars(charset: Charset = UTF_8): Seq[Char] = {
      new String(Files.readAllBytes(realPath), charset).toCharArray.toSeq
    }
  }

}
