package com.xebialabs.xltest.domain;

import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.net.UrlEscapers;
import com.jayway.jsonpath.spi.JsonProvider;
import com.jayway.jsonpath.spi.JsonProviderFactory;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xltest.utils.JsonOverHttpHelper;
import com.xebialabs.xltest.utils.JsonResponseParser;


@SuppressWarnings("serial")
@Metadata(description = "ElasticSearch store base type", root = Metadata.ConfigurationItemRoot.CONFIGURATION)
public class ElasticSearchStore extends Store {
    private static final Logger LOG = LoggerFactory.getLogger(ElasticSearchStore.class);

	@Property(description = "Base url for elasticSearch, including its index (but without its type) like http://localhost:9200/movies")
	private String url;

	public ElasticSearchStore() {
	}

	public ElasticSearchStore(String url) {
		super();
		this.url = url;
	}
	
	@Override
	public void store(Event event) throws Exception {
        JsonOverHttpHelper jsonOverHttpHelper = new JsonOverHttpHelper();
        JsonProvider jsonProvider = JsonProviderFactory.createProvider();
        String requestContent = jsonProvider.toJson(event.getProperties());
        jsonOverHttpHelper.httpPostRequest(url + "/" + event.getType() + "/", requestContent);
        if (jsonOverHttpHelper.statusCode() != 201) {
            LOG.error("Could not store message in ElasticSearchStore. Status code: {}, request: '{}', reponse: '{}'", jsonOverHttpHelper.statusCode(), requestContent, jsonOverHttpHelper.content());
        }
	}
	
	@Override
	public List<Event> getEventsOfRun(TestRunId runId) {
		// http://localhost:9200/testxltestevents/_search?q=run_id:2223e40a-e011-4f51-b019-79cf0b1d018e
		String fullUrl = url + "/" + "_search?sort=_ts:asc&q=run_id:" + runId + "&size=1000000";
        LOG.info("Querying ES on url: {}", fullUrl);
        List<Event> results = parseElasticSearchJson(retrieveEvents(fullUrl));
        LOG.info("Queried ES on url: {} yielded {} events", fullUrl, results.size());
        return results;
	}

    @Override
    public List<Event> getEventsOfRun(TestRunId runId, Map queryParameters) {
        // http://localhost:9200/testxltestevents/_search?q=run_id:2223e40a-e011-4f51-b019-79cf0b1d018e
        String fullUrl = constructUrl(runId, queryParameters);
        LOG.info("Querying ES on url: {}", fullUrl);
        List<Event> results = parseElasticSearchJson(retrieveEvents(fullUrl));
        LOG.info("Queried ES on url: {} yielded {} events", fullUrl, results.size());
        return results;
    }

    @Override
    public List<Event> getEventsBetween(long startTime, long endTime, Map queryParameters) {
        String fullUrl = constructUrl(startTime, endTime, queryParameters);
        LOG.info("Querying ES on url: {}", fullUrl);
        List<Event> results = parseElasticSearchJson(retrieveEvents(fullUrl));
        LOG.info("Queried ES on url: {} yielded {} events", fullUrl, results.size());
        return results;
    }

    private String retrieveEvents(String fullUrl) {
		JsonOverHttpHelper jsonOverHttpHelper = new JsonOverHttpHelper();
		try {
			jsonOverHttpHelper.httpGetRequest(fullUrl);
		} catch (Exception e) {
			LOG.error("Unable to contact elastic search", e);
            return null;
		}
        if (jsonOverHttpHelper.statusCode() != 200) {
            LOG.error("Not a successful elastic search query ({}): {}", jsonOverHttpHelper.statusCode(), jsonOverHttpHelper.content());
            return null;
        }
        return jsonOverHttpHelper.content();
	}

    private String constructUrl(TestRunId runId, Map queryParameters) {
        String baseUrl = getSearchQueryBase();
        String query = constructSearchQuery("run_id:" + runId, queryParameters);
        return baseUrl + query;
    }

    private String constructUrl(long startTime, long endTime, Map queryParameters) {
        String baseUrl = getSearchQueryBase();
        String query = constructSearchQuery("_ts:>=" + startTime + " AND _ts:<=" + endTime, queryParameters);
        return baseUrl + query;
    }

    private String getSearchQueryBase() {
        return url + "/_search?size=1000000&sort=_ts:asc&q=";
    }


    private String constructSearchQuery(String query, Map queryParameters) {
        for (Object key : queryParameters.keySet()) {
            String value = (String) queryParameters.get(key);
            query = query + " AND " + key + ":\"" + value + "\"";
        }
        query = UrlEscapers.urlPathSegmentEscaper().escape(query);
        return query;
    }

    List<Event> parseElasticSearchJson(String content) {
        if (content == null) {
            return Collections.emptyList();
        }
        JsonResponseParser parser = new JsonResponseParser();
        parser.parse(content);
        Integer count = (Integer) parser.getValue("hits.total");
        List<Event> retrievedEvents = new ArrayList<Event>();
        JsonProvider jsonProvider = parser.getJsonProvider();
        for (int i = 0; i < count; i++) {
            Object item = parser.getValue("hits.hits[" + i + "]._source");
            if (jsonProvider.isMap(item)) {
                retrievedEvents.add(new Event(jsonProvider.toMap(item)));
            }
        }
        return retrievedEvents;
    }
}
