package com.xebialabs.xltest.domain;

import java.io.Serializable;
import java.util.Collections;
import java.util.Map;
import java.util.TreeMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Event implements Serializable {
    public static final String START_TEST_RUN = "startTestRun";
    public static final String FINISH_TEST_RUN = "finishTestRun";

    public static final String TYPE = "type";
    public static final String RUN_ID = "run_id";
    public static final String UNKNOWN_EVENT_TYPE = "UNKNOWN";
    public static final String TIMESTAMP = "_ts";
    
    private static final Logger LOG = LoggerFactory.getLogger(Event.class);

    private final Map<String, Object> properties;

    public Event(Map<String, Object> properties) {
        this.properties = new TreeMap(properties);
        if (!this.properties.containsKey(TIMESTAMP)) {
            update(TIMESTAMP, System.currentTimeMillis());
        }
    }

    public Event(String type, Map<String, Object> properties) {
        this(properties);
        update(TYPE, type);
    }

    public Event(String type, Event event) {
        this(type, event.getProperties());
    }

    public Event(Event event, Map<String, Object> properties) {
        this(event.getProperties());
        this.properties.putAll(properties);
    }

    public String getType() {
        return (String) properties.get(TYPE);
    }

    private Event update(String key, Object value) {
        properties.put(key, value);
        return this;
    }

    public <T> T get(String key) {
        Object value = properties.get(key);
        if (value == null) {
            LOG.warn(key + " not found in event " + this);
        }
        return (T) value;
    }
    
    public boolean hasProperty(String key) {
        return properties.get(key) != null;
    }

    public Map<String, Object> getProperties() {
        return Collections.unmodifiableMap(properties);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Event event = (Event) o;

        if (!properties.equals(event.properties)) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = properties.hashCode();
        return result;
    }

    @Override
    public String toString() {
        return "Event" + properties;
    }

    public static Map<String, Object> props(Object... keyValues) {
        assert keyValues.length % 2 == 0;
        Map<String, Object> m = new TreeMap();
        for (int i = 0; i < keyValues.length - 1; i += 2) {
            m.put(keyValues[i].toString(), keyValues[i+1]);
        }
        return m;
    }

    // Should return TestRunId
    public TestRunId getTestRunId() {
        Object testRunId = get(RUN_ID);
        if (testRunId instanceof TestRunId) {
            return (TestRunId) testRunId;
        }
        return new TestRunId(testRunId.toString());
    }
}
