package com.xebialabs.xltest.domain;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;

@SuppressWarnings("serial")
@Metadata(description = "SQL store", root = Metadata.ConfigurationItemRoot.CONFIGURATION, virtual = true)
public class SQLStore extends Store {
	private static final Logger LOG = LoggerFactory.getLogger(DerbySQLStore.class);
    private static final String RUN_ID_FIELD = "run_id";
    
	@Property(description = "connection URL, e.g. jdbc:mysql://localhost/test or jdbc:derby:memory:test or jdbc:db2://sysmvs1.stl.ibm.com:5021/san_jose or jdbc:oracle:thin:@//localhost:1521/orcl")
	private String url;
	@Property(description = "Username")
	private String username;
	@Property(description = "Password", password = true)
	private String password;
	@Property(description = "Tablename")
	private String tablename;
	@Property(description = "Property to type map. Contains event property as key (e.g. startTime) and type as value (e.g. DATE)")
	private Map<String, String> propertyTypeMap;

	public SQLStore() {
	}

	public SQLStore(String url, String username, String password, String tablename) {
		super();
		this.url = url;
		this.username = username;
		this.password = password;
		this.tablename = tablename;
	}
	
	@Override
	public synchronized void store(Event event) throws SQLException {
		 Connection connection = null;
	        try {
	            connection = createConnection();
	            createTableIfItDoesNotExist(connection);
	            insertRow(connection, event);
	        } finally {
	            closeConnection(connection);
	        }
	}
	
	@Override
	public List<Event> getEventsOfRun(TestRunId runId) {
		Connection connection = null;
        try {
            connection = createConnection();
            return queryEventsForTestRun(connection, runId);
        } catch (SQLException e) {
            throw new RuntimeException("Unable to get events for test run " + runId, e);
        } finally {
            closeConnection(connection);
        }
	}
	
    private Set<String> getColumnNames(ResultSet rs) throws SQLException {
        Set<String> columnNames = new TreeSet();
        ResultSetMetaData meta = rs.getMetaData();
        for (int i = 1; i <= meta.getColumnCount(); i++) {
            columnNames.add(meta.getColumnName(i));
        }
        return columnNames;
    }
	
	private List<Event> queryEventsForTestRun(Connection connection, TestRunId runId) {
		LOG.info("Searching for events for run {}", runId);
        List<Event> events = new LinkedList();
        PreparedStatement statement = null;
        try {
            String query = "select * from " + tablename + " where \"" + RUN_ID_FIELD + "\" = ?";
            LOG.info("Query is: {}", query);
            statement = connection.prepareStatement(query);
            statement.setString(1, runId.toString());
            ResultSet rs = statement.executeQuery();
            Set<String> columnNames = getColumnNames(rs);
            LOG.info("Creating events with keys: {}", columnNames);
            while (rs.next()) {
                Map<String, Object> properties = new TreeMap();
                for (String columnName : columnNames) {
                    properties.put(columnName, rs.getObject(columnName));
                }
                if (!properties.containsKey("type")) {
                    properties.put("type", "event");
                }
                LOG.info("Have event: " + properties);
                events.add(new Event(properties));
            }
        } catch (SQLException e) {
            LOG.info("Could not retrieve events from table: {}", e.getMessage());
            closeStatement(statement);
        }

        return events;
	}

	private void createTableIfItDoesNotExist(Connection connection) {
		PreparedStatement statement = null;
		try {
			statement = connection.prepareStatement( createTableDefinitionString());
			statement.executeUpdate();
		} catch (SQLException e) {
			// just ignore
		} finally {
			if (statement != null) {
				try {
					statement.close();
				} catch (SQLException e) {
					e.printStackTrace();
				}
			}
		}
	}
	
	private void insertRow(Connection connection, Event enrichedEvent) {
		PreparedStatement statement = null;
		try {
			statement = connection.prepareStatement(createInsertIntoTableString(enrichedEvent));
			int i = 1;
			for (String key : getOrderedKeys()) {
				String type = getPropertyTypeMap().get(key);
				Object plainColumnValue = enrichedEvent.get(key);
				if (type.equals("int")) {
					if (plainColumnValue != null) {
						statement.setInt(i++, (Integer)plainColumnValue);
					} else {
						statement.setNull(i++, Types.INTEGER);
					}
				} else if (type.startsWith("varchar")) {
					if (plainColumnValue != null) {
						statement.setString(i++, plainColumnValue.toString());
					} else {
						statement.setNull(i++, Types.VARCHAR);
					}
				} else if (type.equals("timestamp")) {
					if (plainColumnValue != null) {
						statement.setTimestamp(i++, new Timestamp(((Number)plainColumnValue).longValue()));
					} else {
						statement.setNull(i++, Types.TIMESTAMP);
					}
				} else {
					throw new RuntimeException ("Can't build insert row query for type: " + type);
				}
			}
			statement.executeUpdate();
		} catch (SQLException e) {
			e.printStackTrace();
		} finally {
			if (statement != null) {
				try {
					statement.close();
				} catch (SQLException e) {
					e.printStackTrace();
				}
			}
		}
	}

	public String createInsertIntoTableString(Event event) {
		throw new RuntimeException("The createInsertIntoTableString method MUST be overridden in subclasses");
	}
	
	public String columnDefinitionsAsString() {
		throw new RuntimeException("The method columnDefinitionsAsString MUST be overridden in sub class");
	};
	
	public String makeQuestionMarks() {
		StringBuilder sb = new StringBuilder();
		if (!getPropertyTypeMap().keySet().isEmpty()) {
			for (int i = 0; i < getOrderedKeys().size(); i++) {
				sb.append("?, ");
			}
		}
		String columnValues = sb.toString();
		if (columnValues.length() > 2) {
			columnValues = columnValues.substring(0, columnValues.length() - 2);
		}
		return columnValues;
	}

	private String createTableDefinitionString() {
		return "create table " + getTablename() + createColumnDefinitionsAsString();
	}

	public String createColumnDefinitionsAsString() {
		throw new RuntimeException("The method createColumnDefinitionsAsString MUST be overridden in a subtype");
	}

	private Connection createConnection() throws SQLException {
        Connection connection = DriverManager.getConnection(getUrl(), getUsername(), getPassword());
        connection.setAutoCommit(true);
        return connection;
	}

	private void closeConnection(Connection connection) {
		try {
			if (connection != null) {
				connection.close();
				connection = null;
			}
		} catch (SQLException e) {
			e.printStackTrace();
		}
	}
	
	private void closeStatement(PreparedStatement statement) {
        if (statement != null) {
            try {
                statement.close();
            } catch (SQLException e) {
                e.printStackTrace();
            }
        }
    }
	
	public List<String> getOrderedKeys() {
		List<String> orderedKeys = new ArrayList<String>();
		orderedKeys.addAll(getPropertyTypeMap().keySet());
		Collections.sort(orderedKeys);
		return orderedKeys;
	}
	
	public String getUrl() {
		return url;
	}

	public void setUrl(String url) {
		this.url = url;
	}

	public String getUsername() {
		return username;
	}

	public void setUsername(String username) {
		this.username = username;
	}

	public String getPassword() {
		return password;
	}

	public void setPassword(String password) {
		this.password = password;
	}

	public String getTablename() {
		return tablename;
	}

	public void setTablename(String tablename) {
		this.tablename = tablename;
	}
	
    public Map<String, String> getPropertyTypeMap() {
		return propertyTypeMap;
	}

	public void setPropertyTypeMap(Map<String, String> propertyTypeMap) {
		this.propertyTypeMap = propertyTypeMap;
	}

}
