package com.xebialabs.xltest.resources;

import static com.xebialabs.deployit.plugin.api.reflect.Type.valueOf;

import java.io.IOException;
import java.util.*;

import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.*;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.UriInfo;

import com.jayway.jsonpath.spi.JsonProviderFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.xltest.domain.Report;
import com.xebialabs.xltest.domain.TestRun;
import com.xebialabs.xltest.domain.TestRunId;
import com.xebialabs.xltest.domain.TestSetDefinition;
import com.xebialabs.xltest.repository.NotFoundException;
import com.xebialabs.xltest.repository.TestRuns;

@Controller
@Path("/reports")
@Consumes(MediaType.APPLICATION_JSON)
@Produces(MediaType.APPLICATION_JSON)
public class ReportController {
    private static final Logger LOG = LoggerFactory.getLogger(ReportController.class);

    private final RepositoryService repositoryService;
    private final TestRuns testRuns;

	@Autowired
	public ReportController(RepositoryService repositoryService, TestRuns testRuns) {
		this.repositoryService = repositoryService;
		this.testRuns= testRuns;
	}

	@GET
	@Path("/{testRunId:[\\w-]+}")
	public Collection<Report> getReports(@PathParam("testRunId") String testRunId, @Context final HttpServletResponse response) throws IOException {
        try {
        	TestRun testRun = testRuns.getTestRun(new TestRunId(testRunId));
        	TestSetDefinition testSetDefinition  = testRun.getTestSetDefinition();
        	List<Report> reports = findReportForTestSetDefinitionType(testSetDefinition.getType().toString());
        	if (reports != null && !reports.isEmpty()) {
        		return reports;
        	}
            response.sendError(404, "Can't find any Report associated to test runs of type: " + testRun.getType());
            return null;
        } catch (NotFoundException e) {
            response.sendError(404, "Can't find test run");
            return null;
        }
	}

	private List<Report> findReportForTestSetDefinitionType(String testSetDefinitionType) {
        List<Report> reports = new LinkedList();

        SearchParameters query = new SearchParameters().setType(valueOf(Report.class));
        List<Report> reportCis = repositoryService.listEntities(query);

        for (Report report : reportCis) {
            List<String> testSetTypes = report.getTestSetTypes();
            if (testSetTypes != null && testSetTypes.contains(testSetDefinitionType)) {
                reports.add(report);
            }
        }

        return reports;
	}

    @GET
    @Path("/{testRunId:[\\w-]+}/{reportId:.*}")
    public String generateReport(@PathParam("testRunId") String testRunId, @PathParam("reportId") String reportId, @Context UriInfo uriInfo) throws IOException {
        Report report = repositoryService.read(reportId);
        LOG.info("Found report with value {}", report);
        Map<String, Object> resultObject = testRuns.generateReport(new TestRunId(testRunId), report, convertMultivaluedMapToMap(uriInfo.getQueryParameters()));
        return JsonProviderFactory.createProvider().toJson(resultObject);
    }

    @POST
    @Path("/{testRunId:.+}")
    public String generateReport(@PathParam("testRunId") String testRunId, Report report, @Context UriInfo uriInfo) {
        Map<String, Object> resultObject = testRuns.generateReport(new TestRunId(testRunId), report, convertMultivaluedMapToMap(uriInfo.getQueryParameters()));
        return JsonProviderFactory.createProvider().toJson(resultObject);
    }

    /**
     * Yes, we drop all but the first parameter value.
     *
     * @param multivaluedMap
     * @return
     */
    private Map<String, String> convertMultivaluedMapToMap(MultivaluedMap<String, String> multivaluedMap) {
        Map<String, String> map = new TreeMap();
        for (String key : multivaluedMap.keySet()) {
            map.put(key, multivaluedMap.getFirst(key));
        }
        return map;
    }
}

