package com.xebialabs.xltest.service;

import static com.xebialabs.deployit.engine.tasker.BlockBuilders.steps;
import static java.lang.String.format;

import java.net.URI;
import java.util.Date;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.xebialabs.deployit.engine.api.execution.StepExecutionState;
import com.xebialabs.deployit.engine.api.execution.StepState;
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState;
import com.xebialabs.deployit.engine.api.execution.TaskWithSteps;
import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener;
import com.xebialabs.deployit.engine.spi.execution.StepExecutionStateEvent;
import com.xebialabs.deployit.engine.spi.execution.TaskExecutionStateEvent;
import com.xebialabs.deployit.engine.tasker.Block;
import com.xebialabs.deployit.engine.tasker.BlockBuilder;
import com.xebialabs.deployit.engine.tasker.IEngine;
import com.xebialabs.deployit.engine.tasker.Task;
import com.xebialabs.deployit.engine.tasker.TaskNotFoundException;
import com.xebialabs.deployit.engine.tasker.TaskSpecification;
import com.xebialabs.xltest.domain.TestRun;
import com.xebialabs.xltest.domain.TestRunId;
import com.xebialabs.xltest.plan.CleanupPlan;
import com.xebialabs.xltest.plan.Plan;
import com.xebialabs.xltest.plan.TestPlan;
import com.xebialabs.xltest.repository.TestRuns;

/**
 * Execute a test.
 *
 * This service is implemented as a "top-level" event handler. It takes "start" events and
 * launches a test run based on "testSetId" and "eventUri" properties.
 */
@Service
public class TestRunner implements TestRunStatusProvider {
    private static final Logger LOG = LoggerFactory.getLogger(TestRunner.class);

    private final TestRuns testRuns;
    private IEngine taskExecutionEngine;
    private Map<TestRunId, String> runIdToTaskId = new ConcurrentHashMap<>();

    @Autowired
    public TestRunner(TestRuns testRuns, IEngine taskExecutionEngine) {
        this.testRuns = testRuns;
        this.taskExecutionEngine = taskExecutionEngine;
    }

    public TestRun run(TestRun testRun, URI eventUri) {
        testRuns.addRunOfTestSet(testRun, eventUri);
        LOG.info("Starting new test run for test set id {} on url {}", testRun.getTestSetDefinition().getId(), testRun.getUri());

        run(testRun);

        return testRun;
    }

    public void run(final TestRun testRun) {
        LOG.info("Running job with id " + testRun.getTestRunId() + "; url: " + testRun.getUri());
        BlockBuilder planBuilder = testRun.planWithoutBuilding();
        Block plan = planBuilder.build();

        TaskSpecification taskSpecification = new TaskSpecification(
                String.format("Test Run %s", testRun.getId()), "owner", plan, null, false, false);
        taskSpecification.getListeners().add(new ExecutionStateMonitor(testRun));
        String taskId = taskExecutionEngine.register(taskSpecification);

        runIdToTaskId.put(testRun.getTestRunId(), taskId);

        taskExecutionEngine.execute(taskId);
    }

    public boolean isRunning(TestRunId testRunId) {
        TaskExecutionState state = getState(testRunId);
        return !(state.isFinal() || state == TaskExecutionState.UNREGISTERED);
    }

    public TaskExecutionState getState(TestRunId testRunId) {
        String taskId = runIdToTaskId.get(testRunId.getTopLevel());
        if (taskId != null) {
            try {
                Task task = taskExecutionEngine.retrieve(taskId);
                return task.getState();
            } catch (TaskNotFoundException e) {
                // This happens after a task has been archived.
                LOG.debug("Task not found for " + testRunId);
            }
        }

        TestRun testRun = testRuns.getTestRun(testRunId);
        if (testRun == null) {
            LOG.debug("Task engine state defaults to UNREGISTERED (testRun == null)");
            return TaskExecutionState.UNREGISTERED;
        } else if (testRun.isFinished()) {
            LOG.debug("Task engine state defaults to DONE (testRun is finished)");
            return TaskExecutionState.DONE;
        } else {
            LOG.debug("Task engine state defaults to CANCELLED (not running, not finished)");
            return TaskExecutionState.CANCELLED;
        }
    }

    public void abort(TestRunId testRunId) {
        String taskId = runIdToTaskId.get(testRunId.getTopLevel());
        if (taskId != null) {
            taskExecutionEngine.abort(taskId);
        }
    }

    private Date finishedTime() {
        return new Date();
    }


    /**
     * Here we check the state of the tasks being executed. If tasks fail,
     * set the state to "skipped" and restart the task. This will result in the already executed
     * and skipped step being skipped and it will continue with the next task at hand.
     */
    private class ExecutionStateMonitor implements ExecutionStateListener {
        private TestRun testRun;

        public ExecutionStateMonitor(TestRun testRun) {
            this.testRun = testRun;
        }

        @Override
        public void stepStateChanged(StepExecutionStateEvent stepExecutionStateEvent) {

        }

        @Override
        public void taskStateChanged(TaskExecutionStateEvent taskExecutionStateEvent) {
            TaskWithSteps task = taskExecutionStateEvent.task();
            if (canBeArchived(taskExecutionStateEvent.currentState())) {
                finishTestRun();
                taskExecutionEngine.archive(task.getId());
            } else if (isAborted(taskExecutionStateEvent.currentState())) {
                finishTestRun();
                taskExecutionEngine.cancel(task.getId());
            } else if (notRunningAndNeedsToBeResumed(taskExecutionStateEvent.currentState())) {
            	LOG.info("Task might have to be resumed (depends on time out or not)");
	            markFailedStepsAsSkipped(task);
            	LOG.info("Task has not timed out, so we start executing it again");
            	taskExecutionEngine.execute(task.getId());
            }
        }

        private boolean isAborted(TaskExecutionState taskExecutionState) {
            return taskExecutionState == TaskExecutionState.ABORTED;
        }

        private void finishTestRun() {
            runIdToTaskId.remove(testRun.getId());
            testRun.setFinishedTime(finishedTime());
            LOG.info("Update test run after execution: with stores " + testRun.getUsedStores());
            testRuns.updateTestRun(testRun);            
        }

		private boolean canBeArchived(TaskExecutionState taskExecutionState) {
            return taskExecutionState == TaskExecutionState.EXECUTED;
        }

        private boolean notRunningAndNeedsToBeResumed(TaskExecutionState taskExecutionState) {
            return taskExecutionState == TaskExecutionState.FAILED;
        }

		private void markFailedStepsAsSkipped(TaskWithSteps task) {
            for (StepState stepState : task.getSteps()) {
                LOG.debug("Found step state {} -> {}", stepState, stepState.getState());
            }
            for (Integer stepNr : task.getCurrentStepNrs()) {
                StepState stepState = task.getStep(stepNr);
                LOG.debug("Found current step state {} in state {}", stepState, stepState.getState());
                if (stepState.getState() == StepExecutionState.FAILED) {
                    taskExecutionEngine.skipSteps(task.getId(), Lists.newArrayList(stepNr));
                    // TODO: Should do something with the fact that a step failed?
                }
            }
        }
    }
    
}
