package com.xebialabs.xltest;

import java.io.IOException;
import java.security.Permissions;

import javax.crypto.SecretKey;
import javax.jcr.RepositoryException;

import com.xebialabs.xltest.repository.ScriptExecutor;
import com.xebialabs.xltest.repository.ScriptExecutorHolder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;

import com.xebialabs.deployit.ServerConfiguration;
import com.xebialabs.deployit.core.rest.api.MetadataServiceImpl;
import com.xebialabs.deployit.engine.api.MetadataService;
import com.xebialabs.deployit.engine.api.execution.TaskWithBlock;
import com.xebialabs.deployit.engine.tasker.Archive;
import com.xebialabs.deployit.engine.tasker.IEngine;
import com.xebialabs.deployit.engine.tasker.TaskExecutionEngine;
import com.xebialabs.deployit.jcr.JackrabbitRepositoryFactoryBean;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.repository.JcrRepositoryService;
import com.xebialabs.deployit.repository.RepositoryAdapter;
import com.xebialabs.deployit.repository.RepositoryAdapterFactory;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.RepositoryServiceHolder;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.repository.core.Directory;
import com.xebialabs.deployit.service.validation.Validator;
import com.xebialabs.deployit.util.DeployitKeys;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.xltest.service.EventRegistry;
import com.xebialabs.xltest.service.InMemoryRepositoryService;
import com.xebialabs.xltest.service.SerialEventDispatcher;

/**
 * This class contains the "configurable" part of the XL Test application.
 */
@Configuration
@ComponentScan({"com.xebialabs.deployit.service.validation", "com.xebialabs.xltest"})
public class ContextConfig {
	private static final Logger LOG = LoggerFactory.getLogger(ContextConfig.class.getName());

    /**
     * Bean for ConfigurationItem support.
     * @return
     */
    @Bean
    public MetadataService metadataService() {
        return new MetadataServiceImpl();
    }
    
    @Bean
    public EventRegistry eventDispatcher() {
    	return new SerialEventDispatcher();
    }

    @Bean
    public Repository repository(RepositoryService repositoryService, Validator validator) {
        WorkDir workDir = null;
        return new RepositoryAdapter(repositoryService, validator, workDir);
    }
    
    @Bean
    public RepositoryServiceHolder repositoryServiceHolder(RepositoryService repositoryService) {
        return new RepositoryServiceHolder(repositoryService);
    }

    @Bean
    public RepositoryService repositoryService() throws IOException, RepositoryException {
    	if ("true".equals(System.getProperty("testMode"))) {   
    		LOG.info("Using in memory repository service");
        	return new InMemoryRepositoryService();
    	} else {
	    	JackrabbitRepositoryFactoryBean factory = new JackrabbitRepositoryFactoryBean();
	    	factory.setHomeDir(new FileSystemResource(".")); // use "." form normal, use e.g. "/Users/wilco/snap/server-0.1" for debugging
	    	factory.setRepositoryName("xltest");
	    	Resource configuration = new ClassPathResource("jackrabbit-repository.xml");
			factory.setConfiguration(configuration);
			factory.setAutoInitialize(true);
			
			ServerConfiguration deployitConfig = new ServerConfiguration();
			deployitConfig.setDefaults();
			factory.setDeployitConfig(deployitConfig);
				
			factory.afterPropertiesSet();
			javax.jcr.Repository repo = factory.getObject();
	
	    	SecretKey secretKey = DeployitKeys.getPasswordEncryptionKey("keyStorePassword");
	    	PasswordEncrypter.init(secretKey);
	    	PasswordEncrypter passwordEncrypter = PasswordEncrypter.getInstance();
	    	JcrRepositoryService jcrRepositoryService = new JcrRepositoryService(new JcrTemplate(repo, deployitConfig), passwordEncrypter);
			initDirectoriesInRepository(jcrRepositoryService);
	    	return jcrRepositoryService;
    	} 
    }

	private void initDirectoriesInRepository(JcrRepositoryService jcrRepositoryService) {
		Directory stores = (new Directory());
		stores.setId("Configuration/Stores");
		Directory reports = (new Directory());
		reports.setId("Configuration/Reports");
		Directory listeners = (new Directory());
		listeners.setId("Configuration/Listeners");
		Directory records = (new Directory());
		records.setId("Configuration/Records");
		Directory testSetDefinitions = (new Directory());
		testSetDefinitions.setId("Configuration/TestSetDefinitions");
		Directory testRuns = (new Directory());
		testRuns.setId("Applications/TestRuns");
		jcrRepositoryService.createOrUpdate(stores, reports, listeners, records, testSetDefinitions, testRuns);
	}

    @Bean
    public IEngine taskExecutionEngine(RepositoryService repositoryService, Validator validator) {
    	if (repositoryService == null) {
    		LOG.info("task execution engine wired with a null repo service");
    	} else {
    		LOG.info("task execution engine properly wired with instance of class: " + repositoryService.getClass().getName());
    	}
        return new TaskExecutionEngine(getArchive(), new RepositoryAdapterFactory(repositoryService, validator));
    }

    private Archive getArchive() {
        return new Archive() {
            @Override
            public void archive(TaskWithBlock taskWithBlock) {
                // NOOP
            }
        };
    }
}
