package com.xebialabs.xltest.domain;

import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.script.ScriptContext;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.repository.RepositoryServiceHolder;
import com.xebialabs.xltest.repository.ScriptExecutionException;

@SuppressWarnings("serial")
@Metadata(description = "Listener base type", root = Metadata.ConfigurationItemRoot.CONFIGURATION)
public class EventProcessor extends ScriptedConfigurationItem implements EventHandler {
	@Property(description = "Event type this listener listens for, e.g. fitResult")
	private String eventType;
	@Property(description = "Stores in which this event needs to be persisted")
	private List<Store> stores;
	
	private transient Set<Store> usedStores;
	
	private static final Logger LOG = LoggerFactory.getLogger(EventProcessor.class);
	
	public EventProcessor() {
	}
	
	public EventProcessor(String scriptLocation) {
		super();
		setScriptLocation(scriptLocation);
	}

	public EventProcessor(String eventType, List<Store> stores) {
		super();
		this.eventType = eventType;
		this.stores = stores;
	}

	@Override
	public void onReceive(Event event) {
		LOG.info("Received event of type {}", event.getType());
		if (eventTypeMatches(event) && getStores() != null) {
			List<Exception> exceptions = new ArrayList<Exception>();
			List<Store> failingStores = new ArrayList<Store>();
			// we catch intermediate exceptions, otherwise one failing store implies failure of others
			Event processedEvent = process(event);
			for (Store store : getStores()) {
				try {
					LOG.info("Forwarding event of type {}", processedEvent.getType() + " to store " + store.getId());
					store.store(processedEvent);
					if (usedStores == null) {
						usedStores = new HashSet<Store>();
					}
					usedStores.add(store);
				} catch (Exception e) {
					exceptions.add(e);
					failingStores.add(store);
                    e.printStackTrace();
				}
			}
			if (exceptions.size() > 0) {
				StringBuilder sb = new StringBuilder();
				sb.append("These stores threw exceptions: ");
				int cnt = 0;
				for (Store store : failingStores) {
					sb.append(store.getId());
					sb.append(": ");
					Exception exception = exceptions.get(cnt++);
					exception.printStackTrace();
					sb.append(exception.toString());
					sb.append("; ");
				}
				throw new RuntimeException("Storage exceptions: " + sb.toString());
			}
		}
	}
	
	public Event process(Event event) {
		ScriptContext context = getScriptContext();
        context.setAttribute("event", event, ScriptContext.ENGINE_SCOPE);
        Object runId = event.getProperties().get("run_id");
        TestRun testRun = getTestRun(runId.toString());
        context.setAttribute("testRun", testRun, ScriptContext.ENGINE_SCOPE);
        try {
            return new Event((Map<String, Object>) execute(context));
        } catch (FileNotFoundException e) {
            LoggerFactory.getLogger(getType().toString()).info("Could not find script. Deferring to default.");
            return event;
        } catch (ScriptExecutionException e) {
            throw new RuntimeException("Could not perform event enrichement.", e);
        }

	}

    private TestRun getTestRun(String runId) {
        return RepositoryServiceHolder.getRepositoryService().read("Applications/TestRuns/" + runId);
    }

	private boolean eventTypeMatches(Event event) {
		return event.getType().matches(getEventType());
	}
	
	public String getEventType() {
		return eventType;
	}

	public void setEventType(String eventType) {
		this.eventType = eventType;
	}

	public List<Store> getStores() {
		return stores;
	}

	public void setStores(List<Store> stores) {
		this.stores = stores;
	}
	
	public Set<Store> getUsedStores() {
		return usedStores;
	}
	
	public void resetUsedStores() {
		usedStores = new HashSet<Store>();
	}
	
}
