package com.xebialabs.xltest.domain;

import static com.xebialabs.deployit.engine.tasker.BlockBuilders.parallel;
import static com.xebialabs.deployit.engine.tasker.BlockBuilders.serial;
import static com.xebialabs.deployit.engine.tasker.BlockBuilders.steps;
import static com.xebialabs.deployit.plugin.api.reflect.Type.valueOf;
import static com.xebialabs.xltest.domain.Event.TYPE;
import static com.xebialabs.xltest.domain.Event.props;
import static java.lang.String.format;

import java.io.FileNotFoundException;
import java.net.URI;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.script.ScriptContext;
import javax.ws.rs.core.UriBuilder;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.engine.api.execution.StepState;
import com.xebialabs.deployit.engine.tasker.Block;
import com.xebialabs.deployit.engine.tasker.BlockBuilder;
import com.xebialabs.deployit.engine.tasker.TaskStep;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.RepositoryServiceHolder;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.xltest.plan.LeadTimePlan;
import com.xebialabs.xltest.plan.ParallelPlan;
import com.xebialabs.xltest.plan.Plan;
import com.xebialabs.xltest.plan.SerialPlan;
import com.xebialabs.xltest.plan.TestPlan;
import com.xebialabs.xltest.repository.ScriptExecutionException;
import com.xebialabs.xltest.service.EventDispatcher;
import com.xebialabs.xltest.utils.DashboardCombination;


@SuppressWarnings({"serial", "SpringJavaAutowiredMembersInspection"})
@Metadata(description = "Test run base type", root = Metadata.ConfigurationItemRoot.APPLICATIONS, virtual = true)
public class TestRun extends ScriptedConfigurationItem implements Planable<Block> {
    private static final Logger LOG = LoggerFactory.getLogger(TestRun.class);

    public static final String TEST_SET_ID = "testSetId";
    public static final String TEST_RUN = "testRun";
    public static final String EVENT_URI = "eventUri";
    public static final String ID = "run_id";
    
    @Autowired
    private transient EventDispatcher eventDispatcher;

    @Property(label = "Test set name", description = "test set")
    private TestSetDefinition testSetDefinition;
    @Property(description = "callback URI", required = false)
    private String uri;
    @Property(description = "Start time of the test run", required = false)
    private Date startTime;
    @Property(description = "Time the test run finished", required = false)
    private Date finishedTime;
    @Property(description = "Stores in which events were persisted during this test run", required = false)
	private Set<Store> usedStores;
    @Property(description = "Final qualificationResult", required = false)
    private boolean qualificationResult;

    public TestRunId getTestRunId() {
        String id = getId();
        if (getId().startsWith("Applications/TestRuns/")) {
            id = id.substring("Applications/TestRuns/".length());
        }
        return new TestRunId(id);
    }

    public TestSetDefinition getTestSetDefinition() {
        return testSetDefinition;
    }

    public void setTestSetDefinition(TestSetDefinition testSetDefinition) {
        this.testSetDefinition = testSetDefinition;
    }

	public URI getUri() {
        return UriBuilder.fromUri(uri).build();
    }

    public void setUri(URI uri) {
        this.uri = uri.toString();
    }

    public Date getStartTime() {
        return startTime;
    }

    public void setStartTime(Date startTime) {
        this.startTime = startTime;
    }

    public Date getFinishedTime() {
        return finishedTime;
    }

    public void setFinishedTime(Date finishedTime) {
        this.finishedTime = finishedTime;
    }

    public boolean isFinished() {
        return finishedTime != null;
    }
    
    public Set<Store> getUsedStores() {
    	Set<Store> stores = usedStores != null ? usedStores : Collections.<Store>emptySet();
    	if (stores.isEmpty() && !isFinished()) {
    		// lookup stores in all relevant Listeners as 'usedStores' is not set yet! It is only set
    		// when the run finishes as only then we know the full set of used stores
    		if (eventDispatcher != null) { 
	    		List<EventHandler> runSpecificEventHandlers = eventDispatcher.getRunSpecificEventHandlers(getTestRunId());
	    		for (EventHandler eh: runSpecificEventHandlers) {
	    			if (eh instanceof EventProcessor) {
	    				Set<Store> usedStoresInListener = ((EventProcessor)eh).getUsedStores();
	    				if (usedStoresInListener != null) {
	    					stores.addAll(usedStoresInListener);
	    				}
	    			}
	    		}
    		} else {
    			// hmm not even an event dispatcher. Resort to just all stores
    			RepositoryService repositoryService = RepositoryServiceHolder.getRepositoryService();
    			SearchParameters query = new SearchParameters().setType(valueOf(Store.class));
    	        List<Store> allStores = repositoryService.listEntities(query);
    	        stores.addAll(allStores);
    		}
    	}
    	return stores;
    }
    
    public boolean getQualificationResult() {
		return qualificationResult;
	}

	public void setQualificationResult(boolean qualificationResult) {
		this.qualificationResult = qualificationResult;
	}

	public List<Event> getEvents(Map queryParameters, final String sortingProperty) {
		// e.g. sortingProperty == 'pageName' then order all event alfabetically on value of that property (String only)
    	List<Event> events = getEvents(queryParameters);

        List<Event> filteredEvents = Lists.newArrayList(Iterables.filter(events, new Predicate<Event>() {
            @Override
            public boolean apply(@Nullable Event e) {
                return e.hasProperty(sortingProperty) && e.get(sortingProperty) instanceof Comparable;
            }
        }));

		Collections.sort(filteredEvents, new Comparator<Event>() {
			@Override
			public int compare(Event o1, Event o2) {
                String s1 = o1.get(sortingProperty);
                String s2 = o2.get(sortingProperty);
                return s1.compareTo(s2);
			}});
		return filteredEvents;
    }
	
	public List<Event> getEvents(Map queryParameters) {
    	List<Event> events = new ArrayList<Event>();
		for (Store s : getUsedStores()) {
			events.addAll(s.getEventsOfRun(getTestRunId(), queryParameters));
		}
		return events;
    }
    
    public List<Event> getEvents() {
		List<Event> events = new ArrayList<Event>();
		for (Store s : getUsedStores()) {
			events.addAll(s.getEventsOfRun(getTestRunId()));
		}
		return events;
    }

    public List<Event> getEventsBetweenInRunId(long startTime, long endTime, Map queryParameters, String run_id) {
        if (run_id.equals(getTestRunId().toString())) {
        	return getEventsBetween(startTime, endTime, queryParameters);
        }
        // we are asked for events associated to another run
        RepositoryService repositoryService = RepositoryServiceHolder.getRepositoryService();
        TestRun otherRun = repositoryService.read("Applications/TestRuns/" + run_id);
        if (queryParameters.containsKey("run_id")) {
        	queryParameters.put("run_id", run_id);
        }
        return otherRun.getEventsBetween(startTime, endTime, queryParameters);
    }
    
    public List<Event> getEventsBetween(long startTime, long endTime, Map queryParameters) {
        List<Event> events = new ArrayList<Event>();
        
        Set<Store> stores = getUsedStores();
		for (Store s : stores) {
            events.addAll(s.getEventsBetween(startTime, endTime, queryParameters));
        }
        return events;
    }


    //getEventsFromOtherRunsAsMapGroupedBy('suiteName', olderRuns, 'type:jobStatus', 'status:finished')
    public Map<String, List<Event>>getEventsFromRunsAsMapGroupedBy(String mapProperty, List<TestRun> runs, String... matchingKVs) {
    	Map<String, Object> properties = makeEventProperties(matchingKVs);
    	Set<String> keySet = properties.keySet();
		String[] matchingProperties = keySet.toArray(new String[keySet.size()]); // 'type', 'status'
    	Event event = new Event(properties);
    	List<Event> allEvents = getCorrespondingEventsFromOtherRuns(event, runs, matchingProperties);
    	//now group event by 'mapProperty'
    	Map<String, List<Event>> map = new HashMap<String, List<Event>>();
    	for (Event e : allEvents) {
    		String groupByValue = e.get(mapProperty);
    		if (map.get(groupByValue) == null) {
    			map.put(groupByValue, new ArrayList<Event>());
    		}
    		map.get(groupByValue).add(e);
    	}
    	// order events by ts, the newest up front
    	for (String key : map.keySet()) {
    		Collections.sort(map.get(key), new Comparator<Event>() {
    			@Override
    			public int compare(Event r1, Event r2) {
    				Long ts1 = r1.get("_ts");
    				Long ts2 = r2.get("_ts");
    				return ts2.compareTo(ts1);
    			}});
    	}
    	return map;
    }
    
    public DashboardCombination getDashboardCombination(Map queryParameters) {
    	String environment = this.getProperty("environment");
    	Date startTime = this.getProperty("startTime");
    	SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
		String theDate = format.format(startTime);
		
		Date now = this.getProperty("finishedTime");
		if (now == null) {
			// could happen when the run has been interrupted, or when server is stopped)
			now = startTime;
		}
		int hours = now.getHours();
		int minutes = now.getMinutes();
		int seconds = now.getSeconds();
		int totalseconds = (hours * 60 * 60) + (minutes * 60) + seconds ;
		Date start_of_day = new Date(now.getTime() - totalseconds * 1000);
		if (this.getProperty("finishedTime") != null && ((Date)this.getProperty("startTime")).getDay() != ((Date)this.getProperty("finishedTime")).getDay()) {
		    // we finishes in the next day, so start_of_day which is derived from finishedtime is 24 hours later
		   // and the end of the day is already held by the now computed start_of_day
		    now = start_of_day;
		    start_of_day = new Date(start_of_day.getTime() - (24 * 60 * 60) * 1000);
		}
		Map <String, Object> eventProperties = new HashMap();
		eventProperties.put("type", "jobStatus");
		eventProperties.put("status", "finished");
		if (!environment.equals("DEVELOP")) {
		    eventProperties.put("environment", environment);
		}
		long twentyFourHoursInMillis = 24 * 60 * 60 * 1000;
		long from = start_of_day.getTime();
		long to = from + twentyFourHoursInMillis;
		List<Event> finished_events = this.getEventsBetween(from, to, eventProperties);
		
		// get test run ids
		Set<String> run_ids = new HashSet<String>();
		for (Event finishEvent : finished_events) {
			run_ids.add(finishEvent.getTestRunId().toString());
		}
		
		List<Event> result_events = new ArrayList<Event>();
		eventProperties = new HashMap();
		eventProperties.put("type", "result");
		addQueryParamsToEventProperties(eventProperties, queryParameters);
		for (String theRunId : run_ids) {
			eventProperties.put("run_id", theRunId);
			List<Event> eventsInRunInTimeframe = this.getEventsBetweenInRunId(from, to, eventProperties, theRunId);
			result_events.addAll(eventsInRunInTimeframe);
		}
		
		List<Event> timeouts = new ArrayList<Event>();
		for (Event finishEvent : finished_events) {
			if (isTimeoutEvent(finishEvent) && matchingQueryParameters(finishEvent, queryParameters)) {
				timeouts.add(finishEvent);
			}
		}
		
		
		timeouts = mostRecentSlice(timeouts);
		
		DashboardCombination combination = aggregateResults(result_events, timeouts);
		int jobStatus = 0;
		int results = 0;
		for (String uc : combination.getEventMap().keySet()) {
			Event evInMap = combination.getEventMap().get(uc);
			String type = evInMap.getType();
			if (type.equals("jobStatus")) {
				jobStatus++;
			}
			if (type.equals("result")) {
				results++;
			}
		}
		return combination;
    }
    
    private DashboardCombination aggregateResults(List<Event> events, List<Event> timeouts) {
    	DashboardCombination combinedResult = new DashboardCombination();
		Map<String, Event> agg = new HashMap<String, Event>();
		Set<Event> appliedTimeouts = new HashSet<Event>();
		Set<Event> oldTimeoutsThatNowHaveAResult = new HashSet<Event>();
		String keyAttr = "pageName";

	    for (Event e : events) {
	        String key = e.get(keyAttr);
	        Event other = agg.get(key);
	        if (other == null || (Long)other.get("_ts") < (Long)e.get("_ts")) {
	            agg.put(key, e);
	        }
	        // if a later run of a usecase ran into timeout, mark the page as "timeout" by replacing the test result event by the 
	        // timeout event. Note this will happen for ALL test results in the usecase. 
	        for (Event to : timeouts) {
	        	// Note we need to add a '.' otherwise 
	            if (key.startsWith((String)to.get("suiteName") + ".") && (Long)to.get("_ts") > (Long)e.get("_ts")) {
	                agg.put(key, to);
	                appliedTimeouts.add(to);
	                break;
	            }
	            if (key.startsWith((String)to.get("suiteName") + ".") && (Long)to.get("_ts") < (Long)e.get("_ts")) {
	                oldTimeoutsThatNowHaveAResult.add(to);
	                break;
	            }
	        }
	    }
		
	    combinedResult.setAppliedTimeouts(appliedTimeouts);
	    combinedResult.setOldTimeoutsThatNowHaveAResult(oldTimeoutsThatNowHaveAResult);
	    combinedResult.setEventMap(agg);
	    combinedResult.setTimeouts(timeouts);
		return combinedResult;
	}
    
	private List<Event> mostRecentSlice(List<Event> timeouts) {
	    Map<String, Event> slice_timeout_dict = new HashMap<String, Event>();
	    for (Event to : timeouts) {
	        if (slice_timeout_dict.keySet().contains(to.get("suiteName"))) {
	            // replace if newer
	        	Event  other_to = slice_timeout_dict.get(to.get("suiteName"));
	            if ((Long)to.get("_ts") > (Long)other_to.get("_ts")) {
	                slice_timeout_dict.put((String)to.get("suiteName"), to);
	            } 
	        } else {
	            slice_timeout_dict.put((String)to.get("suiteName"), to);
	        }
	    }
	    List<Event> all = new ArrayList<Event>();
	    all.addAll(slice_timeout_dict.values());
	    return all;
	}
    
	private boolean matchingQueryParameters(Event e, Map<String, Object> queryParameters) {
		for (String k : queryParameters.keySet()) {
			if (!e.hasProperty(k) || !e.get(k).equals(queryParameters.get(k))) {
				return false;
			}
		}
		return true;
	}
    
	private boolean isTimeoutEvent(Event finishEvent) {
		return "timeout".equals(finishEvent.get("reason"));
	}
    
	private void addQueryParamsToEventProperties(Map<String, Object> eventProperties, Map<String, Object> queryParameters) {
		for (String key : queryParameters.keySet()) {
			eventProperties.put(key, queryParameters.get(key));
		}
	}
    
    public List<Event> getResultEventsFromDashboardCombination(DashboardCombination combination, String result) {
    	Map<String, Event> eventMap = combination.getEventMap();
    	List<Event> events = new ArrayList<Event>();
    	for (String testCase : eventMap.keySet()) {
    		Event event = eventMap.get(testCase);
    		if (event.hasProperty("result") && result.equals(event.get("result"))) {
    			events.add(event);
    		}
    	}
    	return events;
    }
    
    public Set<Event> getJobStatusEventsFromDashboardCombination(DashboardCombination combination, String type) {
    	Map<String, Event> eventMap = combination.getEventMap();
    	Set<Event> events = new HashSet<Event>();
    	for (String testCase : eventMap.keySet()) {
    		Event event = eventMap.get(testCase);
    		if (event.getType().equals(type)) {
    			events.add(event);
    		}
    	}
    	return events;
    }
    
    public List<Event> getNonAppliedTimeoutsNotHavingNewerResultsFromDashboardCombination(DashboardCombination combination) {
    	List<Event> events = new ArrayList<Event>();
    	Set<String> pageNamesOfAppliedTimeouts = new HashSet<String>();
    	Set<String> pageNamesOfTimeoutsNowHavingResuts = new HashSet<String>();
    	for (Event ev : combination.getAppliedTimeouts()) {
    		String pageName = ev.get("suiteName");
    		pageNamesOfAppliedTimeouts.add(pageName);
    	}
    	for (Event oldTimeoutThatNowHasResults : combination.getOldTimeoutsThatNowHaveAResult()) {
    		String pageName = oldTimeoutThatNowHasResults.get("suiteName");
    		pageNamesOfTimeoutsNowHavingResuts.add(pageName);
    	}
    	for (Event timeout : combination.getTimeouts()) {
    		String pageName = timeout.get("suiteName");
    		// not applied and not having newer results
			if (!pageNamesOfAppliedTimeouts.contains(pageName) && !pageNamesOfTimeoutsNowHavingResuts.contains(pageName)) {
				events.add(timeout);
			}
    	}
    	return events;
    }
    
    private Map<String, Object> makeEventProperties(String[] matchingKVs) {
		// 'type:jobStatus', 'status:finished'
    	Map<String, Object> eventProperties = new HashMap<String, Object>();
    	for (int i = 0; i < matchingKVs.length; i++) {
    		Pair p = parseKV(matchingKVs[i]);
    		eventProperties.put(p.getKey(), p.getValue());
    	}
		return eventProperties;
	}

	private Pair parseKV(String kv) {
		// key:value
		String key = kv.substring(0, kv.indexOf(':'));
		String value = kv.substring(key.length() + 1);
		return new Pair(key, value);
	}

	public List<Event> getCorrespondingEventsFromOtherRuns(Event event, List<TestRun> otherRuns, String... matchingProperties) {
    	List<Event> events = new ArrayList<Event>();
    	Map queryParameters = makeQueryParameters(event, matchingProperties);
    	for (TestRun previousRun : otherRuns) {
    		List<Event> eventsOfPreviousRun = previousRun.getEvents(queryParameters);
    		LOG.debug("Found " + eventsOfPreviousRun.size() + " events with matching properties in a previous run");
    		events.addAll(eventsOfPreviousRun);
    	}
    	LOG.debug("Found " + events.size() + " CorrespondingEventsFromOlderRuns. They are:");
    	for (Event e : events) {
    		LOG.debug("    " + e.toString());
    	}
    	return events;
    }
    
    public List<Event> getCorrespondingEventsFromOlderRuns(Event event, int max, String... matchingProperties) {
    	List<Event> events = new ArrayList<Event>();
    	List<TestRun> previousRuns = getPreviousRuns(max);
    	LOG.debug("While getting events from max " + max + " previous Runs, I got " + previousRuns.size() + " such runs");
    	Map queryParameters = makeQueryParameters(event, matchingProperties);

    	for (TestRun previousRun : previousRuns) {
    		List<Event> eventsOfPreviousRun = previousRun.getEvents(queryParameters);
    		LOG.debug("Found " + eventsOfPreviousRun.size() + " events with matching properties in a previous run");
    		events.addAll(eventsOfPreviousRun);
    	}
    	LOG.debug("Found " + events.size() + " CorrespondingEventsFromOlderRuns. They are:");
    	for (Event e : events) {
    		LOG.debug("    " + e.toString());
    	}
    	return events;
    }

	private Map makeQueryParameters(Event event, String... matchingProperties) {
		Map queryParameters = new HashMap();
		if (matchingProperties != null && matchingProperties.length != 0) {
			for (int i = 0; i < matchingProperties.length; i++) {
				String propertyKey = matchingProperties[i];
				queryParameters.put(propertyKey, event.get(propertyKey));
			}
		}
		return queryParameters;
	}
	
	public List<TestRun> getPreviousRunsIncludingThisOne(int max) {
		List<TestRun> runs = getPreviousRuns(max);
		runs.add(this);
		return runs;
	}

	public List<TestRun> getPreviousRuns(int max) {
    	RepositoryService repositoryService = RepositoryServiceHolder.getRepositoryService();
    	SearchParameters query = new SearchParameters().setType(valueOf(TestRun.class));
        List<TestRun> allTestRuns = repositoryService.listEntities(query);
        String testSetDefinitionId = getTestSetDefinition().getId();
        List<TestRun> allTestRunsWithTheSameTestSetDefinition = new ArrayList<TestRun>();
        for (TestRun testRun : allTestRuns) {
        	if (testRun.getTestSetDefinition().getId().equals(testSetDefinitionId)) {
        		allTestRunsWithTheSameTestSetDefinition.add(testRun);
        	}
        }
        // now sort (in reverse order!!!)
        Collections.sort(allTestRunsWithTheSameTestSetDefinition, new Comparator<TestRun>() {
			@Override
			public int compare(TestRun r1, TestRun r2) {
				return r2.getStartTime().compareTo(r1.getStartTime());
			}});
        // get the last <max> that were before this run itself
        List<TestRun> previousTestRuns = new ArrayList<TestRun>();
        Date startTime = getStartTime();
        int cnt = 0;
        for (TestRun testRun : allTestRunsWithTheSameTestSetDefinition) {
        	if (testRun.getStartTime().before(startTime)) {
        		previousTestRuns.add(testRun);
        		cnt++;
        	}
        	if (cnt >= max) {
        		break;
        	}
        }
        LOG.debug("I got " + previousTestRuns.size() + " previous runs");
        return previousTestRuns;
    }
    
    public Set<Store> setUsedStores(Set<Store> stores) {
    	return usedStores = stores;
    }

    public void setEventDispatcher(EventDispatcher eventDispatcher) {
        this.eventDispatcher = eventDispatcher;
    }

    /**
     * Retrieve executable steps.
     * @return an execution plan as a TaskEngine Block.
     */
    public Block plan() {
        Plan plan = getTestSetDefinition().plan();
        return serial("Test run " + getId(), Lists.newArrayList(
                    steps("Initialize listeners", taskStep(new InitializeListenersStep())),
                    steps("Start test run", taskStep(new EventEmittingStep(newStartTestRunEvent()))),
                    blockBuilder(plan),
                    steps("Finish test run", taskStep(new EventEmittingStep(newFinishTestRunEvent()))),
                    steps("Unload listeners", taskStep(new RemoveListenersStep()))
                )).build();
    }
    
    public BlockBuilder planWithoutBuilding() {
        Plan plan = getTestSetDefinition().plan();
        return serial("Test run " + getId(), Lists.newArrayList(
                    steps("Initialize listeners", taskStep(new InitializeListenersStep())),
                    steps("Start test run", taskStep(new EventEmittingStep(newStartTestRunEvent()))),
                    blockBuilder(plan),
                    steps("Finish test run", taskStep(new EventEmittingStep(newFinishTestRunEvent()))),
                    steps("Unload listeners", taskStep(new RemoveListenersStep()))
                ));
    }

    private BlockBuilder blockBuilder(Plan plan) {
        if (plan instanceof TestPlan) {
        	LOG.info("Plan is an instance of TestPlan, its class: " + plan.getClass().getName());
            TestSetDefinition testSet = ((TestPlan) plan).getTestSet();
            return testStep((TestPlan) plan);
        } else if (plan instanceof ParallelPlan) {
        	LOG.info("Plan is an instance of ParallelPlan, its class: " + plan.getClass().getName());
            return parallel("In parallel", blockBuilders(((ParallelPlan) plan).getPlans()));
        } else if (plan instanceof SerialPlan) {
        	LOG.info("Plan is an instance of SerialPlan, its class: " + plan.getClass().getName());
            return serial("In sequence", blockBuilders(((SerialPlan) plan).getPlans()));
        } else if (plan instanceof LeadTimePlan) {
        	LeadTimePlan leadTimePlan = (LeadTimePlan)plan;
        	return getLeadTimeBlockBuilder(leadTimePlan.getLeadTime());
        } else {
            throw new RuntimeException("Unknown plan type: " + plan);
        }
    }

    protected List<StepState> taskStep(Step step) {
        return Lists.<StepState>newArrayList(new TaskStep(step));
    }

    private List<BlockBuilder> blockBuilders(List<Plan> plans) {
        List<BlockBuilder> blockList = new LinkedList();
        for (Plan plan : plans) {
            blockList.add(blockBuilder(plan));
        }
        return blockList;
    }

    protected BlockBuilder testStep(TestPlan testPlan) {
        return steps(format("Test run for %s", getTestSetDefinition().getId()), getSteps(testPlan));
    }
    
    protected BlockBuilder getLeadTimeBlockBuilder(int secs) {
        return steps(format("Wait for other processes to start..."), getWaitStep(secs));
    }


    protected List<StepState> getSteps(TestPlan testPlan) {
        return Lists.<StepState>newArrayList(
                new TaskStep(new ScriptExecutionStep(testPlan))
        );
    }
    
    protected List<StepState> getWaitStep(int secs) {
        return Lists.<StepState>newArrayList(
                new TaskStep(new WaitStep(secs))
        );
    }

    protected Map<String, Object> newStartTestRunEvent() {
        return props(
                TYPE, Event.START_TEST_RUN,
                ID, getTestRunId().toString(),
//                TEST_RUN, this,
                TEST_SET_ID, getTestSetDefinition().getId(),
                EVENT_URI, getUri().toString());
    }

    protected Map<String, Object> newFinishTestRunEvent() {
        return props(
                TYPE, Event.FINISH_TEST_RUN,
                ID, getTestRunId().toString(),
                TEST_SET_ID, getTestSetDefinition().getId());
    }

    protected class EventEmittingStep implements Step {
        private final Map<String, Object> properties;

        public EventEmittingStep(Map<String, Object> properties) {
            this.properties = properties;
        }

        @Override
        public int getOrder() {
            return 0;
        }

        @Override
        public String getDescription() {
            return format("Send '%s' event", properties.get("type"));
        }

        @Override
        public StepExitCode execute(ExecutionContext executionContext) throws Exception {
        	eventDispatcher.notify(getTestRunId(), new Event(properties));
            return StepExitCode.SUCCESS;
        }
    }
    
    protected class InitializeListenersStep implements Step {
		@Override
		public StepExitCode execute(ExecutionContext executionContext) throws Exception {
            if (executionContext.getRepository() == null) {
                LOG.error("Repository not set on ExecutionContext. Can not initialize Event processors");
                return StepExitCode.FAIL;
            }
            TestRunId testRunId = getTestRunId();
            Type type = valueOf(EventProcessor.class);
            List<EventProcessor> listeners = executionContext.getRepository().search(type);
	        if (listeners != null) {
	        	LOG.info("I am registering " + listeners.size() + " event processors as handler in eventDispatcher for run " + testRunId.toString());
	        	for (EventProcessor l : listeners) {
	        		l.resetUsedStores();
	        		eventDispatcher.registerHandler(testRunId, l);
	        	}
	        }
			return StepExitCode.SUCCESS;
		}

		@Override
		public String getDescription() {
			return "Initialize event listeners";
		}

		@Override
		public int getOrder() {
			return 0;
		}
    }
    
    protected class RemoveListenersStep implements Step {
    	@Override
		public StepExitCode execute(ExecutionContext executionContext) throws Exception {
    		collectUsedStoresAndPersistWithTestRun(executionContext);
    		eventDispatcher.unregisterAll(getTestRunId());
			return StepExitCode.SUCCESS;
		}

		private void collectUsedStoresAndPersistWithTestRun(ExecutionContext executionContext) {
			List<EventHandler> runSpecificEventHandlers = eventDispatcher.getRunSpecificEventHandlers(getTestRunId());
    		for (EventHandler eh: runSpecificEventHandlers) {
    			if (eh instanceof EventProcessor) {
    				Set<Store> usedStoresInListener = ((EventProcessor)eh).getUsedStores();
    				if (usedStoresInListener != null) {
    					usedStores.addAll(usedStoresInListener);
    				}
    			}
    		}
            LOG.info("Found stores for this Test run: " + usedStores);
		}

		@Override
		public String getDescription() {
			return "Deregister event listeners";
		}

		@Override
		public int getOrder() {
			return 0;
		}
    }
    
    protected class WaitStep implements Step {
    	private int secs = 0;
    	
    	public WaitStep(int secs) {
    		this.secs = secs;
    	}

		@Override
		public int getOrder() {
			return 0;
		}

		@Override
		public String getDescription() {
			return "Wait for " + secs + " seconds to allow parallel processes to start up";
		}

		@Override
		public StepExitCode execute(ExecutionContext ctx) throws Exception {
			Thread.sleep(1000 * secs);
			return StepExitCode.SUCCESS;
		}
    	
    }

    public class ScriptExecutionStep implements Step {

        private TestPlan testPlan;

        public ScriptExecutionStep(TestPlan testPlan) {
            this.testPlan = testPlan;
        }

        @Override
        public int getOrder() {
            return 0;
        }

        @Override
        public String getDescription() {
            return format("Executing Test set '%s' on local environment", getTestSetDefinition().getId());
        }
        @Override
        public StepExitCode execute(ExecutionContext executionContext) throws Exception {
            ScriptContext context = getScriptContext();
            context.setAttribute("testSet", testPlan.getTestSet(), ScriptContext.ENGINE_SCOPE);
            context.setAttribute("commandLine", testPlan.getCommandLine(), ScriptContext.ENGINE_SCOPE);
            context.setAttribute("repository", executionContext.getRepository(), ScriptContext.ENGINE_SCOPE);
            context.setAttribute("eventDispatcher", eventDispatcher, ScriptContext.ENGINE_SCOPE);

            try {
                Object exitCode = TestRun.this.execute(context);
                if (exitCode == null || (exitCode instanceof Number && ((Number) exitCode).intValue() == 0)) {
                    return StepExitCode.SUCCESS;
                }
            } catch (FileNotFoundException e) {
                LOG.error("Could not find script.");
            } catch (ScriptExecutionException e) {
                LOG.error("Could not perform script.", e);
            } catch (Throwable t) {
                LOG.error("Could not perform script.", t);
            }
            return StepExitCode.FAIL;
        }

    }
	
	public class Pair {
		private String key;
		private Object value;
		
		public Pair(String key, Object value) {
			super();
			this.key = key;
			this.value = value;
		}
		public String getKey() {
			return key;
		}
		public void setKey(String key) {
			this.key = key;
		}
		public Object getValue() {
			return value;
		}
		public void setValue(Object value) {
			this.value = value;
		}
		
	}
}
