package com.xebialabs.xltest.resources;

import java.io.PrintWriter;

import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;

import org.codehaus.jettison.json.JSONException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.xltest.domain.Event;
import com.xebialabs.xltest.domain.TestRun;
import com.xebialabs.xltest.domain.TestRunId;
import com.xebialabs.xltest.domain.EventHandler;
import com.xebialabs.xltest.repository.TestRuns;
import com.xebialabs.xltest.serialize.ListOfEventsWriter;
import com.xebialabs.xltest.service.EventRegistry;
import com.xebialabs.xltest.service.TestRunStatusProvider;

@Controller
@Path("/test")
@Produces("text/event-stream")
public class EventFeedController {
    private static final Logger LOG = LoggerFactory.getLogger(EventFeedController.class);
    private final EventRegistry eventRegistry;
    private final RepositoryService repository;
    private final TestRunStatusProvider testRunStatusProvider;
    private final ListOfEventsWriter serializer = new ListOfEventsWriter();

    @Autowired
    public EventFeedController(EventRegistry eventRegistry, TestRunStatusProvider testRunStatusProvider, RepositoryService repository) {
        this.eventRegistry = eventRegistry;
        this.testRunStatusProvider = testRunStatusProvider;
        this.repository = repository;
    }

    @GET
    @Path("/{testRunId:.+}")
    public void getTestRun(@PathParam("testRunId") String testRunId, @Context HttpServletResponse response) throws Exception {
        final TestRunId requestedTestRunId = new TestRunId(testRunId);

        response.setContentType("text/event-stream");
        response.setCharacterEncoding("UTF-8");
        response.addHeader("Cache-Control", "no-cache");
        response.addHeader("Connection", "keep-alive");

        final PrintWriter writer = response.getWriter();

        TestRun testRun = repository.read(TestRuns.TEST_RUNS_ROOT + requestedTestRunId.toString());
        for (Event event : testRun.getEvents()) {
            sendEvent(event, writer);
        }

        // Event handler is removed at the end of the test run.
        eventRegistry.registerHandler(requestedTestRunId, new EventHandler() {
            @Override
            public void onReceive(Event event) throws Exception {
                LOG.info("Received event: " + event);

                sendEvent(event, writer);
            }
        });

        // TODO: We need something until we can manage the response connection ourselves.
        while (testRunStatusProvider.isRunning(requestedTestRunId)) {
            Thread.sleep(1000);
        }

        closeEventStream(writer);
    }

    private void sendEvent(Event event, PrintWriter writer) throws JSONException {
        writer.write("event: event\n");
        writer.write("data: " + serializer.toJson(event) + "\n\n");
        writer.flush();
    }

    private void closeEventStream(PrintWriter writer) {
        writer.write("event: close\n");
        writer.write("data: \n\n");
        writer.close();
    }

}
