package com.xebialabs.xltest.resources;

import java.io.IOException;
import java.net.URI;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.HeaderParam;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriInfo;

import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Interval;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.jayway.jsonpath.spi.JsonProviderFactory;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.xltest.domain.Event;
import com.xebialabs.xltest.domain.TestRun;
import com.xebialabs.xltest.domain.TestRunId;
import com.xebialabs.xltest.domain.TestSetDefinition;
import com.xebialabs.xltest.repository.TestRuns;
import com.xebialabs.xltest.service.EventDispatcher;
import com.xebialabs.xltest.service.TestRunStatusProvider;
import com.xebialabs.xltest.service.TestRunner;

@Controller
@Path("/test")
@Produces(MediaType.APPLICATION_JSON)
@Consumes(MediaType.APPLICATION_JSON)
public class TestController {
    private static final Logger LOG = LoggerFactory.getLogger(TestController.class);

    private final TestRunner testRunner;
    private final TestRunStatusProvider testRunStatusProvider;
    private final TestRuns testRuns;
    private final EventDispatcher eventDispatcher;

    @Autowired
	public TestController(TestRunner testRunner, EventDispatcher eventDispatcher, TestRunStatusProvider testRunStatusProvider, TestRuns testRuns) {
        this.testRunner = testRunner;
        this.testRunStatusProvider = testRunStatusProvider;
        this.testRuns = testRuns;
        this.eventDispatcher = eventDispatcher;
    }

    /**
     * Start a top-level test run.
     *
     * @param uriInfo
     * @param testRun
     * @return
     * @throws IOException
     * @throws InterruptedException
     */
    @POST
    @Path("/")
	public TestRun fireTest(final TestRun testRun, @HeaderParam("xltest-base") String xlTestBase, @Context final UriInfo uriInfo) throws IOException, InterruptedException {
        URI xlTestBaseUri;
        if (xlTestBase != null) {
            xlTestBaseUri = URI.create(xlTestBase + "/test");
        } else {
            xlTestBaseUri = uriInfo.getAbsolutePath();
        }
        return testRunner.run(testRun, xlTestBaseUri);
	}

    /**
     * Get all test runs.
     *
     * @return
     * @throws IOException
     * @throws InterruptedException
     */
    @GET
    @Path("/")
    public Collection<TestRun> getTestRuns(@QueryParam("p") List<String> queries, @QueryParam("date") String queryDate) {
        // , @QueryParam("environment") String queryEnvironment
    	SearchParameters searchParameters = new SearchParameters();
        LOG.info("Query test runs for fields {} on date {}", queries, queryDate);
        addQueryParameters(searchParameters, queries);
        Collection<TestRun> runs = testRuns.getTestRuns(searchParameters);

        if (queryDate != null) {
            DateTime startTime = DateTime.parse(queryDate);
            LOG.info("Filtering test runs from {}", startTime);
            final Interval dateRange = new Interval(startTime, Duration.standardDays(1));
            return Collections2.filter(runs, new Predicate<TestRun>() {
                @Override public boolean apply(TestRun testRun) {
                    return testRun.getStartTime() != null && dateRange.contains(testRun.getStartTime().getTime());
                }
            });
        }
        return runs;
    }

    void addQueryParameters(SearchParameters searchParameters, List<String> queries) {
        for (String q: queries) {
            String[] nameValue = q.split(":", 2);
            if (nameValue.length != 2) {
                throw new IllegalArgumentException("query should have format 'field:value'");
            }
            searchParameters.addProperty(nameValue[0].trim(), nameValue[1].trim());
        }
    }


    /**
     * Receive events for test runs. A test run should have been started by post'ing to /xltest/test/.
     * Test run id's can be nested, e.g. 1/2/3.
     *
     * Note that, if a url ends on a '/', that '/' is not passed as part of the testRunId.
     *
     * @param testRunId
     * @param event
     * @return
     */
    @POST
    @Path("/{testRunId:.+}")
    public String receiveEvent(@Context HttpServletRequest request, @PathParam("testRunId") String testRunId, Event event) {
        LOG.info("Received event {} for run {}", event.getType(), testRunId);

        if ("STOP".equals(event.getType())) {
            LOG.info("Attempting to abort test run {} due to STOP event.", testRunId);
            testRunner.abort(new TestRunId(testRunId));
        } else {
            eventDispatcher.notify(new TestRunId(testRunId), new Event(event, Event.props(
                    "_host", request.getRemoteHost(),
                    "run_id", new TestRunId(testRunId).getTopLevel().toString())));
        }
        return "";
    }

    @GET
    @Path("/{testRunId:[\\w-]+}")
    public String getTestRun(@PathParam("testRunId") String testRunId) {
        Map<String, Object> resultObject = new TreeMap();
        boolean running = testRunStatusProvider.isRunning(new TestRunId(testRunId));
        resultObject.put("running", running);
        if (!running) {
            // TODO: -AJM- Use quantification from TestRun
            resultObject.putAll(testRuns.getQualification(new TestRunId(testRunId)));
        }
        return JsonProviderFactory.createProvider().toJson(resultObject);
    }

}

