package com.xebialabs.xltest.domain;

import java.io.Serializable;
import java.util.Collections;
import java.util.Map;
import java.util.TreeMap;
import java.util.UUID;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Event implements Serializable {
    public static final String RUN_ID = "runId";
    public static final String TIMESTAMP = "timestamp";
    public static final String TYPE = "type";
    public static final String TEST_SPECIFICATION = "testSpecification";
    public static final String LAST_MODIFIED = "lastModified";

    public static final String EXECUTION_STARTED = "executionStarted";
    public static final String EXECUTION_FINISHED = "executionFinished";

    public static final String IMPORT_STARTED = "importStarted";
    public static final String IMPORT_FINISHED = "importFinished";

    public static final String FUNCTIONAL_RESULT = "functionalResult";
    public static final String PERFORMANCE_RESULT = "performanceResult";

    public static final String QUALIFICATION_COMPUTED = "qualificationComputed";

    public static final String UNKNOWN_EVENT_TYPE = "UNKNOWN";

    private static final Logger LOG = LoggerFactory.getLogger(Event.class);

    public static final String QUALIFICATION = "qualification";
    public static final String FAILURE_REASON = "failureReason";

    // For composite execution:
    public static final String CHILD_RUN_ID = "childRunId";
    public static final String CHILD_FINISHED = "childFinished";
    public static final String CHILD_STARTED = "childStarted";

    private final Map<String, Object> properties;

    public Event(Map<String, Object> properties) {
        this.properties = new TreeMap(properties);
        if (!this.properties.containsKey(TIMESTAMP)) {
            update(TIMESTAMP, System.currentTimeMillis());
        }
    }

    public Event(String type, Map<String, Object> properties) {
        this(properties);
        update(TYPE, type);
    }

    public Event(Event event, Map<String, Object> properties) {
        this(event.getProperties());
        this.properties.putAll(properties);
    }

    public Event(UUID testRunId, Event event, Map<String, Object> properties) {
        this(event.getProperties());
        this.properties.putAll(properties);
        this.properties.put(RUN_ID, testRunId.toString());
    }

    public Event(UUID testRunId, String type, Map<String, Object> properties) {
        this(properties);
        update(RUN_ID, testRunId.toString());
        update(TYPE, type);
    }

    public String getType() {
        return (String) properties.get(TYPE);
    }

    private Event update(String key, Object value) {
        properties.put(key, value);
        return this;
    }

    public <T> T get(String key) {
        Object value = properties.get(key);
        if (value == null) {
            LOG.warn(key + " not found in event " + this);
        }
        return (T) value;
    }
    
    public boolean hasProperty(String key) {
        return properties.get(key) != null;
    }

    public Map<String, Object> getProperties() {
        return Collections.unmodifiableMap(properties);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Event event = (Event) o;

        if (!properties.equals(event.properties)) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = properties.hashCode();
        return result;
    }

    @Override
    public String toString() {
        return "Event" + properties;
    }

    public static Map<String, Object> props(Object... keyValues) {
        assert keyValues.length % 2 == 0;
        Map<String, Object> m = new TreeMap();
        for (int i = 0; i < keyValues.length - 1; i += 2) {
            m.put(keyValues[i].toString(), keyValues[i+1]);
        }
        return m;
    }

    public UUID getTestRunId() {
        Object testRunId = get(RUN_ID);
        if (testRunId instanceof UUID) {
            return (UUID) testRunId;
        }
        try {
            return UUID.fromString(testRunId.toString());
        } catch (IllegalArgumentException e) {
            throw new IllegalArgumentException("Event malformed, no valid runId: " + toString());
        }
    }

	public static String normalizeName(String pageName, Character separator) {
		if (pageName.indexOf(separator) > -1) {
			return pageName.replace(separator, ';');
		}
		return pageName;
	}
}
