package com.xebialabs.xltest.domain;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.xltest.repository.ScriptExecutionException;
import com.xebialabs.xltest.repository.ScriptExecutor;
import com.xebialabs.xltest.repository.ScriptExecutorHolder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.script.ScriptContext;
import javax.script.SimpleScriptContext;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.Writer;

@Metadata(virtual = true)
public class ScriptedConfigurationItem extends BaseConfigurationItem {

    @Property(description = "location of the script on the class path", required = false)
    private String scriptLocation;

    public void setScriptLocation(String scriptLocation) {
        this.scriptLocation = scriptLocation;
    }

    public String getScriptLocation() {
        return scriptLocation;
    }

    protected <T> T execute(ScriptContext scriptContext, BaseConfigurationItem invoker) throws ScriptExecutionException, FileNotFoundException {
        Writer executionLog = new LogWriter(getType().toString());
        ResultHolder<T> resultHolder = new ResultHolder();
        scriptContext.setAttribute("resultHolder", resultHolder, ScriptContext.ENGINE_SCOPE);

        // TODO: Inject ScriptExecutor via DI
        ScriptExecutor executor = ScriptExecutorHolder.getScriptExecutor();
        if (executor == null) {
            throw new IllegalStateException("Script execution has not been initialized");
        }

        scriptContext.setWriter(executionLog);
        scriptContext.setErrorWriter(executionLog);

        try {
            executor.evalScriptedCi(invoker, scriptContext);
        } finally {
            LoggerFactory.getLogger(getType().toString()).info("Script executed, result: {}", resultHolder.result);
            try {
                executionLog.close();
            } catch (IOException e) {
                // silent ignore...
            }
        }

        return resultHolder.result;
    }

    protected ScriptContext newScriptContext() {
        ScriptContext context = new SimpleScriptContext();
        context.setAttribute("self", this, ScriptContext.ENGINE_SCOPE);
        return context;
    }

    public static class ResultHolder<T> {
        private T result;

        public void setResult(T result) {
            this.result = result;
        }
        public T getResult() {
            return result;
        }
    }
    
    public static class ProcessHolder<T> {
        private T process;

        public void setProcess(T process) {
            this.process = process;
        }
        public T getProcess() {
            return process;
        }
    }



    public static class LogWriter extends Writer {
        private final Logger log;
        private final StringBuilder buffer;

        public LogWriter(String type) {
            log = LoggerFactory.getLogger(type);
            buffer = new StringBuilder(512);
        }

        @Override
        public void write(char[] cbuf, int off, int len) {
            for (int i = off; i < off + len; i++) {
                char c = cbuf[i];
                if (c == '\n' || c == '\r') {
                    log();
                } else {
                    buffer.append(c);
                }
            }
        }

        @Override
        public void flush() {
            // no-op
        }

        @Override
        public void close() {
            log();
        }

        private void log() {
            if (buffer.length() > 0) {
                log.info(buffer.toString());
                buffer.delete(0, buffer.length());
            }
        }
    }

}
