package com.xebialabs.xltest.domain;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.local.LocalConnection;
import com.xebialabs.xltest.utils.glob.FileMatcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.UUID;

@Metadata(description = "A test specification", root = Metadata.ConfigurationItemRoot.CONFIGURATION)
public class TestSpecification extends BaseTestSpecification {

    private static final Logger LOG = LoggerFactory.getLogger(TestSpecification.class);

    @Property(description = "Working directory for test execution")
    private String workingDirectory;

    @Property(description = "Name of the test tool, such as JMeter, FitNesse or Cucumber")
    private String testToolName;

    @Property(description = "Host where is test tool process is executed")
    private Host host;

    @Property(description = "Pattern used to look for ")
    private String searchPattern;

    @Property(description = "The report used to determine the outcome of a test run (success/fail)", required = false)
    private Qualification qualification;

    public String getWorkingDirectory() {
        return workingDirectory;
    }

    public void setWorkingDirectory(String workingDirectory) {
        this.workingDirectory = workingDirectory;
    }

    public String getTestToolName() {
		return testToolName;
	}

	public void setTestToolName(String testToolName) {
		this.testToolName = testToolName;
	}

	public Host getHost() {
		return host;
	}

	public void setHost(Host host) {
		this.host = host;
	}

    public String getSearchPattern() {
        return searchPattern;
    }

    public void setSearchPattern(String searchPattern) {
        this.searchPattern = searchPattern;
    }

    public OverthereFile getRemotePath() {
        OverthereConnection hostConnection = getHost() != null ? getHost().getConnection() : LocalConnection.getLocalConnection();
        return hostConnection.getFile(workingDirectory);
    }

    public void doImport(UUID testRunId, TestTool testTool, EventHandler eventHandler) {
        doImport(testRunId, testTool, getRemotePath(), eventHandler);
    }

    public void doImport(UUID testRunId, TestTool testTool, OverthereFile remotePath, EventHandler eventHandler) {
        doImport(testRunId, testTool, remotePath, eventHandler, new ProgressCallback() {
            @Override
            public void update(int percentage) {
                /* NOOP */
            }
        });
    }

    public void doImport(UUID testRunId, TestTool testTool, OverthereFile remotePath, EventHandler eventHandler, ProgressCallback progressCallback) {

        ImportHandler handler = new ImportHandler(eventHandler);

        try {
            int counter = 0;
            List<Importable> importables = testTool.findImportables(remotePath, new FileMatcher(remotePath, getSearchPattern()));
            progressCallback.update((++counter) / (importables.size() + 1));

            for (Importable importable : importables) {
                try {
                    importable.doImport(testRunId, handler);
                    int percentage = (++counter * 100) / (importables.size() + 1);
                    progressCallback.update(percentage);
                } catch (Exception e) {
                    LOG.error("Unable to import test run" + importable, e);
                }
            }
        } finally {
            remotePath.getConnection().close();
        }
    }

    public class ImportHandler implements EventHandler {

        private EventHandler eventHandler;

        public int count = 0;

        public ImportHandler(EventHandler eventHandler) {
            this.eventHandler = eventHandler;
        }

        @Override
        public void onReceive(Event event) throws Exception {
            // Add "run specific" properties here
            LOG.debug("Importing event {}", event);
            eventHandler.onReceive(new Event(event, Event.props(Event.TEST_SPECIFICATION, getName())));
            count++;
        }
    }

    public static interface ProgressCallback {

        void update(int percentage);
    }

    public Qualification getQualification() {
        return this.qualification;
    }

    public void setQualification(Qualification qualification) {
        this.qualification = qualification;
    }

}
