package com.xebialabs.xltest.repository;

import com.xebialabs.xltest.domain.TestTool;
import nl.javadude.scannit.Scannit;
import nl.javadude.scannit.scanner.SubTypeScanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Repository;

import java.lang.reflect.Modifier;
import java.util.*;

@Repository
public class TestTools {
    private static final Logger LOG = LoggerFactory.getLogger(TestTools.class);

    private final Map<String, TestTool> testTools;

    public TestTools() {
        nl.javadude.scannit.Configuration config = nl.javadude.scannit.Configuration.config()
                .with(new SubTypeScanner())
                .scan("com.xebialabs.xltest");
        Scannit scannit = new Scannit(config);
        Map<String, TestTool> testTools = new HashMap<>();
        for (Class<? extends TestTool> testToolClass : scannit.getSubTypesOf(TestTool.class)) {
            if (Modifier.isAbstract(testToolClass.getModifiers())) {
                continue;
            }
            try {
                TestTool testTool = testToolClass.newInstance();
                String name = testTool.getName();
                if (testTools.containsKey(name)) {
                    throw new IllegalStateException(String.format("Have two tools with name '%s'", name));
                }
                testTools.put(name, testTool);
                LOG.info("Registered test tool {} ({})", name, testToolClass.getCanonicalName());
            } catch (Exception e) {
                LOG.error("Unable to instantiate class " + testToolClass.getCanonicalName(), e);
            }
        }
        this.testTools = Collections.unmodifiableMap(testTools);
    }

    public TestTools(TestTool testTool) {
        this.testTools = new HashMap<>();
        this.testTools.put(testTool.getName(), testTool);
    }

    public TestTool findByName(String name) {
        TestTool testTool = testTools.get(name);
        if (testTool == null) {
            throw new IllegalArgumentException(String.format("No test tool could be found with name '%s'", name));
        }
        return testTool;
    }

    public Collection<TestTool> getTestTools() {
        return testTools.values();
    }
}
