package com.xebialabs.xltest.resources;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.repository.ItemAlreadyExistsException;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.xltest.domain.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.*;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import static com.xebialabs.deployit.plugin.api.reflect.Type.valueOf;

@Controller
@Path("/data")
@Produces(MediaType.APPLICATION_JSON)
@Consumes(MediaType.APPLICATION_JSON)
public class DataController {
	
	private static final Logger LOG = LoggerFactory.getLogger(DataController.class.getName());

    private RepositoryService repository;

    @Autowired
    public DataController(RepositoryService repository) {
        this.repository = repository;
    }

    @GET
    @Path("/")
    public Collection<BaseConfigurationItem> getAll(@QueryParam("type") String type)  {
        SearchParameters query = new SearchParameters();
        if (type != null) {
            query.setType(valueOf(type));
        }
        return repository.listEntities(query);
    }

    @GET
    @Path("/{id:.+}")
    public BaseConfigurationItem get(@PathParam("id") String id)  {
    	LOG.info("reading item with id: " + id);
        return repository.read(id);
    }

    @PUT
    @Path("/{id:.+}")
    public Response update(@PathParam("id") String id, BaseConfigurationItem item)  {
    	LOG.info("update item with id: " + item.getId());
        // Should rename item on id 'id' to item on id 'item.id'?
    	if (id.equals(item.getId())) {
    		for (PropertyDescriptor propertyDescriptor : item.getType().getDescriptor().getPropertyDescriptors()) {
    			String name = propertyDescriptor.getName();
    			Object propertyValue = propertyDescriptor.get(item);
    			LOG.info("set property " + name + " to value: " + propertyValue); 
    		}
            repository.update(item);
            return Response.status(201).entity(item).build();
        } else {
            return Response.status(409).entity(item).build();
        }
    }

    @POST
    @Path("/")
    public Response create(BaseConfigurationItem item) {
        addPrefixIfNeeded(item);
        LOG.info("create item with id: " + item.getId());
        List<ConfigurationItem> toBeCreatedItems = new ArrayList<>();
        toBeCreatedItems.add(item);
        searchForNestedToBeCreatedCis(item, toBeCreatedItems);
        try {
            repository.create(toBeCreatedItems.toArray(new ConfigurationItem[toBeCreatedItems.size()]));
            return Response.status(201).entity(item).build();
        } catch (ItemAlreadyExistsException e) {
            return Response.status(409).entity(item).build();
        }
    }

    private void searchForNestedToBeCreatedCis(ConfigurationItem item, List<ConfigurationItem> toBeCreatedCis) {
        for (PropertyDescriptor property : item.getType().getDescriptor().getPropertyDescriptors()) {
            if (property.getKind() == PropertyKind.CI) {
                ConfigurationItem child = item.getProperty(property.getName());
                if (child != null && !repository.exists(child.getId())) {
                    toBeCreatedCis.add(0, child);
                    searchForNestedToBeCreatedCis(child, toBeCreatedCis);
                }
            }
        }
    }

    @DELETE
    @Path("/{id:.+}")
    @Consumes(MediaType.WILDCARD)
    public Response delete(@PathParam("id") String id) {
        LOG.info("delete item with id: " + id);
        repository.delete(id);
        return Response.ok().build();
    }

	public void addPrefixIfNeeded(BaseConfigurationItem item) {
		String id = item.getId();
		if (id.startsWith("Configuration") || id.startsWith("Applications") || id.startsWith("Infrastructure")) {
			return;
		}
		String rootNodeName = item.getType().getDescriptor().getRoot().getRootNodeName();
		if (rootNodeName == null) {
			throw new RuntimeException("Type " + item.getType() + " is unrooted!");
		}
		String subdir = "";
		if (item.getType().instanceOf(Type.valueOf(Report.class))) {
			subdir = "Reports";
		}
		if (item.getType().instanceOf(Type.valueOf(TestSpecification.class))) {
			subdir = "TestSpecifications";
		}
        if (!"".equals(subdir)) {
            subdir = subdir + "/";
        }
		item.setId(rootNodeName + "/" + subdir + id);
	}
}
