package com.xebialabs.xltest.resources;

import com.google.common.collect.Lists;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.xltest.domain.*;
import com.xebialabs.xltest.service.EventRepository;
import com.xebialabs.xltest.view.ExecutionParameterRequest;
import com.xebialabs.xltest.view.ExecutionParameters;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.AutowireCapableBeanFactory;
import org.springframework.stereotype.Controller;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.Response;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ExecutorService;


@Controller
@Path("/execute")
@Produces(MediaType.APPLICATION_JSON)
@Consumes(MediaType.APPLICATION_JSON)
public class ExecuteController {
	private static final Logger LOG = LoggerFactory.getLogger(ExecuteController.class);
	
    private final RepositoryService repository;
    private final ExecutorService executorService;
    private final EventRepository eventRepository;
    private final AutowireCapableBeanFactory beanFactory;

    @Autowired
    public ExecuteController(RepositoryService repository,
                             ExecutorService executorService, EventRepository eventRepository,
                             AutowireCapableBeanFactory beanFactory) {
        this.repository = repository;
        this.executorService = executorService;
        this.eventRepository = eventRepository;
        this.beanFactory = beanFactory;
    }
    
	@POST
	@Path("{testSpecificationId}")
	public Response execute(@PathParam("testSpecificationId") String testSpecificationId, ExecutionParameters executionParameters)
			throws IOException, InterruptedException {
        final String id = testSpecificationId.startsWith("Configuration/TestSpecifications/")
                ? testSpecificationId
                : "Configuration/TestSpecifications/" + testSpecificationId;

        ConfigurationItem objectRead = repository.read(id);
        if (! (objectRead instanceof Executable)) {
            return Response.status(400).entity("{\"error\":\"TestSpecification not executable\"}").build();
        }
        final Executable executableTestSpecification = (Executable) objectRead;

        ExecutionParameterRequest request = executableTestSpecification.toExecutionParameterRequest();
        if (!executionParameters.satisfies(request)) {
            return Response.status(422).entity(request).build();
        }

        if (executableTestSpecification.updateExecutionProperties(executionParameters.getCommandLine(), fetchHost(executionParameters))) {
            repository.update(objectRead);
        }

        beanFactory.autowireBean(executableTestSpecification);

		final UUID testRunId = UUID.randomUUID();
        final Map<String, String> parameters = executionParameters.getParameters();

        executorService.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    executableTestSpecification.execute(testRunId, parameters, new EventHandler() {
                        @Override
                        public void onReceive(Event event) throws Exception {
                            eventRepository.insert(event);
                        }
                    });
                } catch (Throwable t) {
                    LOG.error("Error while executing the test specification {}", id, t);
                }
            }
        });

        return Response.ok("{ \"taskId\": \"" + testRunId + "\"}").build();
    }

    private Host fetchHost(ExecutionParameters executionParameters) {
        if (executionParameters.getHostId() == null) return null;
        return (Host) repository.read(executionParameters.getHostId());
    }

}

