package com.xebialabs.xltest.resources;

import com.google.common.io.ByteStreams;
import com.google.common.io.Files;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.overthere.local.LocalFile;
import com.xebialabs.xltest.domain.*;
import com.xebialabs.xltest.repository.HostRepository;
import com.xebialabs.xltest.repository.TestRunsRepository;
import com.xebialabs.xltest.repository.TestTools;
import com.xebialabs.xltest.service.EventRepository;
import com.xebialabs.xltest.repository.TestSpecificationRepository;
import com.xebialabs.xltest.utils.TestToolCategory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.*;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.util.UUID;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

@Controller
@Path("/import")
public class ImportController {
    private static final Logger LOG = LoggerFactory.getLogger(ImportController.class);

    private final TestTools testTools;
    private final RepositoryService repository;
    private final TestSpecificationRepository testSpecifications;
    private final HostRepository hosts;
    private final EventRepository eventRepository;
    private final TestRunsRepository testRunsRepository;

    @Autowired
    public ImportController(TestTools testTools, RepositoryService repository, EventRepository eventRepository, TestRunsRepository testRunsRepository, TestSpecificationRepository testSpecificationRepository, HostRepository hosts) {
        this.testTools = testTools;
        this.repository = repository;
        this.eventRepository = eventRepository;
        this.testRunsRepository = testRunsRepository;
        this.testSpecifications = testSpecificationRepository;
        this.hosts = hosts;
    }

    @POST
    @Path("/{jobName:.+}")
    public Response importTestResultsSentByJenkins(@PathParam("jobName") String jobName, @Context HttpServletRequest request) throws Exception {

        TestSpecification testSpecification = findOrCreateTestSpecificationWithJenkinsHost(
                jobName,
                request.getParameter("tool"),
                request.getParameter("pattern"),
                request.getParameter("jenkinsHost"),
                Integer.parseInt(request.getParameter("jenkinsPort"))
        );

        ServletInputStream inputStream = request.getInputStream();
        File localWorkspaceDirectory = Files.createTempDir();
        extract(localWorkspaceDirectory, inputStream);
        inputStream.close();
        LOG.info("Extracted zip content into: " + localWorkspaceDirectory.getAbsolutePath());

        final UUID testRunId = UUID.randomUUID();

        // 1. do the import, so let testSpecification refer to the result location being the extracted zip
        TestTool testTool = testTools.findByName(testSpecification.getTestToolName());
        setDefaultQualificationIfAbsent(testSpecification, testTool.getCategory());
        final Qualifier qualifier = testSpecification.getQualification().getQualifier(testRunId, testRunsRepository);
        testSpecification.doImport(testRunId, testTool, LocalFile.valueOf(localWorkspaceDirectory),
                new EventHandler() {
                    @Override
                    public void onReceive(Event event) throws Exception {
                        qualifier.update(event);
                        eventRepository.insert(event);
                    }
                });
        eventRepository.insert(new Event(qualifier.getEvent(), Event.props(Event.TEST_SPECIFICATION, testSpecification.getName())));

        // TODO make this configurable localWorkspaceDirectory.delete();

        return Response.ok().build();
    }

    private void extract(File baseDir, InputStream is) throws IOException {
        final ZipInputStream zis = new ZipInputStream(is);
        ZipEntry entry;
        while ((entry = zis.getNextEntry()) != null) {
            if (!entry.isDirectory()) {
                final File file = new File(baseDir, entry.getName());
                LOG.debug("Copying {} into file {}", entry.getName(), file);
                Files.createParentDirs(file);
                Files.write(ByteStreams.toByteArray(zis), file);
            }
        }
    }

    private void setDefaultQualificationIfAbsent(TestSpecification testSpecification, String testToolCategory) {
        if (testSpecification.getQualification() == null) {
            String defaultQualificationId = null;
            if (testToolCategory.equals(TestToolCategory.FUNCTIONAL)) {
                defaultQualificationId = "Configuration/Reports/" + TestSpecification.DEFAULT_FUNCTIONAL_TEST_QUALIFIER;
            } else if (testToolCategory.equals(TestToolCategory.PERFORMANCE)) {
                defaultQualificationId = "Configuration/Reports/" + TestSpecification.DEFAULT_PERFORMANCE_TEST_QUALIFIER;
            }
            try {
                Qualification defaultQualification = repository.read(defaultQualificationId);
                testSpecification.setQualification(defaultQualification);
                repository.update(testSpecification);
                LOG.debug("testSpecification {} updated with default qualification {}", testSpecification.getId(), defaultQualificationId);
            } catch (Exception exception) {
                LOG.warn("Couldn't set default qualification {} on testSpecification {}", defaultQualificationId, testSpecification.getId());
            }
        }
    }

    private TestSpecification findOrCreateTestSpecificationWithJenkinsHost(String jobName, String testToolName, String searchPattern, String jenkinsHostAddress, int port) {
        TestSpecification testSpecification = testSpecifications.findFirstWithHost(
                testToolName,
                searchPattern,
                jenkinsHostAddress,
                ".",
                port,
                HostRepository.HOST_TYPE_OVERTHERE_JENKINS
        );

        if (testSpecification != null) {
            return testSpecification;
        }
        testSpecification = new ExecutableTestSpecification();
        testSpecification.setId("Configuration/TestSpecifications/" + jobName);
        testSpecification.setWorkingDirectory(".");
        testSpecification.setTestToolName(testToolName);
        testSpecification.setSearchPattern(searchPattern);

        Host host = hosts.findOrCreateJenkinsHost(jenkinsHostAddress, port, jobName);
        testSpecification.setHost(host);
        repository.create(testSpecification);
        return testSpecification;
    }

    @GET
    @Path("/{testSpecificationId:.+}")
    @Produces("text/event-stream")
    public void getTestRun(@PathParam("testSpecificationId") String testSpecificationId, @Context HttpServletResponse response) throws Exception {

        response.setContentType("text/event-stream");
        response.setCharacterEncoding("UTF-8");
        response.addHeader("Cache-Control", "no-cache");
        response.addHeader("Connection", "keep-alive");

        final PrintWriter writer = response.getWriter();

        TestSpecification testSpecification;
        try {
            testSpecification = repository.read("Configuration/TestSpecifications/" + testSpecificationId);
        } catch (com.xebialabs.deployit.exception.NotFoundException e) {
            closeEventStream(writer, e.getMessage());
            return;
        }

        final UUID testRunId = UUID.randomUUID();

        try {
            TestTool testTool = testTools.findByName(testSpecification.getTestToolName());
            setDefaultQualificationIfAbsent(testSpecification, testTool.getCategory());
            final Qualifier qualifier = testSpecification.getQualification().getQualifier(testRunId, testRunsRepository);
            testSpecification.doImport(testRunId, testTool, testSpecification.getRemotePath(),
                    new EventHandler() {
                        @Override
                        public void onReceive(Event event) throws Exception {
                            qualifier.update(event);
                            eventRepository.insert(event);
                        }
                    }, new TestSpecification.ProgressCallback() {
                        @Override
                        public void update(int percentage) {
                            sendEvent(writer, percentage);
                        }
                    });
            eventRepository.insert(qualifier.getEvent());
        } finally {
            closeEventStream(writer);
        }
    }

    private void sendEvent(PrintWriter writer, int pct) {
        if (writer != null) {
            writer.write("event: event\n");
            writer.write("data: {\"percentage\":" + pct + "}\n\n");
            writer.flush();
        }
    }

    private void closeEventStream(PrintWriter writer) {
        if (writer != null) {
            writer.write("event: close\n");
            writer.write("data: {}\n\n");
            writer.close();
        }
    }

    private void closeEventStream(PrintWriter writer, String errorMessage) {
        if (writer != null) {
            writer.write("event: close\n");
            writer.write("data: {\"error\":\"" + errorMessage + "\"}\n\n");
            writer.close();
        }
    }

}
