package com.xebialabs.xltest.resources;

import static com.xebialabs.deployit.plugin.api.reflect.Type.valueOf;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.UUID;

import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.UriInfo;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.jayway.jsonpath.spi.JsonProviderFactory;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.xltest.domain.Report;
import com.xebialabs.xltest.domain.TestRun;
import com.xebialabs.xltest.repository.NotFoundException;
import com.xebialabs.xltest.repository.TestRunsRepository;

@Controller
@Path("/reports")
@Consumes(MediaType.APPLICATION_JSON)
@Produces(MediaType.APPLICATION_JSON)
public class ReportController {
    private static final Logger LOG = LoggerFactory.getLogger(ReportController.class);

    private final RepositoryService repositoryService;
    private final TestRunsRepository testRunsRepository;

	@Autowired
	public ReportController(RepositoryService repositoryService, TestRunsRepository testRunsRepository) {
		this.repositoryService = repositoryService;
		this.testRunsRepository = testRunsRepository;
	}

	@GET
	@Path("/{testRunId:[\\w-]+}")
	public Collection<Report> getReports(@PathParam("testRunId") String testRunId, @Context final HttpServletResponse response) throws IOException {
        try {
        	TestRun testRun = testRunsRepository.getTestRun(UUID.fromString(testRunId));
        	List<Report> reports = findReportForTestSetDefinitionType();
        	if (reports != null && !reports.isEmpty()) {
        		return reports;
        	}
            response.sendError(404, "Can't find any Report associated to test runs of: " + testRun.getId());
            return null;
        } catch (NotFoundException e) {
            response.sendError(404, "Can't find test run");
            return null;
        }
	}

	private List<Report> findReportForTestSetDefinitionType() {
        SearchParameters query = new SearchParameters().setType(valueOf(Report.class));
        List<Report> reportCis = repositoryService.listEntities(query);
        return reportCis;
	}

    @GET
    @Path("/{testRunId:[\\w-]+}/Configuration/Reports/{reportName:.*}")
    public String generateReport(@PathParam("testRunId") String testRunId, @PathParam("reportName") String reportName, @Context UriInfo uriInfo) throws IOException {
        // Note that we use 'Configuration/Reports' in the pattern to prevent a clash with the method below
    	Report report = repositoryService.read("Configuration/Reports/" + reportName);
        LOG.info("Found report with value {}", report);
        Map<String, Object> resultObject = testRunsRepository.generateReport(UUID.fromString(testRunId), report, convertMultivaluedMapToMap(uriInfo.getQueryParameters()));
        return JsonProviderFactory.createProvider().toJson(resultObject);
    }
    
    @GET
    @Path("/{reportName}/testspecification/{testSpecificationName:.*}")
    public String generateReportFromTestSpecification(@PathParam("testSpecificationName") String testSpecificationName, @PathParam("reportName") String reportName, @Context UriInfo uriInfo) throws IOException {
		UUID testRunId = findLastRunOf(testSpecificationName);
		Report report = repositoryService.read("Configuration/Reports/" + reportName);
		Map<String, Object> resultObject = testRunsRepository.generateReport(testRunId, report, convertMultivaluedMapToMap(uriInfo.getPathParameters()));
        return JsonProviderFactory.createProvider().toJson(resultObject);
	}

	
	private UUID findLastRunOf(String testSpecificationName) {
		TestRun lastTestRun = null;
		for (TestRun testRun : testRunsRepository.getAllTestRuns()) {
			if (testRun.getTestSpecificationName().equals(testSpecificationName)) {
				if (lastTestRun == null) {
					lastTestRun = testRun;
				} else {
					if (testRun.getStartTime().after(lastTestRun.getStartTime())) {
						lastTestRun = testRun;
					}
				}
			}
		}
		return lastTestRun == null ? null : lastTestRun.getTestRunId();
	}


    @POST
    @Path("/{testRunId:.+}")
    public String generateReport(@PathParam("testRunId") String testRunId, Report report, @Context UriInfo uriInfo) {
        Map<String, Object> resultObject = testRunsRepository.generateReport(UUID.fromString(testRunId), report, convertMultivaluedMapToMap(uriInfo.getQueryParameters()));
        return JsonProviderFactory.createProvider().toJson(resultObject);
    }

    /**
     * Yes, we drop all but the first parameter value.
     *
     * @param multivaluedMap
     * @return
     */
    private Map<String, String> convertMultivaluedMapToMap(MultivaluedMap<String, String> multivaluedMap) {
        Map<String, String> map = new TreeMap();
        for (String key : multivaluedMap.keySet()) {
            map.put(key, multivaluedMap.getFirst(key));
        }
        return map;
    }
}

