package com.xebialabs.xltest.resources;

import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.UUID;

import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;

import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Interval;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.jayway.jsonpath.spi.JsonProviderFactory;
import com.xebialabs.xltest.domain.TestRun;
import com.xebialabs.xltest.repository.TestRunsRepository;
import com.xebialabs.xltest.view.ListOfTestRuns;

@Controller
@Path("/test")
@Produces(MediaType.APPLICATION_JSON)
@Consumes(MediaType.APPLICATION_JSON)
public class TestController {
    private static final Logger LOG = LoggerFactory.getLogger(TestController.class);

    private final TestRunsRepository testRunsRepository;

    @Autowired
	public TestController(TestRunsRepository testRunsRepository) {
        this.testRunsRepository = testRunsRepository;
    }

    /**
     * Get all test runs.
     *
     * @return
     * @throws IOException
     * @throws InterruptedException
     */
    @GET
    @Path("/")
    public ListOfTestRuns getTestRuns(@QueryParam("p") List<String> queries, @QueryParam("date") String queryDate) {
        // , @QueryParam("environment") String queryEnvironment
        LOG.info("Query test runs for fields {} on date {}", queries, queryDate);
        Map searchParameters = addQueryParameters(queries);
        Collection<TestRun> runs = testRunsRepository.getTestRuns(searchParameters);

        if (queryDate != null) {
            DateTime startTime = DateTime.parse(queryDate);
            LOG.info("Filtering test runs from {}", startTime);
            final Interval dateRange = new Interval(startTime, Duration.standardDays(1));
            return new ListOfTestRuns(Collections2.filter(runs, new Predicate<TestRun>() {
                @Override public boolean apply(TestRun testRun) {
                    return testRun.getStartTime() != null && dateRange.contains(testRun.getStartTime().getTime());
                }
            }));
        }
        return new ListOfTestRuns(runs);
    }

    Map addQueryParameters(List<String> queries) {
        Map searchParameters = new HashMap();
        for (String q: queries) {
            String[] nameValue = q.split(":", 2);
            if (nameValue.length != 2) {
                throw new IllegalArgumentException("query should have format 'field:value'");
            }
            searchParameters.put(nameValue[0].trim(), nameValue[1].trim());
        }
        return searchParameters;
    }

    @GET
    @Path("/{taskId:[\\w-]+}")
    public String getTestRun(@PathParam("taskId") String taskId) {
        Map<String, Object> resultObject = new TreeMap();
        TestRun testRun = testRunsRepository.getTestRun(UUID.fromString(taskId));
        resultObject.put("running", !testRun.isFinished());
        if (testRun.isFinished()) {
        	resultObject.put("qualification", testRun.getQualificationResult());
        }
        return JsonProviderFactory.createProvider().toJson(resultObject);
    }

}

