package com.xebialabs.xltest.resources;

import static com.xebialabs.deployit.plugin.api.reflect.Type.valueOf;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;

import org.codehaus.jettison.json.JSONArray;
import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.xltest.domain.BaseTestSpecification;
import com.xebialabs.xltest.domain.Dashboard;
import com.xebialabs.xltest.domain.Executable;
import com.xebialabs.xltest.domain.ReportHolder;
import com.xebialabs.xltest.domain.TestRun;
import com.xebialabs.xltest.domain.TestSpecification;
import com.xebialabs.xltest.domain.TestSpecificationSet;
import com.xebialabs.xltest.domain.TestTool;
import com.xebialabs.xltest.repository.NotFoundException;
import com.xebialabs.xltest.repository.TestRunsRepository;
import com.xebialabs.xltest.repository.TestTools;

@Controller
@Path("/testspecifications")
@Produces(MediaType.APPLICATION_JSON)
public class TestSpecificationController {
	private static final Logger LOG = LoggerFactory.getLogger(TestSpecificationController.class);

	private final RepositoryService repositoryService;
	private final TestTools testTools;
    private final TestRunsRepository testRunsRepository;

    @Autowired
	public TestSpecificationController(RepositoryService repositoryService, TestTools testTools, TestRunsRepository testRunsRepository) {
		this.repositoryService = repositoryService;
        this.testTools = testTools;
        this.testRunsRepository = testRunsRepository;
    }
    
    @GET
    @Path("/{testSpecificationName:.+}/dashboard")
    public BaseConfigurationItem findDashboardForTestSpecification(@PathParam("testSpecificationName") String testSpecificationName, @QueryParam("runId") String runId) {
    	
    	TestSpecification testSpecification = null;
    	try {
    		testSpecification = repositoryService.read("Configuration/TestSpecifications/" + testSpecificationName);
    	} catch (NotFoundException nfe) {
    		LOG.warn("Could not find TestSpecifications named: " + testSpecificationName + ". I will just tell the user and will not deliver any dashboard");
    	}
    	if (testSpecification == null) {    		
    		return null;
    	}
    	String testToolName = testSpecification.getProperty("testToolName");
    	TestTool testTool = testTools.findByName(testToolName);
        Dashboard dashboard = getRelevantDashboard(testTool);
        dashboard.setTestSpecification(testSpecification);

        provideReportUrlsToDashboard(testSpecification, dashboard, runId);
        return dashboard;
    }

    private void provideReportUrlsToDashboard(TestSpecification testSpecification, Dashboard dashboard, String runId) {
        if (dashboard.getReportHolders() == null) {
            return;
        }

        TestRun testRun;
        if (runId != null && !"".equals(runId)) {
        	testRun = testRunsRepository.getTestRun(runId);
        } else {
        	testRun = testRunsRepository.getMostRecentTestRun(testSpecification);
        }
        dashboard.setDate(testRun.getStartTime());
        for (ReportHolder holder : dashboard.getReportHolders()) {
            holder.setTestRunId(testRun.getId());
            if(testSpecification.hasProperty("showCaseData")) {
                holder.setShowCaseData((Boolean)testSpecification.getProperty("showCaseData"));
            }
        }
    }

	private Dashboard getRelevantDashboard(TestTool testTool) {
		// TODO there are easily more than one dashboards which the same category tag. Introduce one of them being the 'default'
		String toolCategory = testTool.getCategory();
    	SearchParameters query = new SearchParameters().setType(valueOf(Dashboard.class));
        List<Dashboard> allDashboards = repositoryService.listEntities(query);
        for (Dashboard dashboard : allDashboards) {
        	List<String> toolCategories = dashboard.getToolCategories();
        	if (toolCategories.contains(toolCategory)) {
        		return repositoryService.read(dashboard.getId());
        	}
        }
        return null;
	}
	
	@DELETE
	@Path("/{testSpecificationName:.+}")
    public void deleteTestSpecification(@PathParam("testSpecificationName") String testSpecificationName) {
        LOG.info("removing test specification:" + testSpecificationName);
		String testSpecificationId = "Configuration/TestSpecifications/" + testSpecificationName;
		SearchParameters query = new SearchParameters().setType(valueOf(BaseTestSpecification.class));
        List<BaseTestSpecification> allTestSpecifications = repositoryService.listEntities(query);
		removeIncludingReferences(testSpecificationId, allTestSpecifications);
        LOG.info("removed test specification:" + testSpecificationName);
	}
	
	@DELETE
	@Path("/{testSpecificationName:.+}/{testSpecificationParentName:.+}")
    public void deleteTestSpecificationReferenceFromParent(@PathParam("testSpecificationName") String testSpecificationName, @PathParam("testSpecificationParentName") String testSpecificationParentName) {
		String testSpecificationId = "Configuration/TestSpecifications/" + testSpecificationName;
		ConfigurationItem testSpecification = repositoryService.read(testSpecificationId);
		String testSpecificationParentId = "Configuration/TestSpecifications/" + testSpecificationParentName;
		TestSpecificationSet testSpecificationParent = repositoryService.read(testSpecificationParentId);
		
		List<BaseTestSpecification> newListOfChildren = new ArrayList<BaseTestSpecification>();
		if (testSpecification != null && testSpecificationParent != null) {
			boolean referencePresent = false;
			for (BaseTestSpecification testSpec : testSpecificationParent.getTestSpecifications()) {
				if (testSpec.getId().equals(testSpecificationId)) {
					referencePresent = true;
				} else {
					newListOfChildren.add(testSpec);
				}
			}
			if (referencePresent) {
				testSpecificationParent.setTestSpecifications(newListOfChildren);
				repositoryService.createOrUpdate(testSpecificationParent);
			}
		}
	}

	private void removeIncludingReferences(String testSpecificationId, List<BaseTestSpecification> allTestSpecifications) {
		for (BaseTestSpecification testSpec : allTestSpecifications) {
			if (testSpec instanceof TestSpecificationSet) {
				removeTestSpecificationReferenceFromSuperSetsRecursively(testSpecificationId, (TestSpecificationSet)testSpec);
			}
		}
		repositoryService.delete(testSpecificationId);
	}

	private void removeTestSpecificationReferenceFromSuperSetsRecursively(String testSpecificationId, TestSpecificationSet testSetSpec) {
		List<BaseTestSpecification> newListOfChildren = new ArrayList<BaseTestSpecification>();
		boolean referencePresent = false;
		for (BaseTestSpecification testSpec : testSetSpec.getTestSpecifications()) {
			if (testSpec.getId().equals(testSpecificationId)) {
				referencePresent = true;
			} else {
				newListOfChildren.add(testSpec);
				if (testSpec instanceof TestSpecificationSet) {
					removeTestSpecificationReferenceFromSuperSetsRecursively(testSpecificationId, (TestSpecificationSet)testSpec);
				}
			}
		}
		if (referencePresent) {
			testSetSpec.setTestSpecifications(newListOfChildren);
			repositoryService.createOrUpdate(testSetSpec);
		}
	}

	@GET
    @Path("/")
    public String treeForTestSpecification() throws JSONException {
		SearchParameters query = new SearchParameters().setType(valueOf(BaseTestSpecification.class));
        List<BaseTestSpecification> allTestSpecifications = repositoryService.listEntities(query);
        if (allTestSpecifications == null) {
        	allTestSpecifications = Collections.emptyList();
        }
		return createTestSpecificationTree(allTestSpecifications).toString();
    }

    protected JSONArray createTestSpecificationTree(List<BaseTestSpecification> allTestSpecifications) throws JSONException {
    	return createTestSpecificationTree(allTestSpecifications, null);
    }

	protected JSONArray createTestSpecificationTree(List<BaseTestSpecification> allTestSpecifications, BaseTestSpecification parent) throws JSONException {
		JSONArray jArray = new JSONArray();
		for (BaseTestSpecification testSpec : allTestSpecifications) {
			JSONObject child = new JSONObject();
			child.put("name", testSpec.getName());
			child.put("type", testSpec.getType());
			if (parent != null) {
				child.put("parent", parent.getName());
			}
			if (testSpec.hasProperty("testToolName")) {
				child.put("testToolName", testSpec.getProperty("testToolName"));
			}
			child.put("executable", (testSpec instanceof Executable) && ((Executable) testSpec).isExecutable());
			if (testSpec.hasProperty("testSpecifications")) {
				List<BaseTestSpecification> children = testSpec.getProperty("testSpecifications");
				child.put("testSpecifications", createTestSpecificationTree(children, testSpec));
			}
            jArray.put(child);
        }
		return jArray;
	}

}
