'use strict';
angular.module('Applications', [
  'ui.router',
  'xl.components',
  'xl.templates',
  'ui.bootstrap',
  'xl.online.commons'
]).config([
  '$stateProvider',
  '$urlRouterProvider',
  function ($stateProvider, $urlRouterProvider) {
    $urlRouterProvider.otherwise('/applications');
    $stateProvider.state('applications', {
      url: '/applications',
      templateUrl: 'web-src/applications/applications.html',
      controller: 'applicationsManagerCtrl'
    });
  }
]).constant('xlDeployServicesRoot', 'deployit/').run([
  '$rootScope',
  'typeSystemService',
  'dipHandlers',
  function ($rootScope, typeSystemService, dipHandlers) {
    typeSystemService.loadTypesDescriptors();
    // Used in DIP Forms
    $rootScope.dipHandlers = dipHandlers;
  }
]);
'use strict';
angular.module('Applications').controller('applicationsManagerCtrl', [
  '$scope',
  '$http',
  '$state',
  'applicationsService',
  'treeService',
  'ciTabsService',
  'constantService',
  'confirmDialogService',
  function ($scope, $http, $state, applicationsService, treeService, ciTabsService, constantService, confirmDialogService) {
    $scope.applications = [];
    $scope.currentNode = {};
    $scope.selectNode = function (node) {
      $scope.currentNode.path = node.path;
    };
    $scope.unfoldPathToNode = function (targetNodePath) {
      var nodeLabelsInPath = targetNodePath.split('/');
      if (targetNodePath[0] === '/') {
        nodeLabelsInPath.splice(0, 1);
      }
      var parentNode = {
          path: constantService.ROOT.APPLICATIONS,
          children: $scope.applications,
          classType: 'dirClass'
        };
      treeService.unfoldNode(nodeLabelsInPath, 0, parentNode, $scope.selectNode, applicationsService.findCIStructureByParent);
    };
    $scope.getRootCIs = function () {
      applicationsService.findCIStructureByParent().then(function (rootCIs) {
        $scope.applications = rootCIs;
      });
    };
    $scope.findNodeChildren = function (node) {
      return applicationsService.findCIStructureByParent(node.path, node.classType);
    };
    $scope.removeNode = function (id) {
      var nodeLabelsInPath = id.split('/');
      treeService.removeNode(nodeLabelsInPath.splice(1), $scope.applications);
    };
    $scope.treeOptions = {
      clickAction: function (node) {
        openTab(node, 'AUTO');
      },
      dblClickAction: function (node) {
        openTab(node, 'FORCE_NEW');
      }
    };
    var openTab = function (node, tabMode) {
      if (node.classType === applicationsService.applicationTypeId) {
        ciTabsService.viewApplication(node.path, tabMode);
      } else if (node.classType === applicationsService.versionTypeId) {
        ciTabsService.viewDeploymentPackage(node.path);
      } else if (node.classType === applicationsService.deployableTypeId) {
        ciTabsService.viewDeployable(node.path, tabMode);
      }
    };
    $scope.getRootCIs();
    $scope.newDeploymentPackage = ciTabsService.newDeploymentPackage;
    $scope.importCIs = ciTabsService.importCIs;
    $scope.confirmDialog = function (cond, action) {
      confirmDialogService.confirmDialog(cond, 'Are you sure you want to leave? Any changes will be lost.', action);
    };
  }
]);
'use strict';
angular.module('Applications').factory('applicationsResource', [
  '$http',
  'xlDeployServicesRoot',
  'constantService',
  'repositoryQueryBuilder',
  function ($http, xlDeployServicesRoot, constantService, repositoryQueryBuilder) {
    return {
      findAllApplications: function (namePattern) {
        var query = repositoryQueryBuilder.newQueryBuilder().withType(constantService.TYPE.UDM_APPLICATIONS).withNamePattern(namePattern).query();
        return $http.get(xlDeployServicesRoot + '/repository/query', query);
      },
      findAllApplicationsByParent: function (parentId, namePattern) {
        var query = repositoryQueryBuilder.newQueryBuilder().withType(constantService.TYPE.UDM_APPLICATIONS).withNamePattern(namePattern).withParent(parentId).query();
        return $http.get(xlDeployServicesRoot + 'repository/query', query);
      }
    };
  }
]);
'use strict';
angular.module('Applications').factory('applicationsService', [
  '$q',
  'ciResource',
  'applicationsResource',
  'directoryResource',
  'versionResource',
  'constantService',
  'utils',
  function ($q, ciResource, applicationsResource, directoryResource, versionResource, constantService, utils) {
    var applicationTypeId = 'appClass';
    var directoryTypeId = 'dirClass';
    var versionTypeId = 'versionClass';
    var deployableTypeId = 'deployableClass';
    function getCi(ciId) {
      return ciResource.get(ciId);
    }
    function getExistingDirectories(namePattern) {
      return directoryResource.findAllDirectoriesByAncestor(constantService.ROOT.APPLICATIONS, namePattern).then(utils.getCiIdsWithoutPrefix);
    }
    function findAllCis(type, namePattern) {
      return ciResource.findAllCis(type, namePattern);
    }
    function addCandidatesApplication(meta, params) {
      return applicationsResource.findAllApplications(params.term).then(function (response) {
        return _.map(response.data, function (ci) {
          ci.id = utils.removePrefix(ci.ref);
          return ci;
        });
      });
    }
    function ciExists(ciName) {
      return ciResource.ciExists(constantService.ROOT.APPLICATIONS + '/' + ciName);
    }
    function addTypeToReturnedData(responseData, classType) {
      _.each(responseData, function (it) {
        it.classType = classType;
      });
      return responseData;
    }
    function parseDeployableRefToNodeData(deployableRef) {
      return {
        ref: deployableRef,
        classType: deployableTypeId
      };
    }
    function parseDataToSingleNode(data) {
      var path = data.ref;
      var classType = data.classType;
      var labels = path.split('/');
      var id = 'id_' + path;
      var idDom = id.replace(new RegExp('/', 'g'), '-');
      idDom = idDom.replace(new RegExp(' ', 'g'), '-');
      var node = {
          id: 'id_' + path,
          path: path,
          label: _.last(labels),
          classType: classType,
          idDom: idDom
        };
      if (classType !== deployableTypeId) {
        node.children = [];
      }
      return node;
    }
    function findCIStructureByParent(parentPath, parentType) {
      if (!parentPath) {
        parentPath = constantService.ROOT.APPLICATIONS;
      }
      var promises = [];
      if (!parentType || parentType === directoryTypeId) {
        // we look for root nodes or children of a directory.
        var directoriesPromise = directoryResource.findAllDirectoriesByParent(parentPath).then(function (response) {
            return addTypeToReturnedData(response.data, directoryTypeId);
          });
        var applicationsPromise = applicationsResource.findAllApplicationsByParent(parentPath).then(function (response) {
            return addTypeToReturnedData(response.data, applicationTypeId);
          });
        promises.push(directoriesPromise);
        promises.push(applicationsPromise);
      } else if (parentType && parentType === applicationTypeId) {
        var versionsPromise = versionResource.findAllVersionsByParent(parentPath).then(function (response) {
            return addTypeToReturnedData(response.data, versionTypeId);
          });
        promises.push(versionsPromise);
      } else if (parentType && parentType === versionTypeId) {
        var deployablePromise = ciResource.get(parentPath).then(function (response) {
            return _.map(response.data.deployables, parseDeployableRefToNodeData);
          });
        promises.push(deployablePromise);
      }
      return $q.all(promises).then(function (allData) {
        var structureData = [];
        _.forEach(allData, function (data) {
          structureData = _.union(structureData, data);
        });
        structureData = _.map(structureData, parseDataToSingleNode);
        return structureData;
      });
    }
    return {
      applicationTypeId: applicationTypeId,
      directoryTypeId: directoryTypeId,
      versionTypeId: versionTypeId,
      deployableTypeId: deployableTypeId,
      getCi: getCi,
      findAllCis: findAllCis,
      ciExists: ciExists,
      getExistingDirectories: getExistingDirectories,
      addCandidatesApplication: addCandidatesApplication,
      findCIStructureByParent: findCIStructureByParent,
      addTypeToReturnedData: addTypeToReturnedData
    };
  }
]);
'use strict';
angular.module('Applications').controller('applicationsEditCtrl', [
  '$scope',
  '$http',
  'applicationsService',
  'applicationCiService',
  'ciService',
  'typeSystemService',
  'constantService',
  '$uiTabs',
  'ciTabsService',
  function ($scope, $http, applicationsService, applicationCiService, ciService, typeSystemService, constantService, $uiTabs, ciTabsService) {
    $scope.$tabData.isLoaded = false;
    $scope.applicationMetadata = typeSystemService.getTypeDescriptorOf(constantService.TYPE.UDM_APPLICATIONS);
    $scope.applicationDynamicFormModel = {};
    if (!$scope.$tabData.initialized) {
      $scope.$tabData.application = {};
      applicationsService.getCi($scope.$tabData.applicationId).then(function (response) {
        var ci = response.data;
        var data = ciService.extractCiStructure(ci.id);
        $scope.$tabData.applicationPath = data.path;
        $scope.$tabData.applicationName = data.name;
        $scope.$tabData.application = ci;
        $scope.$tabData.isLoaded = true;
      }, function (error) {
        console.error(error);
      });
    } else {
      $scope.$tabData.isLoaded = true;
    }
    $scope.getDirectoriesMatching = function (applicationPathPattern) {
      return applicationsService.getExistingDirectories(applicationPathPattern);
    };
    $scope.saveApplicationAndDirectories = function (onSaveCallback) {
      var extendedData = $scope.$tabData.application;
      var applicationCIObj = ciService.normalizeCiId(constantService.ROOT.APPLICATIONS, $scope.$tabData.applicationPath, $scope.$tabData.applicationName);
      var errorFunction = function (error) {
        $scope.$tabData.postApplicationsError = error.isApplicationError ? error.message : undefined;
        $scope.$tabData.postDirectoryError = error.isDirectoryError ? error.message : undefined;
      };
      $scope.$tabData.applicationId = applicationCIObj.id;
      applicationCiService.editCiApplication(applicationCIObj, extendedData).then(function () {
        onSaveCallback(applicationCIObj.idWithoutRoot);
      }, errorFunction);
    };
    $scope.isLoaded = function () {
      return $scope.$tabData.isLoaded;
    };
    $scope.isValid = function () {
      return !$scope.newApplicationForm.$invalid && !$scope.applicationDynamicFormModel.$invalid;
    };
    $scope.$watch(function () {
      return $scope.applicationDynamicFormModel.$pristine && $scope.newApplicationForm.$pristine;
    }, function (value) {
      if (value === false) {
        $scope.$setTabPristine();
      }
    });
    // application tabs
    $scope.viewApplication = ciTabsService.viewApplication;
    $scope.editApplication = ciTabsService.editApplication;
    $scope.switchToViewApplication = ciTabsService.switchToViewApplication;
    $scope.switchToNewVersion = ciTabsService.switchToNewVersion;
    $scope.switchToViewApplicationWithConfirm = function () {
      var currentTab = $uiTabs.getActiveTab();
      $scope.confirmDialog(currentTab.pristine, function () {
        ciTabsService.switchToViewApplication();
      });
    };
    $scope.$tabData.initialized = true;
  }
]);
'use strict';
angular.module('Applications').factory('applicationCiService', [
  '$q',
  'ciService',
  'ciResource',
  'constantService',
  'utils',
  function ($q, ciService, ciResource, constantService, utils) {
    function createJsonCis(applicationObj, extendedData) {
      var cis = ciService.createDirectoriesJsonCis(applicationObj);
      cis.push(ciService.createJsonCiWithData(applicationObj, constantService.TYPE.UDM_APPLICATIONS, extendedData));
      return cis;
    }
    function editCiApplication(applicationCiObj, extendedData) {
      var ci = ciService.createJsonCiWithData(applicationCiObj, constantService.TYPE.UDM_APPLICATIONS, extendedData);
      return ciResource.editCi(ci.id, ci);
    }
    return {
      editCiApplication: editCiApplication,
      createJsonCis: createJsonCis
    };
  }
]);
'use strict';
angular.module('Applications').directive('xlAutocomplete', function () {
  return {
    scope: { xlAutocomplete: '&' },
    require: 'ngModel',
    link: function (scope, element, attr, ctrl) {
      element.autocomplete({
        delay: 200,
        minLength: 1,
        source: function (params, addCandidates) {
          scope.xlAutocomplete(params.term).then(addCandidates);
        },
        select: function (event, ui) {
          scope.$apply(function () {
            ctrl.$setViewValue(ui.item.value);
          });
        }
      });
    }
  };
});
'use strict';
angular.module('Applications').controller('applicationsDeployableCtrl', [
  '$scope',
  '$state',
  '$stateParams',
  'applicationsService',
  'typeSystemService',
  'constantService',
  'ciService',
  'deployableService',
  '$uiTabs',
  'ciTabsService',
  function ($scope, $state, $stateParams, applicationsService, typeSystemService, constantService, ciService, deployableService, $uiTabs, ciTabsService) {
    $scope.$tabData.isLoaded = false;
    $scope.deployableTypeNames = typeSystemService.getDeployableTypeNames();
    $scope.deployableTypeFormValidation = {};
    if (!$scope.$tabData.initialized) {
      $scope.$tabData.deployableTypeFormMetadata = {};
      $scope.$tabData.deployableTypeFormData = {};
      applicationsService.getCi($scope.$tabData.deployableId).then(function (response) {
        var ci = response.data;
        var data = ciService.extractCiStructure(ci.id);
        $scope.$tabData.deployablePath = data.path;
        $scope.$tabData.deployableName = data.name;
        $scope.$tabData.deployableType = ci.type;
        delete ci.$token;
        delete ci.id;
        delete ci.type;
        $scope.$tabData.deployableTypeFormData = ci;
        $scope.$tabData.deployableTypeFormMetadata = typeSystemService.getTypeDescriptorOf($scope.$tabData.deployableType);
        $scope.$tabData.isLoaded = true;
      }, function (error) {
        console.error(error);
      });
    } else {
      $scope.$tabData.isLoaded = true;
    }
    $scope.onTypeSelect = function (typeName) {
      $scope.$tabData.deployableType = typeName;
      $scope.$tabData.deployableTypeFormData = {};
      $scope.$tabData.deployableTypeFormMetadata = typeSystemService.getTypeDescriptorOf(typeName);
    };
    $scope.saveDeployable = function (callback) {
      var deployableObj = ciService.normalizeCiId(constantService.ROOT.APPLICATIONS, $scope.$tabData.deployablePath, $scope.$tabData.deployableName);
      deployableService.editDeployable(deployableObj, $scope.$tabData.deployableType, $scope.$tabData.deployableTypeFormData).then(function () {
        $scope.$tabData.deployableId = deployableObj.id;
        if (callback) {
          callback();
        }
      });
    };
    $scope.isLoaded = function () {
      return $scope.$tabData.isLoaded;
    };
    $scope.isValid = function () {
      return !$scope.deployableTypeFormValidation.$invalid && !$scope.newDeployableForm.$invalid;
    };
    // ==== DYNAMIC FORM ====
    //$scope.$tabData.isPristine = function () {
    //    return $scope.deployableTypeFormValidation.$pristine && $scope.newDeployableForm.$pristine;
    //};
    $scope.$watch(function () {
      return $scope.deployableTypeFormValidation.$pristine && $scope.newDeployableForm.$pristine;
    }, function (value) {
      if (value === false) {
        $scope.$setTabPristine();
      }
    });
    // deployable tabs
    $scope.viewDeployable = ciTabsService.viewDeployable;
    $scope.editDeployable = ciTabsService.editDeployable;
    $scope.switchToViewDeployable = ciTabsService.switchToViewDeployable;
    $scope.switchToViewDeployableWithConfirm = function () {
      var currentTab = $uiTabs.getActiveTab();
      $scope.confirmDialog(currentTab.pristine, function () {
        ciTabsService.switchToViewDeployable();
      });
    };
    $scope.$tabData.initialized = true;
  }
]);
'use strict';
angular.module('Applications').factory('deployableService', [
  '$q',
  'ciService',
  'ciResource',
  function ($q, ciService, ciResource) {
    var editDeployable = function (deployableCIObj, type, data) {
      return ciResource.editCi(deployableCIObj.id, ciService.createJsonCiWithData(deployableCIObj, type, data));
    };
    return { editDeployable: editDeployable };
  }
]);
'use strict';
angular.module('Applications').controller('deployableModalCtrl', [
  '$scope',
  '$xlModalInstance',
  'modalData',
  'typeSystemService',
  'confirmDialogService',
  function ($scope, $xlModalInstance, modalData, typeSystemService, confirmDialogService) {
    var askConfirmation = function () {
      confirmDialogService.openConfirmDialog($scope.formDeployableCiModel.form.$pristine, 'Are you sure you want to leave? Any changes will be lost.').then(function () {
        $xlModalInstance.dismiss();
      });
    };
    $scope.isSourceArtifact = modalData.isSourceArtifact;
    $scope.displayMode = modalData.displayMode;
    $scope.deployableCiModel = modalData.model || { form: {} };
    $scope.formDeployableCiModel = { form: {} };
    $scope.typesDeployableCi = typeSystemService.getDeployableTypes();
    $xlModalInstance.handlers = {
      onEscModal: askConfirmation,
      onCloseModal: askConfirmation
    };
    $scope.cancel = function () {
      askConfirmation();
    };
    $scope.isDisabledSaveButton = function () {
      if (!$scope.isSourceArtifact || $scope.displayMode !== 'creation') {
        return !$scope.formDeployableCiModel.form.$valid;
      }
      var uploadFile = $scope.deployableCiModel.form.__upload_file__;
      var fileChosen = $scope.deployableCiModel.remoteFileUri || uploadFile && uploadFile.file;
      return !$scope.formDeployableCiModel.form.$valid || _.isUndefined(fileChosen);
    };
    $scope.create = function () {
      if ($scope.isSourceArtifact && $scope.displayMode === 'creation') {
        $scope.deployableCiModel.form.fileUri = $scope.deployableCiModel.remoteFileUri;
      }
      $xlModalInstance.close($scope.deployableCiModel);
    };
  }
]);
'use strict';
angular.module('Applications').controller('deploymentPackageCtrl', [
  '$scope',
  'errorService',
  'deploymentPackageService',
  'ciService',
  'constantService',
  'typeSystemService',
  'deployableService',
  'utils',
  'applicationsService',
  '$uiTabs',
  function ($scope, errorService, deploymentPackageService, ciService, constantService, typeSystemService, deployableService, utils, applicationsService, $uiTabs) {
    $scope.versionTypeFormValidation = { form: {} };
    $scope.versionPropertiesFormData = {};
    // Pre-load data
    var tab = $uiTabs.getTab($scope.$$tabId);
    if (tab) {
      $scope.mode = tab.mode;
      if (tab.versionId) {
        deploymentPackageService.getDeploymentPackage(tab.versionId).then(function (deploymentPackage) {
          $scope.existingApplicationName = deploymentPackage.existingApplicationName;
          $scope.applicationVersion = deploymentPackage.applicationVersion;
          $scope.idVersionWithoutRoot = utils.removePrefix(tab.versionId);
          $scope.versionPropertiesFormData = deploymentPackage.versionPropertiesFormData;
          $scope.deployableArtifacts = deploymentPackage.deployableArtifacts;
          $scope.deployableResources = deploymentPackage.deployableResources;
          $scope.showDeploymentDetails = true;
        });
      }
    }
    $scope.existingApplicationNameHandlers = { addCandidates: deploymentPackageService.addCandidatesApplication };
    $scope.$watch('applicationVersion', function () {
      $scope.postError = undefined;
    });
    $scope.createAndConfigure = function () {
      $scope.postError = undefined;
      // Prepare Json
      var newPathAndAppName = utils.addLeadingSlashIfNotExists($scope.existingApplicationName.id);
      var appObj = ciService.extractCiStructure(constantService.ROOT.APPLICATIONS + newPathAndAppName);
      var cis = [];
      if (!$scope.existingApplicationName.type) {
        cis = ciService.createDirectoriesJsonCis(appObj);
        var appCi = ciService.createJsonCi(appObj);
        appCi.type = constantService.TYPE.UDM_APPLICATIONS;
        cis.push(appCi);
      }
      var versionObj = ciService.normalizeCiId(constantService.ROOT.APPLICATIONS, appObj.idWithoutRoot, $scope.applicationVersion);
      cis.push(ciService.createJsonCi(versionObj, constantService.TYPE.UDM_DEPLOYMENT_PACKAGE));
      // SaveJson
      deploymentPackageService.saveCisApplicationAndVersion(cis).then(function () {
        $scope.unfoldPathToNode(versionObj.idWithoutRoot);
        $scope.idVersionWithoutRoot = versionObj.idWithoutRoot;
        $scope.showDeploymentDetails = true;
      }, function (error) {
        $scope.postError = error.data && !error.xlPreventDefault ? error.data : undefined;
      });
    };
    // Deployable Artifacts
    $scope.addDeployableArtifact = function () {
      deploymentPackageService.openDeployableModal($scope.idVersionWithoutRoot, true, constantService.MODES.CREATION).then(function (artifact) {
        if (artifact.form.fileUri) {
          artifact.progress = 100;
          artifact.stripped = true;
        } else {
          artifact.onCancelUpload = function () {
            artifact.cancelUpload = true;
          };
        }
        artifact.progressStatus = 'inProgress';
        artifact.uploaded = false;
        $scope.deployableArtifacts.push(artifact);
        deploymentPackageService.saveDeployableCi($scope.idVersionWithoutRoot, artifact).then(function () {
          if (artifact.progressStatus === 'success') {
            artifact.uploaded = true;
            artifact.progress = undefined;
            $scope.unfoldPathToNode(artifact.ciId.idWithoutRoot);
          } else if (artifact.progressStatus === 'error') {
            _.remove($scope.deployableArtifacts, function (item) {
              return item.ciId.id === artifact.ciId.id;
            });
          }
        });
      });
    };
    $scope.editDeployableArtifact = function (id) {
      var model = _.find($scope.deployableArtifacts, { ciId: { id: id } });
      deploymentPackageService.openDeployableModal($scope.idVersionWithoutRoot, true, constantService.MODES.EDITION, model).then(function (artifact) {
        delete artifact.form.__upload_file__;
        deployableService.editDeployable(artifact.ciId, artifact.type, artifact.form).then(function () {
          var idWithoutRoot = artifact.ciId.id.substring(artifact.ciId.id.indexOf('/'));
          $scope.unfoldPathToNode(idWithoutRoot);
        });
      });
    };
    $scope.deleteDeployableArtifact = function (ciId) {
      deploymentPackageService.deleteDeployable(ciId).then(function () {
        $scope.unfoldPathToNode($scope.idVersionWithoutRoot);
        $scope.removeNode(ciId);
        _.remove($scope.deployableArtifacts, { ciId: { id: ciId } });
      });
    };
    // Resources
    $scope.addDeployableResource = function () {
      deploymentPackageService.openDeployableModal($scope.idVersionWithoutRoot, false, constantService.MODES.CREATION).then(function (resource) {
        $scope.deployableResources.push(resource);
        deploymentPackageService.saveDeployableCi($scope.idVersionWithoutRoot, resource).then(function () {
          $scope.unfoldPathToNode(resource.ciId.idWithoutRoot);
        });
      });
    };
    $scope.editDeployableResource = function (id) {
      var model = _.find($scope.deployableResources, { ciId: { id: id } });
      deploymentPackageService.openDeployableModal($scope.idVersionWithoutRoot, false, constantService.MODES.EDITION, angular.copy(model)).then(function (resource) {
        deployableService.editDeployable(resource.ciId, resource.type, resource.form).then(function () {
          var idWithoutRoot = resource.ciId.id.substring(resource.ciId.id.indexOf('/'));
          $scope.unfoldPathToNode(idWithoutRoot);
          angular.copy(resource, model);
        });
      });
    };
    $scope.deleteDeployableResource = function (ciId) {
      deploymentPackageService.deleteDeployable(ciId).then(function () {
        $scope.unfoldPathToNode($scope.idVersionWithoutRoot);
        $scope.removeNode(ciId);
        _.remove($scope.deployableResources, { ciId: { id: ciId } });
      });
    };
    $scope.deployableArtifacts = [];
    $scope.deployableResources = [];
    $scope.versionTypeFormMetadata = typeSystemService.getTypeDescriptorOf(constantService.TYPE.UDM_DEPLOYMENT_PACKAGE);
    $scope.versionTypeFormMetadata.properties = _.filter($scope.versionTypeFormMetadata.properties, function (it) {
      return it.asContainment === false;
    });
    $scope.saveVersionProperties = function () {
      var newPathAndAppName = utils.addLeadingSlashIfNotExists($scope.existingApplicationName.id);
      var appObj = ciService.extractCiStructure(constantService.ROOT.APPLICATIONS + newPathAndAppName);
      var versionObj = ciService.normalizeCiId(constantService.ROOT.APPLICATIONS, appObj.idWithoutRoot, $scope.applicationVersion);
      deploymentPackageService.editCiVersion(versionObj, constantService.TYPE.UDM_DEPLOYMENT_PACKAGE, $scope.versionPropertiesFormData).then(function () {
        var currentTab = $uiTabs.getActiveTab();
        currentTab.pristine = true;
        // TODO why DPO ?
        switchToViewDeploymentPackage();
      });
    };
    $scope.editDeployablePackage = function () {
      $scope.cloneVersionPropertiesFormData = angular.copy($scope.versionPropertiesFormData);
      $scope.mode = tab.mode = constantService.MODES.EDITION;
    };
    $scope.cancelVersionProperties = function () {
      $scope.versionPropertiesFormData = $scope.cloneVersionPropertiesFormData;
      $scope.mode = tab.mode = constantService.MODES.VIEW;
    };
    $scope.isUploading = function () {
      return !!_.find($scope.deployableArtifacts, function (deployableArtifact) {
        return _.isNumber(deployableArtifact.progress);
      });
    };
    $scope.isCreateAndConfigurationDisabled = function () {
      return $scope.showDeploymentDetails || _.isEmpty($scope.existingApplicationName) || _.isEmpty($scope.applicationVersion);
    };
    $scope.$watch(function () {
      return $scope.newDeploymentPackageForm.$pristine;
    }, function (value) {
      if (value === false) {
        $scope.$setTabPristine();
      }
    });
    function switchToViewDeploymentPackage() {
      $scope.mode = tab.mode = constantService.MODES.VIEW;
    }
  }
]);
'use strict';
angular.module('Applications').filter('percentage', [
  '$window',
  function ($window) {
    return function (input) {
      if ($window.isNaN(input)) {
        return '';
      }
      return Math.round(input * 100) + '%';
    };
  }
]);
angular.module('Applications').filter('humanFileSize', [
  '$window',
  function ($window) {
    var sizes = [
        'KB',
        'MB',
        'GB',
        'TB'
      ];
    return function (input) {
      if ($window.isNaN(input)) {
        return '';
      }
      if (input < 1024) {
        return Number(input) + ' Bytes';
      }
      var u = -1;
      do {
        input = input / 1024;
        ++u;
      } while (input >= 1024);
      return input.toFixed() + ' ' + sizes[u];
    };
  }
]);
'use strict';
angular.module('Applications').factory('deploymentPackageService', [
  '$q',
  'ciService',
  'applicationsResource',
  'ciResource',
  'constantService',
  'utils',
  '$xlModal',
  'confirmDialogService',
  'typeSystemService',
  function ($q, ciService, applicationsResource, ciResource, constantService, utils, $xlModal, confirmDialogService, typeSystemService) {
    function getDeploymentPackage(versionId) {
      var deploymentPackage = {};
      return ciResource.get(versionId).then(function (response) {
        var ci = response.data;
        var data = ciService.extractCiStructure(ci.id);
        deploymentPackage.existingApplicationName = { id: data.path };
        deploymentPackage.applicationVersion = data.name;
        deploymentPackage.versionPropertiesFormData = ci;
        // Load deployable
        deploymentPackage.deployableArtifacts = [];
        deploymentPackage.deployableResources = [];
        if (response.data.deployables && !_.isEmpty(response.data.deployables)) {
          return ciResource.findAllCisByAncestor(versionId).then(function (response) {
            var deployables = response.data;
            angular.forEach(deployables, function (deployable) {
              var deployableData = ciService.extractCiStructure(deployable.ref);
              var item = {
                  lazy: true,
                  name: deployableData.name,
                  ciId: { id: deployableData.id }
                };
              if (typeSystemService.isSourceArtifactType(deployable.type)) {
                item.uploaded = true;
                deploymentPackage.deployableArtifacts.push(item);
              } else {
                deploymentPackage.deployableResources.push(item);
              }
            });
            return deploymentPackage;
          });
        } else {
          return deploymentPackage;
        }
      });
    }
    function addCandidatesApplication(meta, params) {
      return applicationsResource.findAllApplications(params.term).then(function (response) {
        return _.map(response.data, function (ci) {
          ci.id = utils.removePrefix(ci.ref);
          return ci;
        });
      });
    }
    function saveCisApplicationAndVersion(cis) {
      var promises = [];
      angular.forEach(cis, function (ci) {
        var deferred = $q.defer();
        ciResource.ciExists(ci.id).then(function (response) {
          var exists = response.data.boolean;
          if (!exists || exists && ci.type === constantService.TYPE.UDM_DEPLOYMENT_PACKAGE) {
            return deferred.resolve(ci);
          }
          return deferred.resolve();
        });
        promises.push(deferred.promise);
      });
      return $q.all(promises).then(function (cis) {
        return ciResource.saveAllCis(_.without(cis, undefined));
      });
    }
    /********************************* Deployable methods *********************************/
    var deleteDeployable = function (ciId) {
      return confirmDialogService.openConfirmDialog(false, 'Are you sure you want to leave? Any changes will be lost.').then(function () {
        return ciResource.deleteCi(ciId);
      });
    };
    var openDeployableModal = function (path, isSourceArtifact, displayMode, model) {
      function openModal() {
        var modalOptions = {
            templateUrl: 'web-src/applications/tabs/deployment-package/deployable-modal/deployable-modal.html',
            windowClass: 'xl-deployable-modal',
            controller: 'deployableModalCtrl',
            resolve: {
              modalData: function () {
                return {
                  isSourceArtifact: isSourceArtifact,
                  displayMode: displayMode,
                  model: model
                };
              }
            }
          };
        return $xlModal.open(modalOptions).result;
      }
      if (model && !!model.lazy) {
        return ciResource.get(model.ciId.id).then(function (response) {
          var ci = response.data;
          model.type = ci.type;
          model.form = ci;
          delete model.form.$token;
          delete model.form.id;
          delete model.form.type;
          delete model.lazy;
          return openModal();
        });
      } else {
        return openModal();
      }
    };
    var saveDeployableCi = function (path, deployable) {
      function update(deployable) {
        var ci = ciService.createJsonCiWithData(deployable.ciId, deployable.type, deployable.form);
        return ciResource.editCi(deployable.ciId.id, ci);
      }
      function save(deployable) {
        var uploadFile = deployable.form.__upload_file__;
        var clonedForm = angular.copy(deployable.form);
        delete clonedForm.__upload_file__;
        var ci = ciService.createJsonCiWithData(deployable.ciId, deployable.type, clonedForm);
        if (uploadFile && uploadFile.file) {
          return ciResource.saveCiMultipart(ci, uploadFile.file);
        } else {
          return ciResource.saveCi(ci);
        }
      }
      var isCreation = deployable.ciId === undefined;
      deployable.ciId = ciService.normalizeCiId(constantService.ROOT.APPLICATIONS, path, deployable.name);
      return (isCreation ? save(deployable) : update(deployable)).then(function () {
        if (deployable.progressStatus) {
          deployable.progressStatus = 'success';
        }
        return deployable;
      }, function (reason) {
        if (deployable.progressStatus) {
          console.log('Error upload', reason);
          deployable.progressStatus = 'error';
        }
        return deployable;
      }, function (evt) {
        if (evt && evt.status === 'progress') {
          deployable.progress = evt.loaded;
          if (deployable.fileSize !== evt.total) {
            deployable.fileSize = evt.total;
          }
          if (deployable.cancelUpload) {
            evt.abort();
          }
        }
      });
    };
    function editCiVersion(versionCiObj, type, data) {
      return ciResource.editCi(versionCiObj.id, ciService.createJsonCiWithData(versionCiObj, type, data));
    }
    return {
      getDeploymentPackage: getDeploymentPackage,
      addCandidatesApplication: addCandidatesApplication,
      saveCisApplicationAndVersion: saveCisApplicationAndVersion,
      deleteDeployable: deleteDeployable,
      openDeployableModal: openDeployableModal,
      editCiVersion: editCiVersion,
      saveDeployableCi: saveDeployableCi
    };
  }
]);
'use strict';
angular.module('Applications').controller('importCtrl', [
  '$scope',
  '$q',
  '$timeout',
  '$http',
  'xlDeployServicesRoot',
  'importResource',
  function ($scope, $q, $timeout, $http, xlDeployServicesRoot, importResource) {
    // Accordion
    $scope.importObject = {
      fromUrl: {
        opened: true,
        state: undefined,
        message: '',
        model: undefined
      },
      fromDisk: {
        opened: false,
        state: undefined,
        message: '',
        model: undefined
      },
      fromServer: {
        opened: false,
        state: undefined,
        message: '',
        model: undefined
      }
    };
    $scope.openAccordion = function (tabName) {
      _.forEach($scope.importObject, function (value, key) {
        if (key !== tabName && value.state !== 'importing') {
          value.opened = false;
        }
      });
    };
    // Import From URL -------------------------------------------------------
    $scope.importUrl = function () {
      $scope.importObject.fromUrl.state = 'importing';
      $scope.importObject.fromUrl.message = 'Importing...';
      // TODO: To be deleted
      var testDelay = $scope.importObject.fromUrl.model === 'test' ? 5000 : 0;
      $timeout(function () {
        importResource.importFormURL($scope.importObject.fromUrl.model).then(function (response) {
          if (!!response && !!response.data && !!response.data.id) {
            var packageId = response.data.id.replace('Applications/', '');
            $scope.unfoldPathToNode(packageId);
            $scope.importObject.fromUrl.model = '';
            $scope.importObject.fromUrl.state = 'success';
            $scope.importObject.fromUrl.message = 'Imported ' + packageId;
          } else {
            $scope.importObject.fromUrl.state = 'error';
            $scope.importObject.fromUrl.message = 'Error: empty response';
          }
        }, function (error) {
          if (error.xlPreventDefault) {
            $scope.importObject.fromUrl.state = undefined;
          } else {
            $scope.importObject.fromUrl.state = 'error';
            $scope.importObject.fromUrl.message = error.data ? 'Error: ' + error.data : 'Unknown error';
          }
        });
      }, testDelay);
    };
    $scope.$watch('importObject.fromUrl.model', function (value) {
      if (value === undefined) {
        return;
      }
      if ($scope.importObject.fromUrl.stateOld === 'success' || $scope.importObject.fromUrl.state === 'error') {
        $scope.importObject.fromUrl.state = undefined;
        $scope.importObject.fromUrl.message = '';
      }
      $scope.importObject.fromUrl.stateOld = $scope.importObject.fromUrl.state;
      $scope.$setTabPristine();
    });
    // Import From Disk -----------------------------------------------------
    var onStartUpload = function (file) {
      $scope.importObject.fromDisk.state = 'importing';
    };
    var onSuccessFileUpload = function (data, file) {
      $scope.importObject.fromDisk.state = 'success';
      var packageId = data.result.id.replace('Applications/', '');
      $scope.unfoldPathToNode(packageId);
      return 'Imported ' + packageId;
    };
    var onErrorFileUpload = function (data, file) {
      $scope.importObject.fromDisk.state = 'error';
      if (data && data.jqXHR && data.jqXHR.responseText && data.jqXHR.responseText !== '') {
        return 'Error: ' + data.jqXHR.responseText;
      }
    };
    $scope.fileUploadOptions = {
      uploadUrl: xlDeployServicesRoot + 'package/upload/file.dar',
      fileNameField: 'fileData',
      authorization: $http.defaults.headers.common.Authorization,
      onStartUpload: onStartUpload,
      onSuccess: onSuccessFileUpload,
      onError: onErrorFileUpload
    };
    // Import From Server ----------------------------------------------------
    var serverPackages = [];
    importResource.importablePackage().then(function (response) {
      serverPackages = response.data;
    });
    $scope.autocompleteImportHandlers = {
      addCandidates: function () {
        var differed = $q.defer();
        differed.resolve(serverPackages);
        return differed.promise;
      }
    };
    $scope.importPackageFromServer = function () {
      $scope.importObject.fromServer.state = 'importing';
      $scope.importObject.fromServer.message = 'Importing...';
      importResource.importPackageFromServer($scope.importObject.fromServer.model).then(function (response) {
        var packageId = response.data.id.replace('Applications/', '');
        $scope.unfoldPathToNode(packageId);
        $scope.importObject.fromServer.state = 'success';
        $scope.importObject.fromServer.message = 'Imported ' + packageId;
      }, function (error) {
        $scope.importObject.fromServer.state = 'error';
        $scope.importObject.fromServer.message = error.data ? 'Error: ' + error.data : undefined;
      });
    };
    $scope.$watch('importObject.fromServer.model', function (value) {
      $scope.importObject.fromServer.state = undefined;
      $scope.importObject.fromServer.message = '';
      if (value === undefined || value === '') {
        return;
      }
      $scope.$setTabPristine();
    });
  }
]);
'use strict';
angular.module('Applications').factory('importResource', [
  '$http',
  'xlDeployServicesRoot',
  function ($http, xlDeployServicesRoot) {
    function importFormURL(url) {
      return $http.post(xlDeployServicesRoot + 'package/fetch', url);
    }
    function importPackageFromServer(id) {
      return $http.post(xlDeployServicesRoot + 'package/import/' + id);
    }
    function importablePackage() {
      return $http.get(xlDeployServicesRoot + 'package/import');
    }
    return {
      importFormURL: importFormURL,
      importPackageFromServer: importPackageFromServer,
      importablePackage: importablePackage
    };
  }
]);
'use strict';
angular.module('Applications').config([
  '$uiTabsProvider',
  function ($uiTabsProvider) {
    $uiTabsProvider.tab('applications.app.view', {
      title: 'Visual application',
      templateUrl: 'web-src/applications/tabs/applications/application-ci.html',
      controller: 'applicationsEditCtrl'
    }).tab('applications.app.edit', {
      title: 'Edit application',
      templateUrl: 'web-src/applications/tabs/applications/application-ci.html',
      controller: 'applicationsEditCtrl'
    }).tab('applications.deployable.view', {
      title: 'Visual deployable',
      templateUrl: 'web-src/applications/tabs/deployable/deployable.html',
      controller: 'applicationsDeployableCtrl'
    }).tab('applications.deployable.edit', {
      title: 'Edit deployable',
      templateUrl: 'web-src/applications/tabs/deployable/deployable.html',
      controller: 'applicationsDeployableCtrl'
    }).tab('applications.import', {
      title: 'Import package',
      templateUrl: 'web-src/applications/tabs/import/import.html',
      controller: 'importCtrl',
      volatile: false,
      onClose: [
        'confirmDialogService',
        function (confirmDialogService) {
          return function (tab) {
            return confirmDialogService.openConfirmDialog(tab.pristine, 'Are you sure you want to leave? Imports in progress will be cancelled.');
          };
        }
      ]
    }).tab('applications.deployment.package', {
      title: 'New Deployment Package',
      templateUrl: 'web-src/applications/tabs/deployment-package/deployment-package.html',
      controller: 'deploymentPackageCtrl',
      volatile: false
    }).welcome({ templateUrl: 'web-src/applications/tabs/welcome/welcome.html' }).onClose([
      'confirmDialogService',
      function (confirmDialogService) {
        return function (tab) {
          return confirmDialogService.openConfirmDialog(tab.pristine, 'Are you sure you want to leave? Your changes will be lost.');
        };
      }
    ]);
  }
]);
'use strict';
angular.module('Applications').factory('ciTabsService', [
  'utils',
  '$uiTabs',
  'constantService',
  function (utils, $uiTabs, constantService) {
    // application tabs
    var viewApplication = function (applicationId, forceNewTab) {
      openOrReplaceTab('applications.app.view', {
        title: utils.removePrefix(applicationId),
        $tabData: {
          applicationId: applicationId,
          mode: constantService.MODES.VIEW
        }
      }, forceNewTab);
    };
    var editApplication = function () {
      var currentTab = $uiTabs.getActiveTab();
      var $tabData = angular.copy(currentTab.$tabData);
      $tabData.mode = constantService.MODES.EDITION;
      openOrReplaceTab('applications.app.edit', {
        title: currentTab.title,
        $tabData: $tabData
      }, 'FORCE_SAME');
    };
    var switchToViewApplication = function () {
      var currentTab = $uiTabs.getActiveTab();
      openOrReplaceTab('applications.app.view', {
        title: utils.removePrefix(currentTab.$tabData.applicationId),
        $tabData: {
          applicationId: currentTab.$tabData.applicationId,
          mode: constantService.MODES.VIEW
        }
      }, 'FORCE_SAME');
    };
    // deployable tabs
    var viewDeployable = function (deployableId, forceNewTab) {
      openOrReplaceTab('applications.deployable.view', {
        title: utils.removePrefix(deployableId),
        $tabData: {
          deployableId: deployableId,
          mode: constantService.MODES.VIEW
        }
      }, forceNewTab);
    };
    var editDeployable = function () {
      var currentTab = $uiTabs.getActiveTab();
      var $tabData = angular.copy(currentTab.$tabData);
      $tabData.mode = constantService.MODES.EDITION;
      openOrReplaceTab('applications.deployable.edit', {
        title: currentTab.title,
        $tabData: $tabData
      }, 'FORCE_SAME');
    };
    var switchToViewDeployable = function () {
      var currentTab = $uiTabs.getActiveTab();
      var $tabData = angular.copy(currentTab.$tabData);
      $tabData.mode = constantService.MODES.VIEW;
      openOrReplaceTab('applications.deployable.view', {
        title: utils.removePrefix($tabData.deployableId),
        $tabData: $tabData
      }, 'FORCE_SAME');
    };
    // version deployment package
    var newDeploymentPackage = function () {
      openOrReplaceTab('applications.deployment.package', { mode: constantService.MODES.CREATION }, 'FORCE_NEW');
    };
    // view version deployment package
    var viewDeploymentPackage = function (versionId) {
      openOrReplaceTab('applications.deployment.package', {
        title: utils.removePrefix(versionId),
        versionId: versionId,
        mode: constantService.MODES.VIEW
      }, 'FORCE_NEW');
    };
    // import tabs
    var importCIs = function () {
      $uiTabs.addTab('applications.import', {});
    };
    /**
     * Opens or replace a CI tab
     * @param targetState target tab type
     * @param options tab options
     * @param tabMode
     *  FORCE_NEW - always opens new tab,
     *  FORCE_SAME - switches the same tab if possible,
     *  AUTO - if current tab is in visual mode, switches the current tab, if not - opens new
     */
    var openOrReplaceTab = function (targetState, options, tabMode) {
      var currentTab = $uiTabs.getActiveTab();
      if (!currentTab || !!currentTab.isWelcome || tabMode === 'FORCE_NEW') {
        $uiTabs.addTab(targetState, options);
      } else if (currentTab && (currentTab.$tabData.mode === 'visual' || tabMode === 'FORCE_SAME')) {
        $uiTabs.addTab(targetState, options, currentTab.$$id);
      }
    };
    /**
     * Public API
     */
    return {
      viewApplication: viewApplication,
      editApplication: editApplication,
      switchToViewApplication: switchToViewApplication,
      viewDeployable: viewDeployable,
      editDeployable: editDeployable,
      switchToViewDeployable: switchToViewDeployable,
      newDeploymentPackage: newDeploymentPackage,
      viewDeploymentPackage: viewDeploymentPackage,
      importCIs: importCIs
    };
  }
]);