'use strict';
angular.module('Environments', [
  'ui.router',
  'xl.components',
  'xl.templates',
  'xl.online.commons'
]).config([
  '$stateProvider',
  '$urlRouterProvider',
  function ($stateProvider, $urlRouterProvider) {
    $urlRouterProvider.otherwise('/environments');
    $stateProvider.state('environments', {
      url: '/environments',
      templateUrl: 'web-src/environments/environments.html',
      controller: 'environmentsManagerCtrl'
    });
  }
]).constant('xlDeployServicesRoot', 'deployit/').run([
  '$rootScope',
  'typeSystemService',
  'dipHandlers',
  function ($rootScope, typeSystemService, dipHandlers) {
    typeSystemService.loadTypesDescriptors();
    // Used in DIP Forms
    $rootScope.dipHandlers = dipHandlers;
  }
]);
'use strict';
angular.module('Environments').controller('environmentsManagerCtrl', [
  '$scope',
  'treeService',
  'environmentsService',
  'ciTabsService',
  function ($scope, treeService, environmentsService, ciTabsService) {
    $scope.environments = [];
    $scope.currentNode = {};
    $scope.selectNode = function (node) {
      $scope.currentNode.path = node.path;
    };
    $scope.unfoldPathToNode = function (targetNodePath) {
      var nodeLabelsInPath = targetNodePath.split('/');
      if (targetNodePath[0] === '/') {
        nodeLabelsInPath.splice(0, 1);
      }
      var parentNode = {
          path: 'Environments',
          children: $scope.environments,
          classType: 'dirClass'
        };
      treeService.unfoldNode(nodeLabelsInPath, 0, parentNode, $scope.selectNode, environmentsService.findEnvStructureByParent);
    };
    $scope.getRootEnvs = function () {
      environmentsService.findEnvStructureByParent().then(function (rootCIs) {
        $scope.environments = rootCIs;
      });
    };
    $scope.findNodeChildren = function (node) {
      return environmentsService.findEnvStructureByParent(node.path, node.classType);
    };
    $scope.treeOptions = {
      clickAction: function (node) {
        openTab(node, 'AUTO');
      },
      dblClickAction: function (node) {
        openTab(node, 'FORCE_NEW');
      }
    };
    var openTab = function (node, tabMode) {
      if (node.classType === environmentsService.environmentypeId) {
        ciTabsService.viewEnvironment(node.path, tabMode);
      }
    };
    $scope.getRootEnvs();
    $scope.newFolder = ciTabsService.newFolder;
    $scope.newEnvironment = ciTabsService.newEnvironment;
    $scope.discoverEnvironment = ciTabsService.discoverEnvironment;
  }
]);
'use strict';
angular.module('Environments').factory('environmentsResource', [
  '$http',
  'constantService',
  'xlDeployServicesRoot',
  'repositoryQueryBuilder',
  function ($http, constantService, xlDeployServicesRoot, repositoryQueryBuilder) {
    return {
      findAllEnvironmentsByParent: function (parentId, namePattern) {
        var query = repositoryQueryBuilder.newQueryBuilder().withType(constantService.TYPE.UDM_ENVIRONMENT).withNamePattern(namePattern).withParent(parentId).query();
        return $http.get(xlDeployServicesRoot + 'repository/query', query);
      }
    };
  }
]);
'use strict';
angular.module('Environments').factory('environmentsService', [
  '$q',
  'ciResource',
  'environmentsResource',
  'constantService',
  'directoryResource',
  function ($q, ciResource, environmentsResource, constantService, directoryResource) {
    var environmentypeId = 'envClass';
    var directoryTypeId = 'dirClass';
    function addTypeToReturnedData(responseData, classType) {
      _.each(responseData, function (it) {
        it.classType = classType;
      });
      return responseData;
    }
    function parseDataToSingleNode(data) {
      var path = data.ref;
      var classType = data.classType;
      var labels = path.split('/');
      var id = 'id_' + path;
      var idDom = id.replace(new RegExp('/', 'g'), '-');
      idDom = idDom.replace(new RegExp(' ', 'g'), '-');
      var node = {
          id: 'id_' + path,
          path: path,
          label: _.last(labels),
          classType: classType,
          idDom: idDom
        };
      if (classType !== environmentypeId) {
        node.children = [];
      }
      return node;
    }
    function findEnvStructureByParent(parentPath, parentType) {
      if (!parentPath) {
        parentPath = constantService.ROOT.ENVIRONMENTS;
      }
      var promises = [];
      if (!parentType || parentType === directoryTypeId) {
        // we look for root nodes or children of a directory.
        var directoriesPromise = directoryResource.findAllDirectoriesByParent(parentPath).then(function (response) {
            return addTypeToReturnedData(response.data, directoryTypeId);
          });
        var applicationsPromise = environmentsResource.findAllEnvironmentsByParent(parentPath).then(function (response) {
            return addTypeToReturnedData(response.data, environmentypeId);
          });
        promises.push(directoriesPromise);
        promises.push(applicationsPromise);
      }
      return $q.all(promises).then(function (allData) {
        var structureData = [];
        _.forEach(allData, function (data) {
          structureData = _.union(structureData, data);
        });
        structureData = _.map(structureData, parseDataToSingleNode);
        return structureData;
      });
    }
    return {
      environmentypeId: environmentypeId,
      directoryTypeId: directoryTypeId,
      findEnvStructureByParent: findEnvStructureByParent
    };
  }
]);
'use strict';
angular.module('Environments').controller('newContainerModalCtrl', [
  '$scope',
  '$xlModalInstance',
  '$q',
  'ciResource',
  'modalData',
  'typeSystemService',
  'containerService',
  'constantService',
  'confirmDialogService',
  'ciService',
  function ($scope, $xlModalInstance, $q, ciResource, modalData, typeSystemService, containerService, constantService, confirmDialogService, ciService) {
    $scope.containerNameForm = {};
    $scope.typeDescriptor = {};
    $scope.containerTypeNames = typeSystemService.getContainerTypeNames();
    $scope.environmentId = modalData.environmentId;
    $scope.container = {
      name: undefined,
      path: ''
    };
    $scope.onTypeSelect = function (typeName) {
      $scope.typeDescriptor = typeSystemService.getTypeDescriptorOf(typeName);
      var metaParent = _.find($scope.typeDescriptor.properties, function (propertie) {
          return propertie.asContainment && propertie.kind === 'CI';
        });
      if (metaParent && modalData.parentContainer) {
        $scope.container[metaParent.name] = {
          id: modalData.parentContainer.id,
          type: modalData.parentContainer.type
        };
      }
    };
    $scope.containerTypeNames = _.map(modalData.containerTypes, 'type');
    if ($scope.containerTypeNames.length === 1) {
      $scope.container.type = $scope.containerTypeNames[0];
      $scope.onTypeSelect($scope.container.type);
    }
    $scope.create = function () {
      createContainer($scope.container).then(function () {
        $xlModalInstance.close($scope.container);
      }, function (error) {
        console.error('Error', error);
      });
    };
    // Autocomplete container path
    $scope.containerPathAutocompleteHandlers = { addCandidates: containerService.addCandidatesPathContainers };
    var createContainer = function (container) {
      var path = modalData.parentContainer ? ciService.extractCiStructure(modalData.parentContainer.id).idWithoutRoot : $scope.container.path;
      var containerCiObj = ciService.normalizeCiId(constantService.ROOT.INFRASTRUCTURE, path, $scope.container.name);
      var containerCopy = _.clone(container, true);
      containerCopy.id = containerCiObj.id;
      $scope.container.id = containerCiObj.id;
      delete containerCopy.name;
      delete containerCopy.path;
      return containerService.saveCiContainer(containerCiObj, containerCopy);
    };
    var openConfirmDialogModal = function () {
      confirmDialogService.openConfirmDialog($scope.containerNameForm.internal.$pristine, 'Are you sure you want to leave? Any changes will be lost.').then(function () {
        $xlModalInstance.dismiss();
      });
    };
    $xlModalInstance.handlers = {};
    if (modalData.handleEsc) {
      $xlModalInstance.handlers.onEscModal = function () {
        openConfirmDialogModal();
      };
    }
    if (modalData.hasClose) {
      $xlModalInstance.handlers.onCloseModal = function () {
        openConfirmDialogModal();
      };
    }
    $scope.close = function () {
      openConfirmDialogModal();
    };
  }
]);
'use strict';
angular.module('Environments').factory('containerService', [
  '$q',
  'ciResource',
  'directoryResource',
  'constantService',
  'ciService',
  'typeSystemService',
  'utils',
  function ($q, ciResource, directoryResource, constantService, ciService, typeSystemService, utils) {
    var saveCiContainer = function (containerCiObj, ciContainer) {
      var promises = [];
      var metaType = typeSystemService.getTypeDescriptorOf(ciContainer.type);
      var allCis = [];
      if (metaType && metaType.root) {
        allCis = ciService.createDirectoriesJsonCis(containerCiObj);
      }
      allCis.push(ciContainer);
      angular.forEach(allCis, function (ci) {
        var deferred = $q.defer();
        ciResource.ciExists(ci.id).then(function (response) {
          var exists = response.data.boolean;
          if (!exists) {
            // ci doesn't exist so we can create it
            return deferred.resolve(ci);
          } else {
            ciResource.get(ci.id).success(function (returnedCi) {
              if (ci.type === constantService.TYPE.CORE_DIRECTORY && returnedCi.type !== constantService.TYPE.CORE_DIRECTORY) {
                return deferred.reject({
                  isDirectoryError: true,
                  message: 'A ci named ' + utils.removePrefix(ci.id) + ' already exists'
                });
              } else if (ci.type !== constantService.TYPE.CORE_DIRECTORY) {
                return deferred.resolve(ci);
              }
              return deferred.resolve();
            });
          }
        });
        promises.push(deferred.promise);
      });
      return $q.all(promises).then(function (cis) {
        var deferred = $q.defer();
        ciResource.saveAllCis(_.without(cis, undefined)).then(deferred.resolve, function (error) {
          return deferred.reject({
            ciError: true,
            message: error.data
          });
        });
        return deferred.promise;
      });
    };
    var addCandidatesPathContainers = function (meta, params) {
      return directoryResource.findAllDirectoriesByAncestor(constantService.ROOT.INFRASTRUCTURE, params.term).then(utils.getCiIdsWithoutPrefix);
    };
    return {
      saveCiContainer: saveCiContainer,
      addCandidatesPathContainers: addCandidatesPathContainers
    };
  }
]);
'use strict';
angular.module('Environments').controller('editDictionaryModalCtrl', [
  '$scope',
  '$xlModalInstance',
  'modalData',
  'typeSystemService',
  'confirmDialogService',
  function ($scope, $xlModalInstance, modalData, typeSystemService, confirmDialogService) {
    var askConfirmation = function () {
      confirmDialogService.openConfirmDialog($scope.formEditDictionary.form.$pristine, 'Are you sure you want to leave? Any changes will be lost.').then(function () {
        $xlModalInstance.dismiss();
      });
    };
    $scope.formEditDictionary = { form: undefined };
    $scope.dictionary = modalData.dictionary;
    $xlModalInstance.handlers = {
      onEscModal: askConfirmation,
      onCloseModal: askConfirmation
    };
    $scope.isValidDictionary = function () {
      return $scope.formEditDictionary.form.$valid && !_.isEmpty($scope.dictionary.id);
    };
    $scope.cancel = function () {
      askConfirmation();
    };
    $scope.create = function () {
      $xlModalInstance.close($scope.dictionary);
    };
  }
]);
'use strict';
angular.module('Environments').controller('environmentCtrl', [
  '$scope',
  '$xlModal',
  '$q',
  'WizardHandler',
  'environmentService',
  'ciService',
  'typeSystemService',
  'utils',
  'confirmDialogService',
  'constantService',
  'ciTabsService',
  function ($scope, $xlModal, $q, WizardHandler, environmentService, ciService, typeSystemService, utils, confirmDialogService, constantService, ciTabsService) {
    var $tabData = $scope.$tabData;
    /**
     * ContainerBundle
     */
    var ContainerBundle = function (container) {
      this.type = container.type;
      this.subtypes = container.subtypes;
      this.container = undefined;
      this._isNewContainer = false;
    };
    ContainerBundle.prototype.activeButton = function () {
      var index = $tabData.containerHierarchyBundles.indexOf(this);
      if (index === 0) {
        return true;
      }
      return $tabData.containerHierarchyBundles[index - 1].isContainerSelected();
    };
    ContainerBundle.prototype.displayButton = function () {
      var index = $tabData.containerHierarchyBundles.indexOf(this);
      return !$tabData.containerHierarchyBundles[index].isContainerSelected();
    };
    ContainerBundle.prototype.getTitle = function () {
      return this.container ? {
        title: ciService.extractCiStructure(this.container.id).name,
        type: this.container.type
      } : { title: this.type };
    };
    ContainerBundle.prototype.isContainerSelected = function () {
      return !!this.container;
    };
    ContainerBundle.prototype.clearContainer = function () {
      this._isNewContainer = false;
      this.container = undefined;
    };
    ContainerBundle.prototype.isNewContainer = function () {
      return this._isNewContainer;
    };
    ContainerBundle.prototype.setNewContainer = function (value) {
      this._isNewContainer = value;
    };
    ContainerBundle.prototype.clearContainerAndChild = function () {
      var index = $tabData.containerHierarchyBundles.indexOf(this);
      for (var i = index; i < $tabData.containerHierarchyBundles.length; i++) {
        $tabData.containerHierarchyBundles[i].clearContainer();
      }
    };
    ContainerBundle.prototype.getContainer = function () {
      return this.container;
    };
    ContainerBundle.prototype.getParent = function () {
      var index = $tabData.containerHierarchyBundles.indexOf(this);
      if (index === 0 || index === -1) {
        return undefined;
      }
      var parent = $tabData.containerHierarchyBundles[index - 1].container;
      return parent ? {
        id: parent.id,
        type: parent.type
      } : undefined;
    };
    ContainerBundle.prototype.setParent = function () {
      var index = $tabData.containerHierarchyBundles.indexOf(this);
      if (index > 0) {
        var parentContainerBundle = $tabData.containerHierarchyBundles[index - 1];
        parentContainerBundle.updateFromId(this.extractParentId()).then(function () {
          parentContainerBundle.setParent();
        });
      }
    };
    ContainerBundle.prototype.extractParentId = function () {
      var objId = ciService.extractCiStructure(this.container.id);
      return objId.root + objId.path;
    };
    ContainerBundle.prototype.updateFromId = function (id) {
      var currentContainerBundle = this;
      return ciService.getCi(id).then(function (response) {
        currentContainerBundle.container = response.data;
      });
    };
    $tabData.isLoaded = false;
    if (!$tabData.initialized) {
      $scope.environmentDynamicFormModel = {};
      $scope.environmentForms = {};
      $tabData.environmentData = {
        members: [],
        dictionaries: []
      };
      $tabData.envSetContainerHelp = false;
      $tabData.containerHelper = {
        deployableCandidates: [],
        containerCandidates: []
      };
      $tabData.containerChoice = 'create';
      $tabData.containerHierarchyBundles = [];
      $tabData.dictionaryChoice = 'create';
      $tabData.newDictionary = {
        id: '',
        entries: {}
      };
    }
    if ($tabData.mode !== constantService.MODES.CREATION && !$tabData.initialized) {
      ciService.getCi($tabData.environmentId).then(function (response) {
        var ci = response.data;
        var data = ciService.extractCiStructure(ci.id);
        $tabData.environmentId = data.idWithoutRoot;
        $tabData.environmentData = ci;
        $tabData.environmentData.dictionaries = _.map(ci.dictionaries, function (dictionary) {
          return { id: dictionary };
        });
        $tabData.environmentData.triggers = _.map(ci.triggers, function (trigger) {
          return {
            id: trigger,
            type: null
          };
        });
        $tabData.environmentData.members = _.map(ci.members, function (member) {
          return {
            id: _.isString(member) ? utils.removePrefix(member) : utils.removePrefix(member.id),
            type: _.isString(member) ? null : member.type
          };
        });
        $tabData.isLoaded = true;
      }, function (error) {
        console.error(error);
      });
    } else {
      $tabData.isLoaded = true;
    }
    $scope.isLoaded = function () {
      return $tabData.isLoaded;
    };
    // Wizard Management
    $scope.verifyEnvNotExistAndGotoNextStep = function () {
      ciService.ciExists(constantService.ROOT.ENVIRONMENTS, $tabData.environmentId).then(function (response) {
        if (response.data.boolean) {
          $tabData.postEnvError = 'Environment already exists';
        } else {
          WizardHandler.wizard().next();
        }
      });
    };
    $scope.verifyContainerIsNotSelectAndGotoNextStep = function () {
      var condition = $tabData.newContainerType && $tabData.newContainerType.length > 0 && $tabData.environmentData.members && $tabData.environmentData.members.length > 0;
      confirmDialogService.openConfirmDialog(condition, 'Containers is empty, are you sure you want to go to the next step?').then(function () {
        WizardHandler.wizard().next();
      });
    };
    $scope.verifyDictonariesIsNotSelectAndGotoNextStep = function () {
      var condition = $tabData.newDictionary.id && $tabData.newDictionary.id > 0 && $tabData.environmentData.dictionaries && $tabData.environmentData.dictionaries.length > 0;
      confirmDialogService.openConfirmDialog(condition, 'Dictionaries is empty, are you sure you want to go to the next step?').then(function () {
        WizardHandler.wizard().next();
      });
    };
    // STEP 1 : name/path
    // Autocomplete path
    $scope.envPathAutocompleteHandlers = { addCandidates: environmentService.addCandidatesDirectories };
    $scope.$watch('$tabData.environmentId', function () {
      $tabData.postEnvError = undefined;
    });
    // STEP 2 : containers
    // New Container Modal
    $scope.existingContainersHandlers = {
      itemSelectorHandler: function () {
        return environmentService.addCandidatesAllContainers(undefined, '').then(function (data) {
          return _.map(data, function (container) {
            return {
              id: utils.removePrefix(container.id),
              type: container.type
            };
          });
        });
      }
    };
    // Create New Containers
    $scope.containerTypeNames = typeSystemService.getContainerTypeNames();
    $scope.newContainerTypeSelected = function (type) {
      $tabData.containerHierarchyBundles.length = 0;
      var containerHierarchy = environmentService.getContainerHierarchy(type);
      _.forEach(containerHierarchy, function (container) {
        $tabData.containerHierarchyBundles.push(new ContainerBundle(container));
      });
    };
    $scope.$watch('$tabData.newContainerType', function (type) {
      if (type === undefined) {
        $tabData.containerHierarchyBundles.length = 0;
      }
    });
    var openNewContainerModal = function (containerBundle) {
      var xlModalInstance = $xlModal.open({
          templateUrl: 'web-src/environments/tabs/environment/container/container-modal.html',
          controller: 'newContainerModalCtrl',
          windowClass: 'xl-container-modal',
          resolve: {
            modalData: function () {
              return {
                parentContainer: containerBundle.getParent(),
                containerTypes: containerBundle.subtypes,
                handleEsc: true,
                hasClose: true,
                environmentId: $tabData.environmentId
              };
            }
          }
        });
      xlModalInstance.result.then(function (selectedItem) {
        var index = $tabData.containerHierarchyBundles.indexOf(containerBundle);
        var containerCopy = angular.copy(selectedItem);
        delete containerCopy.name;
        delete containerCopy.path;
        containerBundle.container = containerCopy;
        containerBundle.setNewContainer(true);
        for (var i = index + 1; i < $tabData.containerHierarchyBundles.length; i++) {
          $tabData.containerHierarchyBundles[i].clearContainer();
          $tabData.containerHierarchyBundles[i].setNewContainer(true);
        }
      });
    };
    $scope.createNewContainer = function (containerBundle) {
      openNewContainerModal(containerBundle);
    };
    $scope.selectExistingContainer = function (containerBundle) {
      var parentId = containerBundle.getParent() ? containerBundle.getParent().id : undefined;
      var xlModalInstance = $xlModal.open({
          templateUrl: 'web-src/environments/tabs/environment/select-existing-container/select-existing-container-modal.html',
          controller: 'selectExistingContainerModalCtrl',
          windowClass: 'xl-select-existing-container-modal',
          resolve: {
            modalData: function () {
              return {
                type: containerBundle.type,
                parentId: parentId
              };
            }
          }
        });
      xlModalInstance.result.then(function (selectedItem) {
        containerBundle.container = selectedItem;
        containerBundle.setParent();
      });
    };
    $scope.addContainer = function () {
      var containerHierarchyBundles = $tabData.containerHierarchyBundles;
      if (containerHierarchyBundles !== undefined && containerHierarchyBundles[containerHierarchyBundles.length - 1] !== undefined) {
        var container = containerHierarchyBundles[containerHierarchyBundles.length - 1].getContainer();
        $tabData.environmentData.members.push({
          id: utils.removePrefix(container.id),
          type: container.type
        });
        $tabData.newContainerType = '';
        containerHierarchyBundles.length = 0;
      }
    };
    $scope.isValidContainer = function () {
      var containerHierarchyBundles = $tabData.containerHierarchyBundles;
      return containerHierarchyBundles === undefined || containerHierarchyBundles[containerHierarchyBundles.length - 1] === undefined || !containerHierarchyBundles[containerHierarchyBundles.length - 1].isContainerSelected();
    };
    // Set-Of-Ci
    $scope.envContainersAutocompleteHandlers = {
      addCandidates: environmentService.addCandidatesAllContainers,
      onNewItem: openNewContainerModal
    };
    // Container Helper
    // 1 - $watch package
    $scope.$watch('$tabData.containerHelper.package', function (value, oldValue) {
      if (value === oldValue) {
        return;
      }
      $tabData.containerHelper.deployableCandidates.length = 0;
      $tabData.containerHelper.containerCandidates.length = 0;
      $tabData.containerHelper.deployable = undefined;
    });
    // 2 - $watch deployable
    $scope.$watch('$tabData.containerHelper.deployable', function (value, oldValue) {
      if (value === oldValue) {
        return;
      }
      $tabData.containerHelper.containerCandidates.length = 0;
    });
    // 1 - handler package
    $scope.envContainersHelperPackageHandlers = {
      addCandidates: environmentService.addCandidatesPackages,
      onSelect: function (packageCi) {
        environmentService.fetchDeployableCis(packageCi).then(function (deployables) {
          $tabData.containerHelper.deployableCandidates = deployables;
        });
      }
    };
    // 2 - handler deployable
    $scope.envContainersHelperDeployableHandlers = {
      addCandidates: function (meta, params) {
        return environmentService.addCandidatesDeployables($tabData.containerHelper.deployableCandidates, params);
      },
      onSelect: function (deployableCi) {
        environmentService.fetchContainerCis([deployableCi]).then(function (data) {
          $tabData.containerHelper.containerCandidates = data;
        });
      }
    };
    // 3 - handler container
    $scope.envContainersHelperContainerHandlers = {
      addCandidates: function (meta, params) {
        if ($tabData.containerHelper.containerCandidates.length === 0) {
          return environmentService.addCandidatesAllContainers(meta, params);
        } else {
          return environmentService.addCandidatesContainers($tabData.containerHelper.containerCandidates, params);
        }
      },
      onNewItem: function (value) {
        alert('Hey, it is still to be implemented: ' + value);
      }
    };
    // STEP 3 : dictionaries
    // List-Of-Ci
    $scope.existingDictionariesHandlers = {
      itemSelectorHandler: function () {
        return environmentService.addCandidatesAllDictionaries(undefined, '').then(function (data) {
          return data;
        });
      }
    };
    $scope.envDictionariesAutocompleteHandlers = {
      addCandidates: environmentService.addCandidatesAllDictionaries,
      onEdit: function (item) {
        environmentService.openEditDictionaryModal(item).then(function (dictionary) {
          environmentService.editDictionary(dictionary).then(function (response) {
            var dictionary = _.find($tabData.environmentData.dictionaries, { id: response.data.id });
            angular.copy(response.data, dictionary);
          });
        });
      },
      onDelete: function (item) {
        _.remove($tabData.environmentData.dictionaries, item);
      }
    };
    $scope.isValidDictionary = function () {
      return !_.isEmpty($tabData.newDictionary.id);
    };
    $scope.addDictionary = function () {
      environmentService.saveDictionary($tabData.newDictionary).then(function (response) {
        $tabData.environmentData.dictionaries.push(response.data);
        $tabData.newDictionary.id = '';
        $tabData.newDictionary.entries = {};
        $tabData.newDictionary.encryptedEntries = {};
      });
    };
    // STEP 4 : advanced properties
    $scope.environmentMetadata = typeSystemService.getTypeDescriptorOf(constantService.TYPE.UDM_ENVIRONMENT) || [];
    $scope.environmentMetadata.properties = _.filter($scope.environmentMetadata.properties, function (it) {
      return it.name !== 'members' && it.name !== 'dictionaries';
    });
    // SAVE ENVIRONMENT
    $scope.saveEnvironment = function () {
      var extendedData = _.clone($tabData.environmentData, true);
      _.forEach(extendedData.members, function (container) {
        container.id = constantService.ROOT.INFRASTRUCTURE + '/' + container.id;
      });
      // Prepare Json
      var id = utils.addLeadingSlashIfNotExists($tabData.environmentId);
      var environmentCIObj = ciService.extractCiStructure(constantService.ROOT.ENVIRONMENTS + id);
      var errorFunction = function (error) {
      };
      environmentService.createCiEnvironment(environmentCIObj, extendedData).then(function () {
        $scope.unfoldPathToNode(environmentCIObj.idWithoutRoot);
        ciTabsService.switchToViewEnvironment(environmentCIObj.idWithoutRoot);
      }, errorFunction);
    };
    // Global
    $scope.$watch('environmentForms.envNameForm.$pristine', function (value) {
      if (value === false) {
        $scope.$setTabPristine();
      }
    });
    $tabData.initialized = true;
  }
]);
'use strict';
angular.module('Environments').factory('environmentService', [
  '$q',
  'typeSystemService',
  'directoryResource',
  'versionResource',
  'environmentsResource',
  'ciResource',
  'constantService',
  'utils',
  'ciService',
  'dipHandlers',
  '$xlModal',
  function ($q, typeSystemService, directoryResource, versionResource, environmentsResource, ciResource, constantService, utils, ciService, dipHandlers, $xlModal) {
    function createJsonCis(environmentObj, extendedData) {
      var cis = ciService.createDirectoriesJsonCis(environmentObj);
      cis.push(ciService.createJsonCiWithData(environmentObj, constantService.TYPE.UDM_ENVIRONMENT, extendedData));
      return cis;
    }
    function createCiEnvironment(environmentObj, extendedData) {
      var promises = [];
      var allCis = createJsonCis(environmentObj, extendedData);
      angular.forEach(allCis, function (ci) {
        var deferred = $q.defer();
        if (ci.type === constantService.TYPE.ENVIRONMENTS) {
          angular.extend(ci, extendedData);
        }
        ciResource.ciExists(ci.id).then(function (response) {
          var exists = response.data.boolean;
          if (!exists) {
            // ci doesn't exist so we can create it
            return deferred.resolve(ci);
          } else {
            ciResource.get(ci.id).success(function (returnedCi) {
              if (ci.type === constantService.TYPE.CORE_DIRECTORY && returnedCi.type === constantService.TYPE.UDM_ENVIRONMENT) {
                return deferred.reject({
                  isDirectoryError: true,
                  message: 'An environment named ' + utils.removePrefix(ci.id) + ' already exists'
                });
              } else if (ci.type === constantService.TYPE.UDM_ENVIRONMENT) {
                return deferred.resolve(ci);
              }
              return deferred.resolve();
            });
          }
        });
        promises.push(deferred.promise);
      });
      return $q.all(promises).then(function (cis) {
        var deferred = $q.defer();
        ciResource.saveAllCis(_.without(cis, undefined)).then(deferred.resolve, function (error) {
          return deferred.reject({
            isEnvironmentError: true,
            message: error.data
          });
        });
        return deferred.promise;
      });
    }
    var fetchContainerInstances = function (containerTypes) {
      var promises = _.map(containerTypes, function (containerType) {
          return ciResource.findAllCis(containerType);
        });
      return $q.all(promises);
    };
    var fetchDeployableCis = function (packageCi) {
      if (packageCi && packageCi.id) {
        return ciResource.get(packageCi.id).then(function (deployableCis) {
          return _.map(deployableCis.data.deployables, function (deployableRef) {
            return { id: deployableRef };
          });
        });
      } else {
        var differed = $q.defer();
        differed.resolve([]);
        return differed.promise;
      }
    };
    var fetchContainerCis = function (deployables) {
      var arrPromises = _.map(deployables, function (deployable) {
          return ciResource.get(deployable.id);
        });
      return $q.all(arrPromises).then(function (deployableCis) {
        var allTypes = [];
        _.forEach(deployableCis, function (deployableCi) {
          var deployableType = deployableCi.data.type;
          var types = typeSystemService.getTypesByDeployableType(deployableType);
          allTypes = _.union(allTypes, types);
        });
        var containerTypes = _.map(allTypes, 'containerType');
        return fetchContainerInstances(containerTypes).then(function (result) {
          return _.chain(result).reduceRight(function (a, b) {
            return a.concat(b.data);
          }, []).uniq('ref').map(function (item) {
            item.id = item.ref;
            delete item.ref;
            return item;
          }).value();
        });
      });
    };
    var addCandidatesPackages = function (meta, params) {
      return dipHandlers.ciHandler(constantService.TYPE.UDM_VERSION, params.term);
    };
    var addCandidatesDeployables = function (deployables, params) {
      var differed = $q.defer();
      var candidates = _.filter(deployables, function (item) {
          return item.id.indexOf(params.term) !== -1;
        });
      differed.resolve(candidates);
      return differed.promise;
    };
    var addCandidatesContainers = function (containers, params) {
      var differed = $q.defer();
      var candidates = _.filter(containers, function (item) {
          return item.id.indexOf(params.term) !== -1;
        });
      differed.resolve(candidates);
      return differed.promise;
    };
    var addCandidatesAllDictionaries = function (containers, params) {
      return dipHandlers.ciHandler(constantService.TYPE.UDM_DICTIONARY, params.term);
    };
    var addCandidatesDirectories = function (meta, params) {
      return directoryResource.findAllDirectoriesByAncestor(constantService.ROOT.ENVIRONMENTS, params.term).then(utils.getCiIdsWithoutPrefix);
    };
    var addCandidatesAllContainers = function (meta, params) {
      return dipHandlers.ciHandler(constantService.TYPE.UDM_CONTAINER, params.term);
    };
    var getContainerHierarchy = function (typeName) {
      var containerHierarchy = computeContainerHierarchy(typeName);
      return enrichContainerHierarchy(containerHierarchy);
    };
    var enrichContainerHierarchy = function (containerHierarchy) {
      var enrichContainerHierarchy = [];
      _.forEach(containerHierarchy, function (containerDescription) {
        var elem = {};
        elem.type = containerDescription.type;
        elem.subtypes = [];
        if (containerDescription.virtual) {
          elem.subtypes = typeSystemService.getTypeDescriptorOfSuperType(containerDescription.type);
        } else {
          elem.subtypes.push(containerDescription);
        }
        enrichContainerHierarchy.unshift(elem);
      });
      return enrichContainerHierarchy;
    };
    var computeContainerHierarchy = function (typeName, containers) {
      if (containers === undefined) {
        containers = [];
      }
      var typeDescriptorOf = typeSystemService.getTypeDescriptorOf(typeName);
      if (typeDescriptorOf) {
        containers.push(typeDescriptorOf);
        var asContainmentProperties = _.find(typeDescriptorOf.properties, 'asContainment');
        if (asContainmentProperties) {
          computeContainerHierarchy(asContainmentProperties.referencedType, containers);
        }
      } else {
        //Virtual
        containers.push({
          'type': typeName,
          'virtual': true
        });
      }
      return containers;
    };
    var saveDictionary = function (newDictionary) {
      var ciObj = ciService.normalizeCiId(constantService.ROOT.ENVIRONMENTS, undefined, newDictionary.id);
      var ci = ciService.createJsonCiWithData(ciObj, constantService.TYPE.UDM_DICTIONARY, {
          entries: newDictionary.entries,
          encryptedEntries: newDictionary.encryptedEntries
        });
      return ciResource.saveCi(ci);
    };
    var editDictionary = function (newDictionary) {
      var ciObj = ciService.extractCiStructure(newDictionary.id);
      var ci = ciService.createJsonCiWithData(ciObj, newDictionary.type, {
          entries: newDictionary.entries,
          encryptedEntries: newDictionary.encryptedEntries
        });
      return ciResource.editCi(ci.id, ci);
    };
    var openEditDictionaryModal = function (dictionary) {
      function openModal() {
        var modalOptions = {
            templateUrl: 'web-src/environments/tabs/environment/edit-dictionary-modal/edit-dictionary-modal.html',
            windowClass: 'edit-dictionary-modal',
            controller: 'editDictionaryModalCtrl',
            resolve: {
              modalData: function () {
                return { dictionary: angular.copy(dictionary) };
              }
            }
          };
        return $xlModal.open(modalOptions).result;
      }
      if (!dictionary.entries) {
        return ciService.getCi(dictionary.id).then(function (response) {
          dictionary.entries = response.data.entries;
          dictionary.encryptedEntries = response.data.encryptedEntries;
          return openModal();
        });
      } else {
        return openModal();
      }
    };
    return {
      fetchDeployableCis: fetchDeployableCis,
      fetchContainerCis: fetchContainerCis,
      addCandidatesPackages: addCandidatesPackages,
      addCandidatesDeployables: addCandidatesDeployables,
      addCandidatesContainers: addCandidatesContainers,
      addCandidatesDirectories: addCandidatesDirectories,
      addCandidatesAllContainers: addCandidatesAllContainers,
      addCandidatesAllDictionaries: addCandidatesAllDictionaries,
      createCiEnvironment: createCiEnvironment,
      getContainerHierarchy: getContainerHierarchy,
      computeContainerHierarchy: computeContainerHierarchy,
      enrichContainerHierarchy: enrichContainerHierarchy,
      saveDictionary: saveDictionary,
      editDictionary: editDictionary,
      openEditDictionaryModal: openEditDictionaryModal
    };
  }
]);
'use strict';
angular.module('Environments').controller('selectExistingContainerModalCtrl', [
  '$scope',
  '$xlModalInstance',
  'modalData',
  'utils',
  'ciResource',
  'constantService',
  function ($scope, $xlModalInstance, modalData, utils, ciResource, constantService) {
    $scope.selectedContainers = [];
    $scope.existingContainersHandlers = {
      itemSelectorHandler: function () {
        return ciResource.findAllCisByType(modalData.type, modalData.parentId).then(function (response) {
          return _.map(response.data, function (item) {
            item.id = utils.removePrefix(item.ref);
            delete item.ref;
            return item;
          });
        });
      }
    };
    $scope.hasSelectedContainer = function () {
      return $scope.selectedContainers.length > 0;
    };
    $scope.cancel = function () {
      closeDialogModal();
    };
    $scope.addContainer = function () {
      var selected = $scope.selectedContainers[0];
      selected.id = constantService.ROOT.INFRASTRUCTURE + '/' + selected.id;
      $xlModalInstance.close(selected);
    };
    /**
     * Close dialog
     */
    var closeDialogModal = function () {
      $xlModalInstance.dismiss();
    };
    $xlModalInstance.handlers = {
      onEscModal: closeDialogModal,
      onCloseModal: closeDialogModal
    };
  }
]);
'use strict';
angular.module('Environments').config([
  '$uiTabsProvider',
  function ($uiTabsProvider) {
    $uiTabsProvider.tab('environments.env.create', {
      title: 'New environment',
      templateUrl: 'web-src/environments/tabs/environment/environment-ci.html',
      controller: 'environmentCtrl'
    }).tab('environments.env.view', {
      title: 'Visual environment',
      templateUrl: 'web-src/environments/tabs/environment/environment-ci.html',
      controller: 'environmentCtrl'
    }).welcome({ templateUrl: 'web-src/environments/tabs/welcome/welcome.html' }).onClose([
      'confirmDialogService',
      function (confirmDialogService) {
        return function (tab) {
          return confirmDialogService.openConfirmDialog(tab.pristine, 'Are you sure you want to leave? Any changes will be lost.');
        };
      }
    ]);
  }
]);
'use strict';
angular.module('Environments').factory('ciTabsService', [
  'utils',
  '$uiTabs',
  'constantService',
  function (utils, $uiTabs, constantService) {
    // environment tabs
    var newEnvironment = function () {
      openOrReplaceTab('environments.env.create', { $tabData: { mode: constantService.MODES.CREATION } }, 'FORCE_NEW');
    };
    var viewEnvironment = function (environmentId, forceNewTab) {
      openOrReplaceTab('environments.env.view', {
        title: utils.removePrefix(environmentId),
        $tabData: {
          environmentId: environmentId,
          mode: constantService.MODES.VIEW
        }
      }, forceNewTab);
    };
    var editEnvironment = function () {
      var currentTab = $uiTabs.getActiveTab();
      openOrReplaceTab('environments.env.edit', {
        title: currentTab.title,
        initialized: currentTab.initialized,
        data: currentTab.data
      }, 'FORCE_SAME');
    };
    var switchToViewEnvironment = function () {
      $uiTabs.getActiveTab().pristine = true;
      $uiTabs.removeTab($uiTabs.getActiveTab().$$id);
    };
    /**
     * Opens or replace a CI tab
     * @param targetState target tab type
     * @param options tab options
     * @param tabMode
     *  FORCE_NEW - always opens new tab,
     *  FORCE_SAME - switches the same tab if possible,
     *  AUTO - if current tab is in visual mode, switches the current tab, if not - opens new
     */
    var openOrReplaceTab = function (targetState, options, tabMode) {
      var currentTab = $uiTabs.getActiveTab();
      if (!currentTab || !!currentTab.isWelcome || tabMode === 'FORCE_NEW') {
        $uiTabs.addTab(targetState, options);
      } else if (currentTab && (currentTab.$tabData.mode === 'visual' || tabMode === 'FORCE_SAME')) {
        $uiTabs.addTab(targetState, options, currentTab.$$id);
      }
    };
    /**
     * Public API
     */
    return {
      newEnvironment: newEnvironment,
      viewEnvironment: viewEnvironment,
      editEnvironment: editEnvironment,
      switchToViewEnvironment: switchToViewEnvironment
    };
  }
]);