/*
 * This file is part of Maven Deployit plugin.
 *
 * Maven Deployit plugin is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Maven Deployit plugin is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Maven Deployit plugin.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.xebialabs.deployit.maven;

import java.io.File;
import java.util.Collections;
import java.util.List;
import org.apache.maven.settings.Server;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.apache.maven.settings.Settings;
import org.apache.maven.settings.crypto.DefaultSettingsDecryptionRequest;
import org.apache.maven.settings.crypto.SettingsDecrypter;
import org.apache.maven.settings.crypto.SettingsDecryptionResult;
import org.apache.maven.wagon.authentication.AuthenticationInfo;
import com.google.common.base.Function;

import com.xebialabs.deployit.booter.remote.BooterConfig;
import com.xebialabs.deployit.booter.remote.DeployitCommunicator;
import com.xebialabs.deployit.booter.remote.RemoteBooter;
import com.xebialabs.deployit.maven.helper.DeploymentHelper;
import com.xebialabs.deployit.maven.helper.DeploymentLogger;
import com.xebialabs.deployit.maven.helper.ServerVersionCompatibility;
import com.xebialabs.deployit.maven.packager.MavenDarPackager;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Lists.transform;
import static java.lang.String.format;

/**
 * Provides common code for deployit mojos
 *
 * @author Benoit Moussaud
 */
public abstract class AbstractDeployitMojo extends AbstractMojo {


    /**
     * The maven project.
     *
     * @parameter expression="${project}"
     * @required
     * @readonly
     */
    protected MavenProject project;

    /**
     * All maven projects in the reactor.
     *
     * @parameter expression="${reactorProjects}"
     * @required
     * @readonly
     */
    protected List<MavenProject> reactorProjects;

    /**
     * Plugin dependencies.
     *
     * @parameter default-value="${plugin.artifacts}"
     * @required
     * @readonly
     */
    private java.util.List<org.apache.maven.artifact.Artifact> pluginArtifacts = Collections.emptyList();

    SettingsDecrypter settingsDecrypter;

    /**
     * The name of the application for this DAR file, i.e. the value of
     * the {@code CI-Application} attribute.
     *
     * @parameter default-value="${project.artifactId}"
     * @required
     * @since 3.9.0
     */
    protected String applicationName;

    /**
     * If the project packaging is {@code ear} or {@code war}, sets the
     * {@code CI-Name} property for the EAR/WAR artifact in the DAR.
     *
     * @parameter default-value="${project.artifactId}"
     * @required
     */
    protected String mainArtifactName;


    /**
     * Activate the skip mode: generate the plan, skip all the steps, validate the task
     *
     * @parameter default-value=false
     */
    protected boolean skipMode;

    /**
     * Set the orchestrator used during the deployment.
     *
     * @parameter default-value=""
     * @since 3.7.0
     */
    protected String orchestrator;

    /**
     * Activate the test mode, generate the plan, display all the steps, validate the task
     *
     * @parameter default-value=false
     */
    protected boolean testMode;

    /**
     * If a deployments leads no steps, fail the build.
     *
     * @parameter default-value=false
     */
    protected boolean failIfNoStepsAreGenerated;

    /**
     * Deployit server address
     *
     * @parameter default-value="localhost" expression="${deployit.server}"
     */
    protected String serverAddress;

    /**
     * Deployit Listen port. The default value of -1 will automatically use 4516 for HTTP and 4517 for HTTPS connections.
     *
     * @parameter default-value="-1" expression="${deployit.port}"
     */
    protected int port;


    /**
     * Deployit username. See <b>FAQ</b> for more details
     *
     * @parameter default-value="" expression="${deployit.username}"
     */
    protected String username;


    /**
     * Deployit password. See <b>FAQ</b> for more details
     *
     * @parameter default-value="" expression="${deployit.password}"
     */
    protected String password;

    /**
     * context of the deployit server
     *
     * @parameter default-value="deployit" expression="${deployit.context}"
     */
    private String context = "deployit";
    /**
     * Id of the server in the setting.xml file, will be used if you don't specify <i>username</i> and <i>password</i> in pom.xml.
     *
     * @parameter default-value="deployit-credentials" expression="${deployit.credentials}"
     * @since 3.6.4
     */
    private String server;

    /**
     * If true the communication with the deployit server is secured (https).
     *
     * @parameter default-value=false
     */
    protected boolean secured;


    /**
     * Id of the environment used for the deployment.
     *
     * @parameter default-value="" expression="${deployit.environmentId}"
     */
    protected String environmentId;

    /**
     * List of package-level properties
     *
     * @parameter
     */
    protected List<XmlFragment> deploymentPackageProperties;

    /**
     * List of the deployeds: extensions or complete if you use explicitDeployeds options
     *
     * @parameter
     */
    protected List<MavenDeployed> deployeds;

    /**
     * With explicitDeployeds true, the deployed are not generated but fully loaded from the plugin configuration.
     *
     * @parameter default-value=false
     */
    protected boolean explicitDeployeds;


    /**
     * List of the deployables, including artifacts or middleware resource specification (eg Datasource)
     *
     * @parameter
     */
    protected List<MavenDeployable> deployables = newArrayList();

    /**
     * List of container in the target environment, if you want to create the environment through the Maven plugin.
     *
     * @parameter
     */
    protected List<MavenContainer> environment;

    /**
     * Use this attribute to add a timestamp to the version of the deployit package.
     * by default,the SNAPSHOT versions are automatically timestamped. This flag is useful only if you want to timestamp non SNAPSHOT version.
     *
     * @parameter default-value = false  expression="${deployit.timestamp}"
     */
    protected boolean timestampedVersion;

    /**
     * Package version to be used instead of the automatically generated version.
     *
     * @parameter default-value = ""  expression="${deployit.packageVersion}"
     */
    protected String packageVersion;

    /**
     * Package version to be used instead of the automatically generated version.
     *
     * @Parameter(defaultValue = "${settings}", readonly = true)
     */
    private Settings settings;

    /**
     * Delete the previous deployed dar. Useful if you work with the SNAPSHOT versions you don't want to keep in your repository.
     *
     * @parameter default-value=false  expression="${deployit.delete.previous.dar}"
     */
    protected boolean deletePreviouslyDeployedDar;

    /**
     * Flag controlling whether, during the upgrade operation, the Deployed objects should generated (like an initial deployment) or reused.
     * For security reasons, the default value is false but should be set to true to apply the modifications (new Ear, removed links) even during upgrade.
     *
     * @parameter default-value=false  expression="${deployit.generate.deployed.on.upgrade}"
     * @since 3.6.2
     */
    protected boolean generateDeployedOnUpgrade;

    /**
     * When a task falls in error, it is cancelled. Sometime it could be interesting to debug it using the UI or the CLI. When this flag is set to false, the
     * task will be left as is.
     *
     * @parameter default-value=true  expression="${deployit.cancel.task.on.error}"
     * @since 3.6.2
     */
    protected boolean cancelTaskOnError;

    protected DeployitCommunicator communicator;

    private MavenDarPackager packager;

    private DeploymentHelper deploymentHelper;

    private ServerVersionCompatibility versionChecker;

    /**
     * Socket timeout in milliseconds. Used when connection is made to XL Deploy server.
     * The default value is 10 seconds.
     *
     * @parameter default-value="10000" expression="${deployit.socketTimeout}"
     * @since 4.5.2
     */
    protected int socketTimeout;

    MavenDarPackager getPackager() {
        if (packager == null) {
            packager = new MavenDarPackager(project, reactorProjects, applicationName, timestampedVersion || isSnapshotVersion(), getLog(), packageVersion);
            if (deployables.isEmpty())
                addMainArtifact();
            packager.addProperties(deploymentPackageProperties);
            //Inject the project into the MavenDeployables
            packager.addDeployables(transform(deployables, new Function<MavenDeployable, MavenDeployable>() {
                @Override
                public MavenDeployable apply(MavenDeployable input) {
                    input.setProject(project);
                    return input;
                }
            }));
        }
        return packager;
    }

    private void addMainArtifact() {
        final MavenDeployable mainArtifactDeployable;
        try {
            mainArtifactDeployable = generateMainDeployable();
            if (mainArtifactDeployable != null) {
                deployables.add(mainArtifactDeployable);
            }
        } catch (MojoExecutionException e) {
            e.printStackTrace();
        }
    }

    private MavenDeployable generateMainDeployable() throws MojoExecutionException {

        final String packaging = project.getPackaging();
        if (packaging.equalsIgnoreCase("dar") || packaging.equalsIgnoreCase("pom")) {
            getLog().debug("No main artifact for this packaging " + packaging);
            return null;
        }
        String ciType;
        if (packaging.equalsIgnoreCase("war")) {
            ciType = "jee.War";
        } else if (packaging.equalsIgnoreCase("ear")) {
            ciType = "jee.Ear";
        } else {
            getLog().error("Unsupported pom packaging type. You must supply the deployable definition for the main artifact.");
            return null;
        }

        MavenDeployable mainArtifact = new MavenDeployable();
        mainArtifact.setType(ciType);
        mainArtifact.setGroupId(project.getGroupId());
        mainArtifact.setArtifactId(mainArtifactName);
        final File file = project.getArtifact().getFile();
        if (file != null) {
            mainArtifact.setLocation(file);
        } else {
            throw new MojoExecutionException("Cannot automatically discover main artifact file location. You must supply the deployable definition for the main artifact.");
        }
        return mainArtifact;
    }

    private boolean isSnapshotVersion() {
        return project.getVersion().contains("SNAPSHOT");
    }

    protected String getUsername() {
        if (this.username == null) {
            getLog().debug("No username defined in pom.xml and " +
                    "no system property defined either. Trying to look up " +
                    "username in settings.xml under server element " + server);
            AuthenticationInfo authenticationInfo = this.getAuthenticationInfo(server);
            if (authenticationInfo == null) {
                getLog().warn(format("In settings.xml server element '%s' was not defined. Using default username.", server));
                return null;
            }
            if (authenticationInfo.getUserName() != null) {
                getLog().debug(format(" username '%s' found in the settings.xml file", authenticationInfo.getUserName()));
                return authenticationInfo.getUserName();
            } else {
                getLog().warn(format(
                        "In settings.xml no username was found for server element '%s'. Does the element exist? Using default username.", server));
                return null;
            }
        }
        return username;
    }

    protected String getPassword() {
        if (this.password == null) {
            getLog().debug("No password defined in pom.xml and " +
                    "no system property defined either. Trying to look up " +
                    "password in settings.xml under server element " + server);
            AuthenticationInfo authenticationInfo = this.getAuthenticationInfo(server);
            if (authenticationInfo == null) {
                getLog().warn(format("In settings.xml server element '%s' was not defined. Using default password.", server));
                return null;
            }
            if (authenticationInfo.getPassword() != null) {
                getLog().debug(format(" password '%s' found in the settings.xml file", "********"));
                return authenticationInfo.getPassword();
            } else {
                getLog().warn(format(
                        "In settings.xml no password was found for server element '%s'. Does the element exist? Using default password", server));
                return null;
            }
        }
        return password;
    }

    /**
     * Boots REST API
     */
    protected void boot() {
        BooterConfig config = BooterConfig.builder()
                .withProtocol(secured ? BooterConfig.Protocol.HTTPS : BooterConfig.Protocol.HTTP)
                .withCredentials(getUsername(), getPassword())
                .withHost(serverAddress)
                .withPort(port)
                .withSocketTimeout(socketTimeout)
                .withContext(context)
                .build();

        communicator = RemoteBooter.boot(config);

        try {
            //getVersionChecker().check();
        } catch (IllegalStateException ex) {
            getLog().error(ex.getMessage());
            communicator.shutdown();
            throw new IllegalStateException(ex.getMessage());
        }
    }

    protected void shutdown() {
        communicator.shutdown();
    }

    public void setProject(MavenProject project) {
        this.project = project;
    }

    public MavenProject getProject() {
        return project;
    }

    public void setEnvironmentId(String environmentId) {
        this.environmentId = environmentId;
    }

    public void setEnvironment(List<MavenContainer> environment) {
        this.environment = environment;
    }

    public void setMainArtifactName(String mainArtifactName) {
        this.mainArtifactName = mainArtifactName;
    }

    public void setDeletePreviouslyDeployedDar(boolean deletePreviouslyDeployedDar) {
        this.deletePreviouslyDeployedDar = deletePreviouslyDeployedDar;
    }

    public void setExplicitDeployeds(boolean explicitDeployeds) {
        this.explicitDeployeds = explicitDeployeds;
    }

    public void setPort(int port) {
        this.port = port;
    }

    /**
     * For testing purposes
     *
     * @param deploymentHelper deployment helper injection for testing
     */
    public void setDeploymentHelper(DeploymentHelper deploymentHelper) {
        this.deploymentHelper = deploymentHelper;
    }

    protected DeploymentHelper getDeploymentHelper() {
        if (deploymentHelper == null) {
            deploymentHelper = new DeploymentHelper(new DeploymentLogger() {
                @Override
                public void debug(final String s) {
                    getLog().debug(s);
                }

                @Override
                public void info(final String s) {
                    getLog().info(s);
                }

                @Override
                public void warn(final String s) {
                    getLog().warn(s);
                }

                @Override
                public void error(final String s) {
                    getLog().error(s);
                }
            }, communicator);
        }
        return deploymentHelper;
    }

    public AuthenticationInfo getAuthenticationInfo(String id) {
        if (id != null) {
            List<Server> servers = settings.getServers();

            if (servers != null) {
                for (Server server : servers) {
                    if (id.equalsIgnoreCase(server.getId())) {
                        SettingsDecryptionResult result =
                            settingsDecrypter.decrypt(new DefaultSettingsDecryptionRequest(server));
                        server = result.getServer();

                        AuthenticationInfo authInfo = new AuthenticationInfo();
                        authInfo.setUserName(server.getUsername());
                        authInfo.setPassword(server.getPassword());
                        authInfo.setPrivateKey(server.getPrivateKey());
                        authInfo.setPassphrase(server.getPassphrase());

                        return authInfo;
                    }
                }
            }
        }

        return null;
    }

    public void setVersionChecker(ServerVersionCompatibility versionChecker) {
        this.versionChecker = versionChecker;
    }

    protected ServerVersionCompatibility getVersionChecker() {
        if (versionChecker == null) {
            versionChecker = new ServerVersionCompatibility(communicator, pluginArtifacts);
        }

        return versionChecker;
    }

    public void setTestMode(boolean testMode) {
        this.testMode = testMode;
    }

    protected boolean hasDeployeds() {
        return deployeds != null && deployeds.size() > 0;
    }

    public void setGenerateDeployedOnUpgrade(final boolean generateDeployedOnUpgrade) {
        this.generateDeployedOnUpgrade = generateDeployedOnUpgrade;
    }

    public List<MavenDeployable> getDeployables() {
        return deployables;
    }

    public List<XmlFragment> getDeploymentPackageProperties() {
        return deploymentPackageProperties;
    }

    public String getPackageVersion() {
        return packageVersion;
    }

    public void setPackageVersion(final String packageVersion) {
        this.packageVersion = packageVersion;
    }

    public boolean isTimestampedVersion() {
        return timestampedVersion;
    }

    public void setTimestampedVersion(final boolean timestampedVersion) {
        this.timestampedVersion = timestampedVersion;
    }

    public Settings getSettings() {
        return settings;
    }

    public void setSettings(Settings settings) {
        this.settings = settings;
    }

    public SettingsDecrypter getSettingsDecrypter() {
        return settingsDecrypter;
    }

    public void setSettingsDecrypter(SettingsDecrypter settingsDecrypter) {
        this.settingsDecrypter = settingsDecrypter;
    }
}
