import akka.http.javadsl.model.HttpRequest
import akka.http.javadsl.model.headers.HttpCredentials
import com.xebialabs.impact.api.CrawlerCredentialsDTO
import com.xebialabs.impact.api.DataSourceTypeEnum
import com.xebialabs.impact.wave.scripting.PluginInterface
import com.xebialabs.impact.wave.scripting.http.PushDataRequestVM
import com.xebialabs.impact.wave.scripting.um.UmModel

import java.util.stream.StreamSupport


class WorkspacesToCrawlFields {
    String id;
    String numericId
}

class CrawledFields {
    List<CrawledField> fields = new ArrayList<>()
    Map<String, CrawledField> fieldsById = new HashMap<>()
    Map<String, CrawledField> fieldsByDisplayName = new HashMap<>()
}

enum FieldType {
    STRING, MULTIVALUE
}

class CrawledField {
    String objectType;
    String refObjectName;
    String displayName;
    String elementName;
    String workspaceId
    String compositeDisplayName
    String compositeId
    FieldType fieldType
}

class CrawledFieldsOfWorkspace {
    List<CrawledField> fields = new ArrayList<>()
}

PluginInterface.map(CrawlFields).into(RequestCustomFieldsFromCache).via({crawlFields, ctx ->
    ctx.emit(new RequestCustomFieldsFromCache(true))
})

class RequestCustomFields {

}

PluginInterface.map(RequestCustomFieldsFromCache).into(RequestCustomFields, CachedCustomFields).via({request, ctx ->
    if (request.force || CustomFieldsLocalCache.crawledFields == null) {
        ctx.emit(new RequestCustomFields())
    } else {
        ctx.emit(new CachedCustomFields(CustomFieldsLocalCache.crawledFields))
    }
})

class RequestCustomFieldsForSubscription {
    String subscriptionId
}

PluginInterface.request(RequestCustomFields).into(RequestCustomFieldsForSubscription).mapToRequest({request, ctx ->
    RequestHelper.get(ctx.getCrawlerMessageWithCredentials().credentials.endpoint + "webservice/v2.0/subscription", ctx)
}).parseResponse({request, httpResponse, parser, ctx ->
    parser.readStringValueAndThen(".Subscription.ObjectID", {subscriptionId ->
        RequestCustomFieldsForSubscription r = new RequestCustomFieldsForSubscription()
        r.subscriptionId = subscriptionId
        ctx.emit(r)
    })
})

PluginInterface.request(RequestCustomFieldsForSubscription).into(WorkspacesToCrawlFields).mapToRequest({request, ctx ->
    RequestHelper.get(ctx.getCrawlerMessageWithCredentials().credentials.endpoint + "webservice/v2.0/Subscription/" + request.subscriptionId + "/Workspaces", ctx)
})
.parseResponse({request, httpResponse, parser, ctx ->
    parser.readStringValueAndThen(".QueryResult.Results[].ObjectID", {workspaceId ->
        List<String> filterValues = ctx.crawlerMessageWithCredentials.credentials.getFilterValues("WORKSPACE_IDS")
        if (filterValues.isEmpty() || filterValues.stream().filter({workspaceId.equals(it)}).count() > 0) {
            def workspace = new WorkspacesToCrawlFields()
            workspace.id = workspace.numericId = workspaceId
            ctx.emit(workspace)
        }
    })
})

PluginInterface.request(WorkspacesToCrawlFields).into(CrawledFieldsOfWorkspace, WorkspacesToCrawlFields)
        .mapToRequest({ workspace, ctx ->
    CrawlerCredentialsDTO credentials = ctx.crawlerMessageWithCredentials.credentials
    RequestHelper.get(credentials.endpoint + "schema/v2.0/workspace/" + workspace.id, ctx)
})
        .parseResponse({ workspace, httpResponse, parser, ctx ->
    if (httpResponse.status().isRedirection()) {
        def redirectedWorkspace = new WorkspacesToCrawlFields()
        redirectedWorkspace.numericId = workspace.numericId
        redirectedWorkspace.id = StreamSupport.stream(httpResponse.getHeaders().spliterator(), false)
                .filter({ h -> h.lowercaseName().equals("location") })
                .findFirst()
                .orElseThrow({ new IllegalArgumentException("Strange redirect without location header") })
                .value()
                .replaceAll('^.*\\/slm\\/schema\\/v2\\.0\\/workspace\\/', '')
        ctx.emit(redirectedWorkspace)
    } else {
        CrawledFieldsOfWorkspace crawledFieldsOfWorkspace = new CrawledFieldsOfWorkspace();
        parser.onObject(".QueryResult.Results[]", {
            List<CrawledField> fieldsOfThisType = new ArrayList<>()
            def type = parser.readValueInto("._refObjectName")
            parser.onObject(".Attributes[]", {
                CrawledField crawledField = new CrawledField();
                crawledField.workspaceId = workspace.numericId
                parser.readBoolValueAndThen(".Custom", {
                    if (it) {
                        fieldsOfThisType.add(crawledField)
                    }
                })
                parser.readStringValueAndThen(".AttributeType", {
                    crawledField.fieldType = it.equals("COLLECTION") ? FieldType.MULTIVALUE : FieldType.STRING
                })
                parser.readStringValueAndThen(".Name", { crawledField.displayName = it })
                parser.readStringValueAndThen("._refObjectName", { crawledField.refObjectName = it })
                parser.readStringValueAndThen(".ElementName", { crawledField.elementName = it })
            })
            parser.onFinish({
                fieldsOfThisType.forEach({
                    it.objectType = type.toString()
                    it.compositeDisplayName = it.workspaceId + "/" + it.objectType + "/" + it.displayName.toUpperCase()
                    it.compositeId = it.workspaceId + "/" + it.objectType + "/" + it.elementName
                })
                crawledFieldsOfWorkspace.fields.addAll(fieldsOfThisType)
            })
        })
        parser.onFinish({ ctx.emit(crawledFieldsOfWorkspace)})
    }
})

PluginInterface.reduce(CrawledFieldsOfWorkspace).into(PushDataPackage, CachedCustomFields)
        .via({ stream ->
    def fields = new CrawledFields()
    stream.get({ fieldsOfWorkspace ->
        fieldsOfWorkspace.fields.forEach({
            fields.fieldsByDisplayName.put(it.compositeDisplayName, it)
            fields.fieldsById.put(it.compositeId, it)
        })
        fields.fields.addAll(fieldsOfWorkspace.fields)
    })
    stream.onFinish({
        PushDataPackage pushDataPackage = new PushDataPackage()
        pushDataPackage.crawledFields = fields
        stream.emit(pushDataPackage)
        stream.emit(new CachedCustomFields(fields))
        CustomFieldsLocalCache.crawledFields = fields
    })
})

class FieldTitle {
    String displayName
}

enum FieldUmType {
    multiSelect, stringNotAnalyzed
}

class FieldUmModel {
    FieldUmType type
    FieldTitle title = new FieldTitle()
}

class FieldsUmModel extends UmModel {
    Map<String, FieldUmModel> fields = new HashMap<>();
}