class ReleasePage {
    constructor() {
        Browser.waitFor('#release');
    }

    waitForReleaseRefreshed() {
        return new ReleasePage();
    }

    expectOnePhaseDisplayed() {
        expect(element(By.css(".phase")).isPresent()).toBe(true);
        return this;
    }

    toggleNavigationDropDown() {
        return element(By.css('#release-header .subpage-links button')).click();
    }

    openSummary() {
        this.openSubPage('Release dashboard');
        return new ReleaseSummaryPage();
    }

    openGantt() {
        this.selectView('Planner');
        return new ReleaseGanttPage();
    }

    openXFile() {
        this.selectView('Releasefile');
        return new DslPage();
    }

    openProperties() {
        this.openSubPage('Properties');
        return new ReleasePropertiesPage();
    }

    openVariables() {
        this.openSubPage('Variables');
        return new VariablesPage('#release-variables');
    }

    openTriggers() {
        this.openSubPage('Triggers');
        return new TriggersPage();
    }

    openTemplateProperties() {
        this.openSubPage('Properties');
        return new TemplatePropertiesPage();
    }

    openEditor() {
        this.openSubPage('Release flow');
        return new ReleasePage();
    }

    openGrid() {
        this.selectView('Table');
        return new ReleaseGridPage();
    }

    openLogs() {
        this.openSubPage('Activity logs');
        return new ActivityLogsPage();
    }

    openPermissions() {
        this.openSubPage('Permissions');
        return new ReleasePermissionsPage();
    }

    selectView(name) {
        return element(By.$(`#release-header a:contains('${name}')`)).click();
    }

    downloadExcel() {
        this.exportTemplateAs('Excel');
        return Download.expectDownloadAt('export/excel/');
    }

    exportToZip() {
        this.exportTemplateAs('XLR');
        return Download.expectDownloadAt('export/zip/');
    }

    openSubPage(name) {
        Browser.waitFor('#release-header .subpage-links button');
        element(By.css('#release-header .subpage-links button')).click();
        element(By.$(`#subpages-container div:contains('${name}')`)).click();
        return this;
    }

    expectSubPage(name, toBePresent = true) {
        element(By.css('#release-header .subpage-links button')).click();
        expect(element(By.$(`#subpages-container a:contains('${name}')`))).toBePresent(toBePresent);
        Browser.clickElseWhere();
        return this;
    }

    expectTitleToBe(releaseTitle) {
        expect(element(By.$('.bread-crumbs span')).getText()).toContain(releaseTitle);
        return this;
    }

    expectSubPageTitleToBe(title) {
        expect(element(By.css('#release-header .subpage-links button')).getText()).toContain(title);
        return this;
    }

    restartPhases() {
        element(By.css('#release-header .restart-phases.button')).click();
        return new RestartPhaseModal();
    }

    start() {
        element(By.css('.start-release')).click();
        element(By.$(".modal:visible .button:contains('Start')")).click();
        return this;
    }

    resume() {
        element(By.css('.resume.button')).click();
        return this;
    }

    addPhase() {
        element(By.css(".add-phase")).click();
        return this;
    }

    getPhase(phaseName) {
        return new ReleasePhase(`.phase:contains('${phaseName}')`);
    }

    openScriptDetails(title) {
        return this.openTaskDetails(title, () => new ScriptDetails());
    }

    openManualTaskDetails(title) {
        return this.openTaskDetails(title, () => new ManualTaskDetails());
    }

    openGateTaskDetails(title) {
        return this.openTaskDetails(title, () => new GateTaskDetails());
    }

    openNotificationDetails(title) {
        return this.openTaskDetails(title, () => new NotificationDetails());
    }

    openCustomScriptDetails(title) {
        return this.openTaskDetails(title, () => new CustomScriptDetails());
    }

    openUserInputTaskDetails(title) {
        return this.openTaskDetails(title, () => new UserInputTaskDetails());
    }

    openCreateReleaseTaskDetails(title) {
        return this.openTaskDetails(title, () => new CreateReleaseTaskDetails());
    }

    openSequentialGroupTaskDetails(title) {
        return this.openTaskDetails(title, () => new SequentialGroupTaskDetails());
    }

    openTaskDetails(title, buildResult) {
        element.all(By.$(`.task-title:contains('${title}')`)).first().click();
        return buildResult();
    }

    openPhaseDetails(title) {
        element(By.$(`.phase:contains('${title}') .phase-details`)).click();
        return new ReleasePhaseDetails();
    }

    openVersionControl() {
        this.openSubPage('Version control');
        return new VersionControl();
    }

    waitForCompletion(timeout) {
        Browser.waitFor('#release.completed', timeout);
        return this;
    }

    waitForTaskStarted(title, timeout) {
        Browser.waitFor(`.task:contains('${title}').in_progress`, timeout);
        return this;
    }

    waitForTaskCompleted(title, timeout) {
        Browser.waitFor(`.task:contains('${title}').completed`, timeout);
        return this;
    }

    waitForTaskFailed(title, timeout) {
        Browser.waitFor(`.task:contains('${title}').failed`, timeout);
        return this;
    }

    waitForTaskNeedsInput(title, timeout) {
        Browser.waitFor(`.task:contains('${title}').waiting_for_input`, timeout);
        return this;
    }

    waitForTaskSkipped(title, timeout) {
        Browser.waitFor(`.task:contains('${title}').skipped`, timeout);
        return this;
    }

    waitForTaskPending(title, timeout) {
        Browser.waitFor(`.task:contains('${title}').pending`, timeout);
        return this;
    }

    waitForStatusLine(title, timeout) {
        Browser.waitFor(`.task .status-line:contains('${title}')`, timeout);
        return this;
    }

    expectTaskPlanned(title) {
        expect(element(By.$(`.task:contains('${title}').planned`)).isPresent()).toBe(true, `expected task [${title}] to be planned`);
        return this;
    }

    expectTaskInProgress(title) {
        expect(element(By.$(`.task:contains('${title}').in_progress`)).isPresent()).toBe(true, `expected task [${title}] to be in progress`);
        return this;
    }

    expectTaskCompleted(title) {
        expect(element(By.$(`.task:contains('${title}').completed`)).isPresent()).toBe(true, `expected task [${title}] to be completed`);
        return this;
    }

    expectTaskPending(title) {
        expect(element(By.$(`.task:contains('${title}').pending`)).isPresent()).toBe(true, `expected task [${title}] to be pending`);
        return this;
    }

    expectTaskToHaveIcon(title, icon) {
        expect(element(By.$(`.task:contains('${title}') .labels ${icon}`)).isPresent()).toBe(true, `expected task [${title}] to have icon ${icon}`);
        return this;
    }

    expectTaskToHaveBeenFlagged(title) {
        expect(element(By.$(`.task:contains('${title}') .task-infos .flag-count`)).isPresent()).toBe(true);
    }

    expectTaskToHaveFailed(title, failCount) {
        expect(element(By.$(`.task:contains('${title}') .task-infos .failure-count`)).isPresent()).toBe(true);
        expect(element(By.$(`.task:contains('${title}') .task-infos .failure-count:visible`)).getText()).toContain(failCount);
    }

    expectTaskToHaveBeenDelayed(title) {
        expect(element(By.$(`.task:contains('${title}') .task-infos .delay-count`)).isPresent()).toBe(true);
    }

    expectTaskToBePresent(title) {
        expect(element(By.$(`.task:contains('${title}')`))).toBePresent();
        return this;
    }

    expectTaskNotToBePresent(title) {
        expect(element(By.$(`.task:contains('${title}')`)).isPresent()).toBe(false);
        return this;
    }

    toggleTaskFilter(filterOption) {
        this.clickFilterOptions();
        element(By.css(`#filter-container #${filterOption}`)).click();
        this.clickFilterOptions();
        return this;
    }

    clickOnStatusLine(title) {
        element(By.$(`.task .status-line a:contains('${title}')`)).click();
        return this;
    }

    clickFilterOptions() {
        element(By.css("#release-header .completed-release-filter-button")).click();
        return this;
    }

    expandCompletedPhases() {
        browser.sleep(100);
        element(By.css('.phase .expand')).click();
        return this;
    }

    collapseCompletedPhases() {
        element(By.css(".collapse-completed-phases")).click();
        return this;
    }

    expectCompletedPhasesCollapsed(count) {
        expect(element.all(By.$('.collapsed:visible')).count()).toBe(1);
        expect(element(By.css('.collapsed .phase-title')).getText()).toContain(count);
        return this;
    }

    expectCompletedPhases(count) {
        expect(element.all(By.$('.collapsed:visible')).count()).toBe(0);
        expect(element.all(By.css('.phase.done')).count()).toBe(count);
        return this;
    }

    expectPlannedPhasesExpanded(count) {
        expect(element.all(By.css('.phase .planned-phase .arrow-down-icon')).count()).toBe(count);
        return this;
    }

    abort() {
        element(By.$('.abort-release')).click();
        element(By.$('.modal .continue')).click();
        return this;
    }

    expectStatus(status) {
        expect(element(By.$(`#release.${status}`)).isPresent()).toBe(true);
        return this;
    }

    expectReleaseStatus(status) {
        expect(element(By.$(`.release-status:contains(${status})`)).isPresent()).toBe(true);
        return this;
    }

    expectAborted() {
        return this.expectStatus('aborted');
    }

    expectInProgress() {
        return this.expectStatus('in_progress');
    }

    expectHasPhase(title) {
        expect(element(By.$(`.phase-title:contains('${title}')`)).isPresent()).toBe(true, `Release should have a phase with title ${title}`);
    }

    newRelease() {
        element(By.css('.new-release')).click();
        return new ReleasePropertiesPage();
    }

    exportToDsl() {
        this.exportTemplateAs('Releasefile');
        return Download.expectDownloadAt('api/v1/dsl/export/');
    }

    exportTemplateAs(name) {
        element(By.css(".export-buttons.dropdown-button")).click();
        return element(By.$(`.export-buttons-container div:contains('${name}')`)).click();
    }

    expectTemplateLabelToBePresent(expectation = true) {
        expect(element(By.$(".template-label")).isPresent()).toBe(expectation);
    }
}

class ReleasePhase {
    constructor(phase) {
        this.phase = phase;
    }

    setTitle(title, withScroll = true) {
        let editor = new InlineEditor(`${this.phase} .phase-title`);
        editor.set(title, withScroll);
        return this;
    }

    focusTitle() {
        let editor = new InlineEditor(`${this.phase} .phase-title`);
        editor.focus();
    }

    openTaskGenerator() {
        element(By.$(`${this.phase} .add-task`)).click();
        return this;
    }

    getTask(taskName, type) {
        let typeClass = !type ? '' : `.task-type-${type.toLowerCase()}`;
        return new ReleaseTask(`${this.phase} .task:contains('${taskName}')${typeClass}`);
    }

    getTaskGenerator() {
        return new TaskGenerator(this.phase);
    }

    getNumberOfTasks() {
        return element.all(By.$(`${this.phase} .task`)).count();
    }

    expectTaskBorderWithColor(task, color) {
        expect(element(By.$(`.task-box:contains('${task}')`)).getCssValue('border-top-color')).toBe(color);
    }

    toggleCollapseState() {
        browser.sleep(100);
        element(By.$(`${this.phase} .expander i`)).click();
        browser.sleep(100);
        return this;
    }

    expectToHaveCollapsedTasks(count) {
        expect(element.all(By.$(`${this.phase} .task:visible`)).count()).toBe(0);
        let expectedCount = count === 1 ? 'one' : count;
        expect(element(By.$(`${this.phase} .view-nested-tasks`)).getText()).toContain(expectedCount);
        return this;
    }

    expectToHaveExpandedTasks(present) {
        if (present) {
            expect(element.all(By.$(`${this.phase} .task:visible`)).count()).toBeGreaterThan(0);
        } else {
            expect(element.all(By.$(`${this.phase} .task:visible`)).count()).toBe(0);
        }
        expect(element.all(By.$(`${this.phase} > .view-nested-tasks:visible`)).count()).toBe(0);
        return this;
    }

    expectToHaveTitle(title) {
        let editor = new InlineEditor(`${this.phase} .phase-title`);
        expect(editor.value()).toBe(title);
        return this;
    }

    expectToHaveNoneditableTitle(title) {
        expect(element(By.$(`${this.phase} .phase-title > span`)).getText()).toBe(title);
        return this;
    }

}


class ReleasePhaseDetails extends Modal {
    constructor() {
        super(...arguments);
        Browser.waitFor('.modal');
    }

    setTitle(title) {
        let editor = new InlineEditor(".phase-title");
        editor.set(title);
        return this;
    }

    setDescription(description) {
        let editor = new TextareaInlineEditor('.modal:visible .phase-description');
        editor.set(description);
        return this;
    }

    getDescription() {
        let editor = new TextareaInlineEditor('.modal:visible .phase-description');
        return editor.value();
    }

    setDueDate(date) {
        let dateTimePicker = new DateTimePicker('.modal .due-date .date-editor');
        dateTimePicker.setDateTime(date);
        return this;
    }

    expectDueTimeToBe(timeFormat, date) {
        Dates.expectInputContainingDate(By.$('.modal .due-date .time-picker-holder input'), timeFormat, date);
        return this;
    }

    setScheduledStartDate(date) {
        let dateTimePicker = new DateTimePicker('.modal .scheduled-start-date .date-editor');
        dateTimePicker.setDateTime(date);
        return this;
    }

    setDuration(duration) {
        let dateTimePicker = new DurationPicker(".modal .duration-editor");
        dateTimePicker.setDuration(duration);
        return this;
    }

    close() {
        element(By.$('.modal button.close')).click();
        return this;
    }

    expectScheduledStartDateToBeDisplayed() {
        expect(element(By.$('.modal .scheduled-start-date'))).toBeDisplayed();
        return this;
    }

    expectScheduledStartDateToBe(formattedDate) {
        expect(element(By.$(`.modal .scheduled-start-date .date:contains("${formattedDate}")`)));
        return this;
    }

    expectDueDateToBeDisplayed() {
        expect(element(By.$('.modal .due-date'))).toBeDisplayed();
        return this;
    }

    expectStartDateToBeDisplayed() {
        expect(element(By.$('.modal .start-date'))).toBeDisplayed();
        return this;
    }

    expectEndDateToBeDisplayed() {
        expect(element(By.$('.modal .end-date'))).toBeDisplayed();
        return this;
    }

    expectScheduledStartDateDayToBe(day) {
        new DateTimePicker('.scheduled-start-date .date-editor').expectDateToContain(day);
        return this;
    }

    expectDurationToBe(duration) {
        let durationPicker = new DurationPicker(".modal .duration-editor:first");
        durationPicker.expectDurationToBe(duration);
        return this;
    }
}


class TaskGenerator {
    constructor(selector) {
        this.selector = selector;
        this.taskGenerator = element(By.$(this.selector));
    }

    open() {
        browser.sleep(100);
        this.taskGenerator.element(By.css('.add-task')).click();
        return this;
    }

    addTask(taskName, taskGroup, taskType) {
        this.open();
        this.addTaskForVisibleTitle(taskName, taskGroup, taskType);
        return this;
    }

    addTaskForVisibleTitle(taskName, taskGroup, taskType) {
        this.taskGenerator.element(By.css('.quick-title')).sendKeys(taskName);
        this.taskGenerator.element(By.css('.xlr-ctx-menu-toggle')).click();
        $$(`li.xlr-ctx-menu-item[data-test='${taskGroup}']`)
            .filter((el) => el.isDisplayed().then(_ => _))
            .first()
            .click();
        $$(`li.xlr-ctx-menu-item[data-test='${taskType}']`)
            .filter((el) => el.isDisplayed().then(_ => _))
            .first()
            .click();
        this.taskGenerator.element(By.css(".add")).click();
        return this;
    }

    expectTaskTypeDisabled(taskGroup, taskType) {
        this.open();
        this.taskGenerator.element(By.css('.xlr-ctx-menu-toggle')).click();
        $$(`li.xlr-ctx-menu-item[data-test='${taskGroup}']`)
            .filter((el) => el.isDisplayed().then(_ => _))
            .first()
            .click();
        expect($$(`li.xlr-ctx-menu-item[data-test='${taskType}']`).count()).toBe(0);
        this.taskGenerator.element(By.css('.xlr-ctx-menu-toggle')).click();
        return this.close();
    }

    addManualTask(taskName) {
        return this.addTask(taskName, 'Core', 'xlrelease.Task');
    }

    addNotificationTask(taskName) {
        return this.addTask(taskName, 'Core', 'xlrelease.NotificationTask');
    }

    addScriptTask(taskName) {
        return this.addTask(taskName, 'Script', 'xlrelease.ScriptTask');
    }

    addGateTask(taskName) {
        return this.addTask(taskName, 'Core', 'xlrelease.GateTask');
    }

    addParallelGroup(taskName) {
        return this.addTask(taskName, 'Core', 'xlrelease.ParallelGroup');
    }

    addCreateReleaseTask(taskName) {
        return this.addTask(taskName, 'Core', 'xlrelease.CreateReleaseTask');
    }

    addSequentialGroup(taskName) {
        return this.addTask(taskName, 'Core', 'xlrelease.SequentialGroup');
    }

    close() {
        this.taskGenerator.element(By.css('.cancel')).click();
        return this;
    }
}

class ReleaseTask {
    constructor(task) {
        this.task = task;
    }

    toggleCollapseState() {
        browser.sleep(100);
        element(By.$(`${this.task} .expander i`)).click();
        return this;
    }

    viewNestedTasks() {
        element(By.$(`${this.task} .view-nested-tasks`)).click();
        return this;
    }

    expectToHaveCollapsedTasks(count) {
        expect(element.all(By.$(`${this.task} .task:visible`)).count()).toBe(0);
        let expectedCount = count === 1 ? 'one' : count;
        expect(element(By.$(`${this.task} .task-group .view-nested-tasks`)).getText()).toContain(expectedCount);
        return this;
    }

    expectToHaveExpandedTasks(present) {
        if (present) {
            expect(element.all(By.$(`${this.task} .task:visible`)).count()).toBeGreaterThan(0);
        } else {
            expect(element.all(By.$(`${this.task} .task:visible`)).count()).toBe(0);
        }
        expect(element.all(By.$(`${this.task} > .view-nested-tasks:visible`)).count()).toBe(0);
        return this;
    }

    getTaskGenerator() {
        return new TaskGenerator(this.task);
    }

    openContextMenu() {
        element(By.$(`${this.task} .context-menu-button`)).click();
        expect(element.all(By.css("#context-menu-container")).count()).toBe(1);
        return this;
    }

    openChangeTypeMenu() {
        browser.actions().mouseMove(element(By.css(`.dropdown-submenu[data-content-id="Change type"]`))).perform();
        return this;
    }

    openSubmenu(menuId) {
        browser.actions().mouseMove(element(By.css(`.dropdown-submenu[data-content-id="${menuId}"]`))).perform();
        return this;
    }

    closeContextMenu() {
        expect(element.all(By.css("#context-menu-container")).count()).toBe(1);
        element(By.$(`${this.task} .context-menu-button`)).click();
        return this;
    }

    expectNoChangeTypeSubMenu() {
        expect(element(By.$('.dropdown-submenu'))).toBePresent(false);
        return this;
    }

    selectMenuItem(item) {
        this.openContextMenu();
        return element(By.$(`#context-menu-container a:contains('${item}')`)).click();
    }

    expectDisabledContextMenuItem(item) {
        this.openContextMenu();
        expect(element.all(By.$(`#context-menu-container a:contains('${item}')[disabled]`)).count()).toBe(1);
        return this;
    }

    expectEnabledContextMenuItem(item) {
        this.openContextMenu();
        expect(element.all(By.$(`#context-menu-container a:contains('${item}')`)).count()).toBe(1);
        expect(element.all(By.$(`#context-menu-container a:contains('${item}')[disabled]`)).count()).toBe(0);
        return this;
    }

    expectStartsDateTimeToBe(formattedDate, formattedTime) {
        expect(element(By.$(`${this.task} span:contains("${formattedDate} at ${formattedTime}")`))).toBePresent();
        return this;
    }

    complete() {
        this.selectMenuItem('Complete');
        expect(element(By.$('.modal:visible .modal-body')).getText()).toContain('complete the task');
        element(By.$(".modal:visible .button:contains('Complete')")).click();
        return this;
    }

    delete() {
        this.selectMenuItem('Delete');
        expect(element(By.$('.modal:visible .modal-body')).getText()).toContain('delete the task');
        element(By.$(".modal:visible .button:contains('Delete')")).click();
        return this;
    }

    duplicate() {
        this.selectMenuItem('Duplicate');
        return this;
    }

    expectTitleToBe(title) {
        expect(element(By.$(`${this.task} .task-title`)).getText()).toBe(title);
        return this;
    }
}

class VersionControl {

    showChanges() {
        element(By.$('button:contains("Show changes")')).click();
        return this;
    }

    hideChanges() {
        element(By.$('button:contains("Hide changes")')).click();
        return this;
    }

    saveANewVersion(title, description) {
        element(By.$('button:contains("Save a new version")')).click();
        element(By.$('.modal #versionName')).sendKeys(title);
        element(By.$('.modal #versionDescription')).sendKeys(description);
        element(By.$('.modal button:contains("Save")')).click();
        return this;
    }

    restore(description) {
        element.all(by.cssContainingText('.template-history-item', description))
            .first()
            .element(By.$('a:contains("Restore")'))
            .click();
        element(By.$('.modal button:contains("Restore")')).click();
        return this;
    }

    clickOnCompare() {
        element(By.$('button:contains("Compare")')).click();
        return this;
    }

    closeCompareModal() {
        element(By.$('.modal button:contains("Close")')).click();
        return this;
    }

    select(number) {
        browser.actions().mouseMove(element.all(by.css(`.template-history-item-wrapper`)).get(number)).perform();
        element.all(by.css(`.template-history-item-wrapper .compare-checkbox>input`)).get(number).click();
        return this;
    }

    closeToastMessage() {
        element(By.$('.toast-close-button')).click();
        return this;
    }

}

global.ReleasePage = ReleasePage;
global.ReleasePhaseDetails = ReleasePhaseDetails;
