const defaultSize = {
    width: 1280,
    height: 800
};

class Browser {
    static open() {
        browser.get('#/');
        return this;
    }

    static setDefaultSize() {
        return this.setSize(defaultSize.width, defaultSize.height);
    }

    static setSize(width, height) {
        const resize = (windowWidth, windowHeight) =>
            window.resizeTo(windowWidth, windowHeight);
        this.executeVoidScript(resize, width, height);
    }

    static expectUrlToBe(url) {
        expect(browser.getCurrentUrl()).toContain(url);
        return this;
    }
    static expectUrlToContain = Browser.expectUrlToBe;

    static expectUrlParams(params) {
        const parseToUrl = params => Object.entries(params).map(([key, value]) => {
            if (_.isBoolean(value))
                return key;
            else if (_.isArray(value))
                return value.map(e => `${key}=${e}`).join('&');
            else
                return `${key}=${value}`;
        }).join('&');

        const paramString = `?` + parseToUrl(params);

        return this.wait(EC.urlContains(paramString), 'waiting for URL params to change');
    }

    static wait(conditionFunction, message, timeout = 30000) {
        return browser.wait(conditionFunction, timeout, message);
    }

    static waitFor(cssSelector, timeout) {
        return this.wait(EC.presenceOf(element(By.$(cssSelector))), `waiting for '${cssSelector}' to be present`, timeout);
    }

    static waitForDisplayed(cssSelector, timeout) {
        return this.wait(EC.visibilityOf(element(By.$(cssSelector))), `waiting for '${cssSelector}' to be present && displayed`, timeout);
    }

    static scrollTo(locator) {
        let scrollIntoView = function () {
            return arguments[0].scrollIntoView(false);
        };
        return this.executeVoidScript(scrollIntoView, browser.findElement(locator));
    }

    static executeVoidScript(scriptFunction, ...scriptArguments) {
        const script = `(${scriptFunction}).apply(null, arguments);`;
        return browser.executeScript(script, ...scriptArguments);
    }

    static hover(locator) {
        return browser.actions().mouseMove(element(locator)).perform();
    }

    static blurActiveElement() {
        return this.executeVoidScript(() => document.activeElement.blur());
    }

    static clickElseWhere() {
        return this.executeVoidScript(() => {
            $(document).mousedown();
            $(document).click();
        });
    }

    static softReload() {
        let remoteScript = function () {
            let state = angular.element('body').injector().get('$state');
            return state.reload();
        };

        browser.sleep(500); // to provide running requests with some time to finish
        Browser.executeVoidScript(remoteScript);
    }

    // used to isolate tests with download bar in separate window
    static switchToNewWindow() {
        const script = `window.open('#/', '_blank', 'menubar=0,width=${defaultSize.width},height=${defaultSize.height}')`;
        browser.executeScript(script).then(() => {
            browser.getAllWindowHandles().then((handles) => {
                const newWindow = handles[1];
                browser.switchTo().window(newWindow);
            });
        });
    }

    static backToOriginalWindow() {
        browser.executeScript('window.close()').then(() => {
            browser.getAllWindowHandles().then((handles) => {
                const originalWindow = handles[0];
                browser.switchTo().window(originalWindow);
            });
        });
    }

    static waitWithDump(by, timeout = 30000) {
        let found = false;
        const until = new Date().getTime() + timeout;

        const check = (elements) => {
            if (!found && !elements.length) {
                if (new Date().getTime() < until) {
                    return new Promise(res => setTimeout(res, 100));
                }
            } else if (elements.length) {
                found = true;
            }
        };
        for (let i = 0; i < timeout / 100; i ++) {
            element.all(by).then(check);
        }

        return browser.getPageSource()
            .then((source) => {
                if (!found) {
                    throw new Error(`No luck waiting for ${by}, here is what we have: ${source}`);
                }
            });
    }

    static debug(selector) {
        let timeout = 0;
        element.all(By.$(selector)).then(function waitForElements(elements) {
            timeout = timeout || (new Date().getTime() + 5000);
            console.log(`debugging [${selector}]`);
            if (! elements.length) {
                if (new Date().getTime() < timeout) {
                    return new Promise(res => { setTimeout(res, 100); }).then(() => element.all(By.$(selector))).then(waitForElements);
                }
                console.log(`Selector [${selector}] did not appear. Come connect and help me! I\'m at pid = ${process.pid}, if you did not start with --inspect and you are on Mac or Linux, do\n\nkill -s SIGUSR1 ${process.pid}\n\n to enable debugging then connect with NodeJS debugger (Chrome chrome://inspect/#devices), CLEAR YOUR CONSOLE and say\n\nx = element.all(By.$(${JSON.stringify(selector)}))\n\n in the console`);
                global.x = false;
                return Promise.resolve().then(function xxx() {
                    return new Promise((res, rej) => {
                        function yyy() {
                            if (global.x === false) {
                                setTimeout(yyy, 100);
                            } else {
                                res();
                            }
                        }

                        yyy();
                    }).then(() => {
                        if (global.x === true) {
                            global.x = false;
                            return;
                        } else if (global.x === 1) {
                            global.x = false;
                        debugger;
                            return xxx();
                        } else if ('string' === typeof global.x) {
                            try {
                                console.log(eval(global.x));
                            } catch (e) {
                                console.log(e);
                            }
                            global.x = false;
                            return xxx();
                        } else {
                            if ('function' === typeof global.x.then) {
                                try {
                                    return global.x.then(
                                        (r) => {
                                            console.log(r);
                                            global.x = false;
                                            return xxx();
                                        },
                                        (e) => {
                                            console.log(e);
                                            global.x = false;
                                            return xxx();
                                        });
                                } catch (e) {
                                    console.log(e);
                                    return xxx();
                                }
                            } else {
                                console.log('not a promise');
                                global.x = false;
                                return xxx();
                            }
                        }
                    });
                });
            }
        });
    }
}

global.Browser = Browser;
