class TaskDetails extends Modal {
    constructor() {
        super(...arguments);
        this.taskDetails = element(By.id('resizable-modal'));
    }

    getDescriptionEditor() {
        return new TextareaInlineEditor("#resizable-modal .task-description");
    }

    close() {
        return this.taskDetails.element(By.$('button.close')).click();
    }

    setTitle(title) {
        let editor = new InlineEditor("#resizable-modal .task-title");
        editor.set(title);
        return this;
    }

    getAttachmentsList() {
        Browser.waitForDisplayed('.attachments-title-expanded:visible, .attachments-title-collapsed:visible');
        element(By.$('.attachments-title-collapsed')).isPresent().then(function (isPresent) {
            if (isPresent) {
                return element.all(By.$('.attachments-details span')).first().click();
            }
        });
        return new AttachmentsList('#resizable-modal .attachments-details');
    }

    flag(flagName, flagComment) {
        this.taskDetails.element(By.$('.flags .dropdown-button')).click();
        expect(element(By.$("#resizable-modal .flags .popover"))).toBeDisplayed();
        this.taskDetails.element(By.$(`.flags .flag-picker-popover .${flagName}`)).click();
        let editor = new InlineEditor('#resizable-modal .flag-comment');
        editor.set(flagComment);
        return this;
    }

    completeTask() {
        this.taskDetails.element(By.$(".button:contains('Complete'):visible")).click();
        Browser.waitFor("#resizable-modal textarea[ng-model=feedback]:visible");
        expect(element(By.$("#resizable-modal textarea[ng-model=feedback]:visible"))).toBeDisplayed();
        return this.taskDetails.element(By.$("form[name=completeTaskForm] button:contains('OK')")).click();
    }

    failTask(comment) {
        this.taskDetails.element(By.$(".button:contains('Fail'):visible")).click();
        expect(element(By.$("#resizable-modal textarea[ng-model=failFeedback]:visible"))).toBeDisplayed();
        element(By.$("#resizable-modal textarea[ng-model=failFeedback]:visible")).sendKeys(comment);
        return this.taskDetails.element(By.$("form[name=failTaskForm] button:contains('OK')")).click();
    }

    abortTask(comment) {
        this.taskDetails.element(By.$(".button:contains('Abort'):visible")).click();
        expect(element(By.$("#resizable-modal textarea[ng-model=abortFeedback]:visible"))).toBeDisplayed();
        element(By.$("#resizable-modal textarea[ng-model=abortFeedback]:visible")).sendKeys(comment);
        this.taskDetails.element(By.$("form[name=abortTaskForm] button:contains('OK')")).click();
        return this;
    }

    retryTask(comment) {
        Browser.scrollTo(by.$(".button:contains('Retry'):visible"));
        this.taskDetails.element(By.$(".button:contains('Retry'):visible")).click();
        expect(element(By.$("#resizable-modal textarea[ng-model=retryFeedback]:visible"))).toBeDisplayed();
        element(By.$("#resizable-modal textarea[ng-model=retryFeedback]:visible")).sendKeys(comment);
        this.taskDetails.element(By.$("form[name=retryTaskForm] button:contains('OK')")).click();
        return this;
    }

    skipTask(comment) {
        this.taskDetails.element(By.$(".button:contains('Skip'):visible")).click();
        expect(element(By.$("#resizable-modal textarea[ng-model=skipFeedback]:visible"))).toBeDisplayed();
        element(By.$("#resizable-modal textarea[ng-model=skipFeedback]:visible")).sendKeys(comment);
        return this.taskDetails.element(By.$("form[name=skipTaskForm] button:contains('OK')")).click();
    }

    startTask() {
        this.taskDetails.element(By.$(".button:contains('Start'):visible")).click();
        return this;
    }

    startNow(comment, withWarningText) {
        this.taskDetails.element(By.$(".button:contains('Start now'):visible")).click();
        expect(element.all(By.$("textarea[ng-model=feedback]:visible")).count()).toBe(1);
        if (withWarningText) {
            expect(this.taskDetails.element(By.$(`.task-actions-info span.error:contains('${withWarningText}')`))).toBePresent();
        }
        expect(this.taskDetails.all(By.$(".button:contains('Start now'):visible")).count()).toBe(0);
        element(By.$("#resizable-modal textarea[ng-model=feedback]:visible")).sendKeys(comment);
        this.taskDetails.element(By.$(".button:contains('OK')")).click();
        return this;
    }

    assignToMe() {
        expect(this.taskDetails.element(By.css('.assign-to-me'))).toBeDisplayed();
        this.taskDetails.element(By.css('.assign-to-me')).click();
        return this;
    }

    assignTo(name) {
        const editor = new UserInlineEditor('.task-owner');
        editor.focus();
        editor.enter(name);
        return editor.blur();
    }

    assignUsingCompletionTo(name) {
        const editor = new UserInlineEditor('.task-owner');
        editor.set(name);
        return this;
    }

    expectTaskIsStartable(condition) {
        expect(element(By.$(".button:contains('Start'):enabled")).isPresent()).toBe(condition);
        return this;
    }

    expectTaskIsCommentable(isCommentable) {
        expect(element.all(By.$("#resizable-modal .add-comment")).count()).toEqual((isCommentable ? 1 : 0));
        return this;
    }

    expectControlButtonsPresent(present) {
        expect(this.taskDetails.element(By.$(".task-actions .button:visible"))).toBePresent(present);
        return this;
    }

    setPrecondition(script) {
        let editor = new CodeInlineEditor('#resizable-modal .task-precondition');
        editor.set(script);
        return this;
    }

    expectWithComment(comment) {
        element.all(By.$('.comments span')).first().click();
        element(By.$('.comments-title-collapsed')).isPresent().then(function (isPresent) {
            if (isPresent) {
                return element.all(By.$('.comments span')).first().click();
            }
        });
        expect(element(By.$("#resizable-modal .comment")).getText()).toContain(comment);
        return this;
    }

    setVariable(variable, value) {
        let editor = new InlineEditor(`#resizable-modal .form-group:contains('${variable}') div[inline-text-editor]`);
        editor.set(value);
        return this;
    }

    setDueDate(date) {
        let dateTimePicker = new DateTimePicker('#resizable-modal .due-date');
        dateTimePicker.setDateTime(date);
        return this;
    }

    expectScheduledStartDateToBe(date) {
        expect(element(By.$('.scheduled-start-date .date-editor:visible .day')).getText()).toBe(date.format('DD'));
        expect(element(By.$('.scheduled-start-date .date-editor:visible .date-detail')).getText()).toContain(date.format('MMMM YYYY'));
        expect(element(By.$('.scheduled-start-date .date-editor:visible .time input')).getAttribute('value')).toBe(date.format('h:mm A'));
        return this;
    }

    addMeAsWatcher() {
        this.taskDetails.element(By.$('.assign-me-as-watcher')).click();
        return this;
    }

    addAsWatcher(name) {
        const editor = new UserInlineEditor('.add-watcher-input');
        editor.set(name);
        return this;
    }

    removeUserAsWatcher(user) {
        this.taskDetails.element(By.$(`.watchers-list li:contains('${user}') .close-icon`)).click();
        return this;
    }

    expectUserNotToBeAWatcher(user) {
        Browser.waitFor('.watchers-list');
        expect(this.taskDetails.element(By.$(`.watchers-list li:contains('${user}')`)).isPresent()).toBe(false);
        return this;
    }

    expectUserAsWatcher(user) {
        Browser.waitFor('.watchers-list');
        expect(this.taskDetails.element(By.$(`.watchers-list li:contains('${user}')`))).toBePresent();
        return this;
    }

    getComment() {
        return this.taskDetails.element(By.css('.modal-body div.comments p')).getText();
    }
}

class ManualTaskDetails extends TaskDetails {
}

class SequentialGroupTaskDetails extends TaskDetails {
}

class GateTaskDetails extends TaskDetails {

    editLastDependency() {
        element(By.$(`#resizable-modal .dependency:last i.options-icon`)).click();
        element(By.$('#resizable-modal .dependency:last #editDependency:visible')).click();
        return new XlrDependencyWithVariables("#resizable-modal .dependency:last");
    }

    addDependency() {
        element(By.css(".add-dependency")).click();
        return new XlrDependencyWithVariables("#resizable-modal .dependency:last");
    }

    getDependencyCount() {
        return element.all(by.repeater('dependency in task.dependencies')).count();
    }

    expectDependenciesDisplayed(dependencies) {
        expect(element.all(By.css("#resizable-modal .dependency")).count()).toBe(dependencies.length);
        for (let title in dependencies) {
            expect(element(By.$(`#resizable-modal .dependency:contains('${title}') .line`)).isPresent());
        }
        return this;
    }

    expectConditionsDisplayed(conditions) {
        for (let title in conditions) {
            let expected = conditions[title];
            expect(element(By.$(`#resizable-modal .condition:contains('${title}')`)).isPresent()).toBe(expected);
        }
        return this;
    }

    expectCanDeleteDependency(title, canDelete = true) {
        if (canDelete) {
            element(By.$(`#resizable-modal .dependency:contains('${title}') i.options-icon`)).click();
        }
        expect(element(By.$(`#resizable-modal .dependency:contains('${title}') #removeDependency`))).toBePresent(canDelete);
        return this;
    }

    expectContextMenu(title, expected = true) {
        expect(element(By.$(`#resizable-modal .dependency:contains('${title}') i.options-icon`)).isPresent()).toBe(expected);
        return this;
    }

}

class ScriptDetails extends TaskDetails {

    constructor() {
        super();
        this.inlineEditor = new CodeInlineEditor('.script-content');
    }

    setScript(value) {
        this.inlineEditor.set(value);
        return this;
    }

    getScript() {
        return this.inlineEditor.value();
    }

    openCodeEditor() {
        this.inlineEditor.focus();
    }

    openFullScreenEditor() {
        return this.inlineEditor.fullScreen();
    }
}

class NotificationDetails extends TaskDetails {
    expectAddressToContain(address) {
        expect(element(By.css('#resizable-modal .email-addresses')).getText()).toContain(address);
        return this;
    }

    expectSubjectToBe(subject) {
        expect(element(By.css('#resizable-modal .email-subject')).getText()).toBe(subject);
        return this;
    }

    expectBodyToBe(body) {
        expect(element(By.css('#resizable-modal .email-body')).getText()).toBe(body);
        return this;
    }
}

class CustomScriptDetails extends TaskDetails {
    groupPath(field) {
        return `#resizable-modal #${field}`;
    }

    editorPath(field) {
        return `#resizable-modal #${field} .field`;
    }

    setIntegerField(field, value) {
        const integerField = element(By.$(`#resizable-modal #${field} .editor input:visible`));
        integerField.click();
        integerField.sendKeys(value);
        Browser.clickElseWhere();
        return this;
    }

    setTextField(field, value) {
        new InlineEditor(this.editorPath(field)).set(value);
        return this;
    }

    setSelectField(field, value) {
        new InlineEditor(this.editorPath(field)).select(value);
        return this;
    }

    getAutocompleteField(field) {
        return new XlWidgetAutocomplete(this.groupPath(field));
    }

    setAutocompleteField(field, value) {
        this.getAutocompleteField(field).focus().select(value);
        return this;
    }

    getPasswordField(field) {
        return new XlrInlinePasswordWithVariables(this.groupPath(field));
    }

    getStringMapField(field) {
        return new XlrMapStringStringWithVariables(this.groupPath(field));
    }

    getStringListField(field) {
        return new XlrListStringWithVariables(this.groupPath(field));
    }

    getStringSetField(field) {
        return new XlrSetStringWithVariables(this.groupPath(field));
    }

    getBooleanField(field) {
        return new XlDipCheckbox(this.groupPath(field));
    }

    setBooleanField(field, value) {
        this.getBooleanField(field).check(value);
        return this;
    }

    expectOptionNotPresent(field, value) {
        new InlineEditor(this.editorPath(field)).expectOptionNotPresent(value);
        return this;
    }

    expectFieldNotEmpty(field) {
        expect(element(By.$(`#${field} .display`)).getText()).not.toBe('');
        return this;
    }

    setTextAreaField(field, value) {
        new TextareaInlineEditor(this.editorPath(field)).set(value);
        return this;
    }

    getValue(field) {
        return new InlineEditor(this.editorPath(field)).value();
    }

    setDateTime(field, date) {
        new DateTimePicker(`#${field}`).setDateTime(date);
        return this;
    }

    expectOutputValueToContain(field, value) {
        expect(this.taskDetails.element(By.$(`#${field} .field-readonly`)).getText()).toContain(value);
        return this;
    }

    expectOutputValueToBe(field, value) {
        expect(this.taskDetails.element(By.$(`#${field} .field-readonly`)).getText()).toMatch(value);
        return this;
    }

    expectOutputValueToReallyBe(field, value) {
        expect(this.taskDetails.element(By.$(`#${field} .field-readonly`)).getText()).toBe(value);
        return this;
    }

    expectOutputVarHelpText(expected) {
        expect(this.taskDetails.element(By.$("#outputVarHelpText")).getText()).toEqual(expected);
        return this;
    }

    expectElementValueToBe(element, value) {
        expect(this.taskDetails.element(By.$(element)).getText()).toContain(value);
        return this;
    }

    expectFieldNotPresent(field) {
        expect(element(By.$(this.editorPath(field))).isPresent()).toBe(false);
    }

    expectDateTimeToBe(field, dateFomat, timeFormat, date) {
        let dateTimePicker = new DateTimePicker(`[name='${field}'] > .date-editor`);
        dateTimePicker.expectDateToBe(dateFomat, date);
        dateTimePicker.expectTimeToBe(timeFormat, date);
        return this;
    }
}


global.TaskDetails = TaskDetails;
global.GateTaskDetails = GateTaskDetails;
global.ScriptDetails = ScriptDetails;
global.ManualTaskDetails = ManualTaskDetails;
global.NotificationDetails = NotificationDetails;
global.CustomScriptDetails = CustomScriptDetails;
global.SequentialGroupTaskDetails = SequentialGroupTaskDetails;
