package com.xebialabs.xlrelease.versioning.ascode.scm

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.model.{CiSpec, Definition}
import com.xebialabs.ascode.yaml.writer.DefinitionWriter.WriterConfig
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.security.Permissions.getAuthenticatedUserName
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import com.xebialabs.xlrelease.ascode.Version.XLR_API_VERSION
import com.xebialabs.xlrelease.ascode.metadata.MetadataFields
import com.xebialabs.xlrelease.ascode.service.GenerateService.{CisConfig, EMPTY_DEFINITION_ERROR, GeneratorConfig}
import com.xebialabs.xlrelease.ascode.service._
import com.xebialabs.xlrelease.ascode.utils.StaticVariables.XLR_TEMPLATE_KIND
import com.xebialabs.xlrelease.ascode.yaml.parser.{ValueTagPreprocessingConstructor, XLRDefinitionParser}
import com.xebialabs.xlrelease.ascode.yaml.writer.XLRDefinitionWriter
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings.getDefinitionsPath
import com.xebialabs.xlrelease.domain.{Release, Trigger}
import com.xebialabs.xlrelease.repository.TriggerRepository
import com.xebialabs.xlrelease.scm.connector.JGitConnector._
import com.xebialabs.xlrelease.scm.connector._
import com.xebialabs.xlrelease.scm.data.{GitTag, ValidatedCommitInfo, VersionInfo}
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.XLReleasePermissions.GENERATE_FOLDER_CONFIGURATION
import com.xebialabs.xlrelease.service._
import com.xebialabs.xlrelease.triggers.service.TriggerService
import com.xebialabs.xlrelease.versioning.ascode.form.{CreateVersionResult, ValidationReport, ValidationReportMessage}
import com.xebialabs.xlrelease.versioning.ascode.repository.FolderValidationMessageRepository
import com.xebialabs.xlrelease.versioning.ascode.scm.connector.AsCodeJGitConnector
import com.xebialabs.xlrelease.versioning.ascode.{ExportValidator, FolderInfo, ImportValidator}
import grizzled.slf4j.Logging
import org.eclipse.jgit.api.errors.{GitAPIException, JGitInternalException}
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.data.domain.PageRequest
import org.springframework.stereotype.Service
import org.springframework.transaction.support.TransactionTemplate
import org.yaml.snakeyaml.Yaml

import java.io.ByteArrayOutputStream
import java.nio.charset.StandardCharsets
import java.util.Date
import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._
import scala.util.control.Exception.catchingPromiscuously
import scala.util.{Try, Using}

@Service
class FolderVersioningService @Autowired()(userProfileService: UserProfileService,
                                           generateService: GenerateService,
                                           definitionParser: XLRDefinitionParser,
                                           folderVersioningConfigService: FolderVersioningConfigService,
                                           configurationVariableService: ConfigurationVariableService,
                                           folderAsCodeService: FolderAsCodeService,
                                           releaseService: ReleaseService,
                                           val folderService: FolderService,
                                           @Qualifier("xlrRepositoryTransactionTemplate")
                                           transactionTemplate: TransactionTemplate,
                                           importService: ImportService,
                                           permissions: PermissionChecker,
                                           orphanDeletionService: OrphanDeletionService,
                                           folderValidationMessageRepository: FolderValidationMessageRepository,
                                           triggerRepository: TriggerRepository,
                                           triggerService: TriggerService
                                          ) extends Logging {

  // wrapping all exceptions related to SCM operations
  private lazy val wrapExceptions = catchingPromiscuously(classOf[AsCodeException], classOf[GitAPIException], classOf[JGitInternalException])
    .withApply(e => throw ScmException(s"Error performing version control operation: ${e.getMessage}", e.getCause))

  def findSettings(folderId: String): Option[FolderVersioningSettings] = wrapExceptions {
    folderVersioningConfigService.findSettings(folderId)
  }

  def getSettings(folderId: String): FolderVersioningSettings = wrapExceptions {
    findSettings(folderId)
      .getOrElse(throw ScmException("No version control settings defined for folder"))
  }

  def createOrUpdateSettings(newConfig: FolderVersioningSettings, remoteValidations: Boolean = true): FolderVersioningSettings = wrapExceptions {
    validateSettings(newConfig, remoteValidations)
    folderVersioningConfigService.createOrUpdateSettings(newConfig)
  }

  def deleteSettings(folderId: String): Unit = wrapExceptions {
    folderVersioningConfigService.deleteSettings(folderId)
  }

  def getVersions(folderId: String): (Date, Seq[VersionInfo]) = wrapExceptions {
    val config = getSettings(folderId)
    Using.resource(initConnector(config)) { scmConnector =>
      val versions = scmConnector.listVersions()
      config.lastFetched -> versions.sortBy(_.commitTime)(Ordering[Date].reverse)
    }
  }

  def fetchChanges(folderId: String): (Date, Seq[VersionInfo]) = wrapExceptions {
    val config = getSettings(folderId)
    Using.resource(initConnector(config)) { scmConnector =>
      val versions = scmConnector.pullAndListVersions()
      config.lastFetched = new Date()
      // don't fire update event on this update
      folderVersioningConfigService.updateSettingsDirectly(config)

      config.lastFetched -> versions.sortBy(_.commitTime)(Ordering[Date].reverse)
    }
  }

  def createVersion(folderId: String, version: String, description: String): CreateVersionResult = wrapExceptions {
    val config = getSettings(folderId)
    Using.resource(initConnector(config)) { scmConnector =>

      val filePath = getDefinitionsPath(config)

      val generated = generateFolder(folderId, config, filePath)

      val tag = GitTag(FolderVersioningSettings.generateTagName(config.getEffectiveTagPrefix, version).get)
      val commitInfo = ValidatedCommitInfo.create(version, description, tag).get
      val userProfile = userProfileService.findByUsername(getAuthenticatedUserName)

      scmConnector.createNewVersion(generated.blob, commitInfo, userProfile).get

      val createVersionResult = CreateVersionResult
        .versionInfoToCreateVersionResult(scmConnector.getVersion(commitInfo.tag.refName))

      // put secrets in generated placeholders
      config.secrets.putAll(generated.secrets.asJava)
      // set applied version
      markAsApplied(createVersionResult.name, config)
      folderVersioningConfigService.updateSettingsDirectly(config)

      val scmData = getScmTraceabilityData(config, createVersionResult)
      applyScmDataToDefinitions(generated.definitions, scmData)

      setValidationMessages(folderId, Seq.empty)

      createVersionResult
    }
  }

  def validate(folderId: String): ValidationReport = {
    val config = getSettings(folderId)
    val definitions = generateDefinitions(folderId, config)
    val folderInfo = FolderInfo(folderId, "", folderAsCodeService.getFolderPath(folderId))
    ValidationReport.messagesToReport(ExportValidator.validate(definitions, folderInfo))
  }

  def applyVersion(folderId: String, version: String): Unit = wrapExceptions {
    val config = getSettings(folderId)
    val includes = ImportIncludes(
      templates = config.exportTemplates,
      triggers = config.exportTriggers,
      patterns = config.exportPatterns,
      dashboards = config.exportDashboards,
      configurations = config.exportConfiguration,
      variables = config.exportVariables,
      notifications = config.exportNotifications,
      permissions = config.exportSecurity
    )

    Using.resource(initConnector(config)) { scmConnector =>
      val versionInfo = scmConnector.getVersion(version)
      val scmData = getScmTraceabilityData(config, versionInfo)

      val filePath = getDefinitionsPath(config)
      // checkout yaml file
      val blobs = scmConnector.checkout(filePath, version).get
      try {
        // generate current state of folder to merge latest secrets with the existing ones
        val currentStateGeneratedFolder = generateFolder(folderId, config, null)
        config.secrets.putAll(currentStateGeneratedFolder.secrets.asJava)
      } catch {
        case e: Throwable =>
          if (!e.getMessage.equals(EMPTY_DEFINITION_ERROR)) {
            logger.warn(s"Can't generate YAML for current state of folder ${folderId}: ${e.getMessage}")
          }
      }
      val currentSecrets = config.secrets.asScala.toMap
      // preprocess yaml (replace placeholders with secrets from the database)
      val yamlStr = new String(blobs.files.head.getContent(), StandardCharsets.UTF_8)
      val specs = preprocessYaml(yamlStr)
      // create Definition objects out of checked out yaml
      val definitions = specs.map(definitionParser.parse)
      // validate pre-applying
      val folderInfo = FolderInfo(folderId, "", folderAsCodeService.getFolderPath(folderId))
      val validator = new ImportValidator(currentSecrets, definitions, folderInfo)

      // apply the Definitions into DB in a single transaction
      val result = applyDefinitionsInTransaction(includes, folderId, definitions, Some(scmData), Some(config.gitConnection), Some(validator))

      markAsApplied(version, config)
      folderVersioningConfigService.updateSettingsDirectly(config)

      val validationReport = ValidationReport.messagesToReport(result.validationMessages)
      setValidationMessages(folderId, validationReport.warnings.asScala.toSeq)
    }
  }

  def generatePreview(folderId: String, version: Option[String]): String = wrapExceptions {
    val config = getSettings(folderId)
    val filePath = getDefinitionsPath(config)
    val blobs = version match {
      case Some(ver) =>
        Using.resource(initConnector(config)) { scmConnector =>
          scmConnector.checkout(filePath, ver, reset = false).get
        }
      case None => generateFolder(folderId, config, filePath).blob
    }
    new String(blobs.files.head.getContent(), StandardCharsets.UTF_8)
  }

  def getValidationMessage(messageId: Integer): ValidationReportMessage = {
    folderValidationMessageRepository.findValidationMessageById(messageId).orNull
  }

  def getValidationMessages(folderId: String): ValidationReport = {
    ValidationReport.createWarningReport(folderValidationMessageRepository.findValidationMessagesByFolderId(folderId))
  }

  def setValidationMessages(folderId: String, validationMessages: Seq[ValidationReportMessage]): Unit = {
    folderValidationMessageRepository.findValidationMessagesByFolderId(folderId).foreach(message => folderValidationMessageRepository.deleteValidationMessage(message.messageId))
    validationMessages.foreach(message => folderValidationMessageRepository.createValidationMessageEntity(folderId, message))
  }

  def updateValidationMessageStatus(messageId: Integer, validationMessage: ValidationReportMessage): Unit = {
    folderValidationMessageRepository.updateValidationMessageResolvedState(messageId, validationMessage)
  }

  def initConnector(config: FolderVersioningSettings): AsCodeJGitConnector = {
    val connectorConfig = config.gitConnection
    configurationVariableService.resolve(connectorConfig)

    new AsCodeJGitConnector(config)
  }

  private def disableTriggers(folderId: String): Seq[Trigger] = {
    logger.warn(s"Disabling all enabled triggers in folder $folderId before changing folder contents")
    val triggers = triggerRepository.findByFolderId(folderId, nestedFolders = true, PageRequest.of(0, Int.MaxValue))
      .getContent
      .asScala
      .filter(trigger => trigger.isEnabled)
      .toSeq


    triggers.foreach(trigger => {
      logger.warn(s"Disabling trigger title=[${trigger.getTitle}] id=[${trigger.getId}]")
      triggerService.updateTriggerStatus(trigger.getId, enabled = false, checkReferencePermissions = false)
    })

    triggers
  }

  private[scm] def applyDefinitionsInTransaction(includes: ImportIncludes,
                                                 folderId: String,
                                                 definitions: Seq[Definition],
                                                 scmData: Option[SCMTraceabilityData] = None,
                                                 gitRepo: Option[Repository] = None,
                                                 validator: Option[ImportValidator] = None): ImportResult = {


    val importResult = executeTransaction(includes, folderId, definitions, scmData, gitRepo, validator)
    importResult.postCommitActions.foreach(action => action.run()) // TODO: should we wrap the run calls in try/catch log failures and continue?
    importResult
  }

  private def executeTransaction(includes: ImportIncludes,
                                 folderId: String,
                                 definitions: Seq[Definition],
                                 scmData: Option[SCMTraceabilityData] = None,
                                 gitRepo: Option[Repository],
                                 validator: Option[ImportValidator] = None): ImportResult = {

    val disabledTriggers = if (includes.triggers) disableTriggers(folderId) else Seq.empty[Trigger]

    try {
      transactionTemplate.execute { _ =>
        val ir = definitions.foldLeft(ImportResult.empty) { case (acc, definition) =>
          val spec = gitRepo match {
            case Some(repo) =>
              definition.spec match {
                case ciSpec: CiSpec =>
                  val cisExcludingVersioningGitRepo = ciSpec.cis.filterNot(config => config.isInstanceOf[Repository]
                    && config.asInstanceOf[Repository].getTitle.equals(repo.getTitle))
                  CiSpec(cisExcludingVersioningGitRepo, ciSpec.references, ciSpec.files)
                case _ =>
                  definition.spec
              }
            case None => definition.spec
          }

          val updatedDefinition = Definition(definition.apiVersion, metadataMap(folderId), definition.kind, spec)
          acc.merge(importService.process(includes, updatedDefinition, scmData, validator))
        }
        val GitRepoConfigId = gitRepo match {
          case Some(repo) => Some(repo.getId)
          case None => None
        }
        val processedCis = orphanDeletionService.processCis(includes, folderId, ir.changedIdsPerTypePerFolder, GitRepoConfigId)
        val processedPermissions = orphanDeletionService.processPermissions(includes, folderId, ir.changedIds)

        val fullIr = ir.merge(processedCis).merge(processedPermissions)
        fullIr
      }
    } catch {
      case e: Throwable =>
        // Transaction was rolled back, re-enable the triggers that were disabled before starting the transaction
        logger.warn(s"Apply failed and changes were rolled back in ${folderId} re-enabling the triggers that were disabled before attempting changes")

        disabledTriggers.foreach(trigger => {
          logger.warn(s"Enabling trigger title=[${trigger.getTitle}] id=[${trigger.getId}]")
          triggerService.updateTriggerStatus(trigger.getId, enabled = true, checkReferencePermissions = false)
        })
        throw e
    }
  }

  private def preprocessYaml(yamlStr: String): List[String] = {
    // TODO add tag handling during definition parsing to avoid additional parsing of yaml and converting back
    val processedYamlStrs = ListBuffer.empty[String]
    val yamlParser = new Yaml(new ValueTagPreprocessingConstructor)
    val data = yamlParser.loadAll(yamlStr)
    data.forEach { spec =>
      if (spec != null) {
        processedYamlStrs.addOne(yamlParser.dump(spec))
      }
    }

    processedYamlStrs.toList
  }

  private case class GeneratedFolder(blob: ScmBlobs, definitions: List[Definition], secrets: Map[String, String])

  private def generateDefinitions(folderId: String, config: FolderVersioningSettings): List[Definition] = {
    val cisConfig = CisConfig(
      generateConfigurations = config.exportConfiguration,
      generateDashboards = config.exportDashboards,
      generateTemplates = config.exportTemplates,
      generateTriggers = config.exportTriggers,
      generateVariables = config.exportVariables,
      generateDeliveryPatterns = config.exportPatterns,
      generateNotificationSettings = config.exportNotifications,
      generatePermissions = config.exportSecurity
    )

    try {
      validateGeneratePermissions(folderId)
      val scope = FolderSearch(folderAsCodeService.getFolderPath(folderId), folderId)
      val excludeIds = Seq(config.gitConnection.getId)
      generateService.generate(GeneratorConfig(None, scope, cisConfig, permissions.hasGlobalPermission(ADMIN), excludedEntities = excludeIds))
    } catch {
      case e: AsCodeException =>
        e.getMessage match {
          case GenerateService.EMPTY_DEFINITION_ERROR =>
            List(Definition(XLR_API_VERSION,
              Option(Map(MetadataFields.HOMEFOLDER.toString -> FolderSearch(folderAsCodeService.getFolderPath(folderId), folderId).path)),
              XLR_TEMPLATE_KIND,
              CiSpec(List.empty, List.empty, Map.empty)))
          case _ => throw ScmException(e.getMessage, e.getCause)
        }
    }
  }

  private def generateFolder(folderId: String, config: FolderVersioningSettings, filePath: String): GeneratedFolder = {
    try {
      val definitions = generateDefinitions(folderId, config)

      val stream = new ByteArrayOutputStream()
      val generateCtxs = XLRDefinitionWriter().writeYaml(stream, WriterConfig(definitions, writeSecrets = false, writeDefaults = false))
      val secrets = generateCtxs.flatMap(_.secrets).toMap

      GeneratedFolder(
        ScmBlobs(Seq(BinaryFile(filePath, () => stream.toByteArray))),
        definitions,
        secrets
      )
    } catch {
      case e: AsCodeException => throw ScmException(e.getMessage, e.getCause)
    }
  }

  private def validateSettings(config: FolderVersioningSettings, remoteValidations: Boolean): Unit = {
    folderVersioningConfigService.validateConfigData(config, resolveFolderTitleFromId(config.getFolderId))
    Using.resource(initConnector(config)) { connector =>
      val url = connector.getConnectionSettings.getUrl
      if (!isFileRepo(url)) {
        verifyRepositoryPath(url)
        if (remoteValidations) {
          connector.verifyRemoteBranch(url, config.branch)
        }
      }
    }
  }

  private def metadataMap(folderId: String): Option[Map[String, String]] = {
    Some(Map(
      MetadataFields.FOLDER.toString -> folderId,
      MetadataFields.HOMEFOLDER.toString -> folderAsCodeService.getFolderPath(folderId)
    ))
  }

  private def markAsApplied(version: String, config: FolderVersioningSettings): Unit = {
    val now = new Date
    config.lastFetched = now
    config.appliedVersion = version
    config.appliedDate = now
    config.appliedBy = getAuthenticatedUserName
  }

  private def applyScmDataToDefinitions(definitions: Seq[Definition], scmData: SCMTraceabilityData): Unit = {
    def flattenCis(cis: Seq[ConfigurationItem]): Seq[ConfigurationItem] = cis.flatMap {
      case folder: Folder => flattenCis(folder.getChildren.asScala.toSeq)
      case ci => Seq(ci)
    }

    val foldersAndCis = definitions.withFilter(_.spec.isInstanceOf[CiSpec]).flatMap(_.spec.asInstanceOf[CiSpec].cis)
    val justCis = flattenCis(foldersAndCis)
    justCis.foreach {
      // only templates support scmData currently
      case t: Release => releaseService.createSCMData(t.getId, scmData)
      case _ => ()
    }
  }

  private def getScmTraceabilityData(config: FolderVersioningSettings, versionInfo: VersionInfo): SCMTraceabilityData = {
    new SCMTraceabilityData(
      "git",
      s"${versionInfo.commitHash}",
      versionInfo.author,
      new DateTime(versionInfo.commitTime),
      s"[${versionInfo.name}] ${versionInfo.shortMessage}",
      config.gitConnection.getUrl,
      getDefinitionsPath(config))
  }

  private def validateGeneratePermissions(folderId: String): Unit = {
    if (!(permissions.hasPermission(GENERATE_FOLDER_CONFIGURATION, folderId) || permissions.isCurrentUserAdmin)) {
      throw ScmException(s"Unable to generate configuration: must have Admin or 'Generate Configuration' permission on the folder")
    }
  }

  private def resolveFolderTitleFromId(folderId: String): String = {
    Try(folderService.findById(folderId))
      .map(_.getTitle)
      .getOrElse("")
  }
}
