package com.xebialabs.xlrelease.ascode.service.validator

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.utils.{Utils => CommonUtils}
import com.xebialabs.ascode.yaml.model.{CiSpec, Definition}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.ascode.metadata.MetadataFields
import com.xebialabs.xlrelease.ascode.utils.Utils
import com.xebialabs.ascode.utils.Utils.splitStringByPathSeparator
import com.xebialabs.xlrelease.ascode.service.FolderAsCodeService
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.service.ConfigurationService
import org.springframework.stereotype.Component
import com.xebialabs.xlrelease.repository.Ids.SEPARATOR

import java.util.{Set => JavaSet}
import scala.jdk.CollectionConverters._

@Component
class CiSpecValidator(configurationService: ConfigurationService, folderAsCodeService: FolderAsCodeService) extends SpecValidator {

  val TYPE_LIMITS = "xlrelease.Limits"
  val MAX_FOLDER_DEPTH = "maxFolderDepth"


  private def allowedFolderDepth: Int = configurationService.getFeatureSettings(TYPE_LIMITS).getProperty(MAX_FOLDER_DEPTH)

  override def isDefinedAt(definition: Definition): Boolean = definition.spec.isInstanceOf[CiSpec]

  private case class CiHelper(id: String, ci: ConfigurationItem)

  private def parseCis(cis: List[ConfigurationItem], parent: String = ""): List[CiHelper] = {
    val (folderCis, otherCis) = cis.partition(_.isInstanceOf[Folder])

    val folders = checkFolders(folderCis.asInstanceOf[List[Folder]], parent)

    folders ::: otherCis.map { ci =>
      val title = Utils.getCiTitle(ci).getOrElse("")
      val id = s"${ci.getType.toString}-${Utils.buildFolderPath(parent, title)}"

      CiHelper(id, ci)
    }
  }

  private def checkFolders(folderCis: List[Folder], parent: String): List[CiHelper] = {
    folderCis.flatMap {
      folder =>
        val folderId = s"Folder-$parent${SEPARATOR}${folder.getTitle}"
        val (folderChildren, otherChildren) = folder.getChildren.asInstanceOf[JavaSet[ConfigurationItem]].asScala.toList.partition(_.isInstanceOf[Folder])
        val folderPath = Utils.buildFolderPath(parent, folder.getTitle)

        val folderChildrenList = checkFolders(folderChildren.asInstanceOf[List[Folder]], folderPath)
        val otherChildrenList = parseCis(otherChildren, folderPath)

        CiHelper(folderId, folder) :: otherChildrenList ::: folderChildrenList
    }
  }

  private def checkFolderDepth(processedCis: List[CiHelper], homeDirectory: Option[String], homeFolderDepth: Int): Unit = {
    val calculatedMaxFolderDepth = allowedFolderDepth - (if (homeFolderDepth > 0) homeFolderDepth - 1 else 0)
    val folderList = processedCis.filter(f => f.ci.isInstanceOf[com.xebialabs.xlrelease.domain.folder.Folder])
      .filter(_.id.split(SEPARATOR).length > calculatedMaxFolderDepth)

    // For each path over allowed depth fail if it does not already exist
    folderList.foreach(folder => {
      val folderPath = homeDirectory match {
        case Some(home) => home + SEPARATOR + folder.id.stripPrefix("Folder-").stripPrefix(SEPARATOR)
        case None => folder.id.stripPrefix("Folder-")
      }
      folderAsCodeService.searchFolder(folderPath, false, false) match {
        case Some(_) => //folder already exists, allow it to exceed folder depth limit
        case None =>
          val errorMessage = s"Creating Folder with path ${folderPath} exceeds allowed sub-folder depth ${allowedFolderDepth}."
          throw new AsCodeException(s"$errorMessage")
      }
    })
  }

  private def checkDupe(processedCis: List[CiHelper]): Unit = {
    val cisList = processedCis.map(_.id)

    val duplicatedCis = CommonUtils.getDuplicatedObjects(cisList)
    if (duplicatedCis.nonEmpty) {
      val filteredCis = processedCis.filter(processedCi => duplicatedCis.contains(processedCi.id)).map(x => x.id -> x.ci).toMap
      val errorMessage = filteredCis.map {
        case (id, ci) =>
          val (ciType, title) = parseIdString(id.substring(id.indexOf("-") + 1, id.length))
          s"""The path [$ciType] with title [$title] is not unique. All ${ci.getType.toString} titles in the template must be unique."""
      }.mkString("\n")
      throw new AsCodeException(s"$errorMessage")
    }
  }

  private def parseIdString(id: String): (String, String) = {
    if (id.contains(SEPARATOR)) {
      (id.substring(0, id.lastIndexOf(SEPARATOR)), id.substring(id.lastIndexOf(SEPARATOR) + 1, id.length))
    } else {
      ("", id)
    }
  }

  private def calculateHomeFolderInfo(metadata: Option[Map[String, String]]): (Option[String], Int) = {
    metadata match {
      case Some(meta) if meta != Map.empty =>
        val homeFolder = meta.getOrElse(MetadataFields.HOMEFOLDER.toString, "")
        (Some(homeFolder), splitStringByPathSeparator(homeFolder).length)
      case _ =>
        (None, 0)
    }
  }

  override def apply(definition: Definition): Unit = {
    val ciSpec = definition.spec.asInstanceOf[CiSpec]
    val processedCis = parseCis(ciSpec.cis)
    val (homeFolder, homeLength) = calculateHomeFolderInfo(definition.metadata)
    checkFolderDepth(processedCis, homeFolder, homeLength)
    checkDupe(processedCis)
  }
}
