package com.xebialabs.xlrelease.versioning.ascode.scm.strategy

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.model.{CiSpec, Definition}
import com.xebialabs.ascode.yaml.writer.DefinitionWriter.WriterConfig
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.ascode.utils.StaticVariables.XLR_TEMPLATE_KIND
import com.xebialabs.xlrelease.ascode.yaml.model.ImportsSpec
import com.xebialabs.xlrelease.ascode.yaml.parser.XLRDefinitionParser
import com.xebialabs.xlrelease.ascode.yaml.writer.XLRDefinitionWriter
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings.{FOLDER_VERSIONING_YAML_FILENAME, getDefinitionsPath}
import com.xebialabs.xlrelease.scm.connector.{BinaryFile, ScmBlobs, ScmException}
import com.xebialabs.xlrelease.service.WorkdirSupport
import com.xebialabs.xlrelease.versioning.ascode.scm.YamlUtils
import com.xebialabs.xlrelease.versioning.ascode.scm.connector.AsCodeJGitConnector
import com.xebialabs.xltype.serialization.CiReference
import org.apache.commons.io.IOUtils

import java.io.ByteArrayOutputStream
import java.nio.charset.StandardCharsets
import scala.collection.mutable.ListBuffer
import scala.util.Using

trait BaseMultiFileStrategy extends VersioningStyleStrategy with WorkdirSupport {

  def generateMultiFileDefinitions(folderId: String, settings: FolderVersioningSettings): Map[String, List[Definition]]

  def definitionParser(): XLRDefinitionParser

  override def generateFolder(folderId: String, settings: FolderVersioningSettings): FolderContent = {
    try {
      val definitions = generateMultiFileDefinitions(folderId, settings)

      val generatedMultiFileContents = definitions.foldLeft(List[FolderContent]())((accumulator, fileDefinitions) => {
        val stream = new ByteArrayOutputStream()
        val generateCtxs = XLRDefinitionWriter().writeYaml(stream, WriterConfig(fileDefinitions._2, writeSecrets = false, writeDefaults = false))
        val secrets = generateCtxs.flatMap(_.secrets).toMap
        val additionalFiles = generateCtxs.flatMap(_.additionalFile)
          .map(af =>
            BinaryFile(
              FolderVersioningSettings.getDefinitionsPath(settings.scmPath, af.name),
              () => {
              withWorkDir {
                Using.resource(af.file.getInputStream) { is =>
                  IOUtils.toByteArray(is)
                }
              }
            }))
        val filesToAdd = Seq(BinaryFile(fileDefinitions._1, () => stream.toByteArray)) ++ additionalFiles
        accumulator :+ FolderContent(
          ScmBlobs(filesToAdd),
          fileDefinitions._2,
          secrets
        )
      })

      val allBinaryFiles = generatedMultiFileContents.foldLeft(Seq[BinaryFile]())((accumulator, generatedFolder) => {
        accumulator ++ generatedFolder.blob.filesToAdd
      })

      val allsecrets = generatedMultiFileContents.foldLeft(Map[String, String]())((accumulator, generatedFolder) => {
        accumulator ++ generatedFolder.secrets
      })

      val allFileDefinitions = generatedMultiFileContents.foldLeft(List[Definition]())((accumulator, generatedFolder) => {
        accumulator ++ generatedFolder.definitions
      })

      FolderContent(ScmBlobs(allBinaryFiles), allFileDefinitions, allsecrets)
    } catch {
      case e: AsCodeException => throw ScmException(e.getMessage, e.getCause)
    }
  }

  override def fetchChildDefinitionsIfNecessary(version: String,
                                                scmPath: String,
                                                scmConnector: AsCodeJGitConnector,
                                                definitionsInReleaseFile: List[Definition]): List[Definition] = {
    val filesToBeImported = definitionsInReleaseFile.head.spec.asInstanceOf[ImportsSpec].imports.map(getDefinitionsPath(scmPath, _))
    val importedDefinitions = ListBuffer.empty[Definition]
    if (filesToBeImported.nonEmpty) {
      scmConnector.checkout(filePath = Array(filesToBeImported: _*), version).filesToAdd.foreach(file => {
        val yamlStr = new String(file.getContent(), StandardCharsets.UTF_8)
        val specs = YamlUtils.preprocessYaml(yamlStr)
        importedDefinitions.addAll(specs.map(definitionParser.parse))
      })
      // Construct a single CIspec for Template Kind specs, only two specs possible here Template and Permissions
      importedDefinitions.groupBy(defition => defition.kind).foldLeft(List[Definition]()) {
        case (accumulator, (kind, definitions)) =>
          if (XLR_TEMPLATE_KIND.equals(kind)) {
            val apiDefinition = definitions.head.apiVersion
            val metaData = definitions.head.metadata
            val ciSpecs = definitions.map(_.spec.asInstanceOf[CiSpec])
            val ciSpec = ciSpecs.foldLeft(CiSpec(List[ConfigurationItem](), List[CiReference]())) { case (accumulator, ciSpec) =>
              CiSpec(accumulator.cis ++ ciSpec.cis, accumulator.references ++ ciSpec.references, accumulator.files ++ ciSpec.files)
            }
            accumulator :+ Definition(apiDefinition, metaData, kind, ciSpec)
          } else {
            accumulator :+ definitions.head
          }
      }
    } else {
      importedDefinitions.toList
    }
  }

  override def getDefinitionsFilenamesFrom(definition: Definition): List[String] = List(FOLDER_VERSIONING_YAML_FILENAME) ++ definition.spec.asInstanceOf[ImportsSpec].imports

  override def getVersionableFileNames(folderId: String, settings: FolderVersioningSettings): List[String] = {
    val releaseFileName = getDefinitionsPath(settings.scmPath)
    val importSpecDefinition = generateMultiFileDefinitions(folderId, settings)(releaseFileName).head
    List(FOLDER_VERSIONING_YAML_FILENAME) ++ importSpecDefinition.spec.asInstanceOf[ImportsSpec].imports
  }
}
