package com.xebialabs.xlrelease.versioning.scheduler

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings
import com.xebialabs.xlrelease.quartz.release.scheduler.ReleaseSchedulerService
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.versioning.scheduler.FolderVersioningAutoApplyJobService._
import grizzled.slf4j.Logging
import org.quartz.JobBuilder.newJob
import org.quartz.SimpleScheduleBuilder.simpleSchedule
import org.quartz.TriggerBuilder.newTrigger
import org.quartz.{JobDataMap, JobDetail, JobKey, SchedulerException}
import org.springframework.retry.RetryCallback
import org.springframework.retry.support.RetryTemplateBuilder
import org.springframework.stereotype.Service

import java.time.Instant
import java.time.temporal.ChronoUnit.HOURS
import java.util.Date
import java.util.concurrent.{ThreadLocalRandom, TimeUnit}
import scala.concurrent.duration.FiniteDuration
import scala.jdk.CollectionConverters._

@Service
class FolderVersioningAutoApplyJobService(releaseSchedulerService: ReleaseSchedulerService) extends Logging {

  private val retryTemplate = new RetryTemplateBuilder().fixedBackoff(250).maxAttempts(8).build()

  private val folderVersioningSettingsType = Type.valueOf(classOf[FolderVersioningSettings]).toString

  def handleAutoApplyGitVersion(folderVersioningSettings: FolderVersioningSettings): Unit = {
    if (folderVersioningSettings.autoImport) {
      schedule(folderVersioningSettings)
    } else {
      unschedule(folderVersioningSettings.getFolderId)
    }
  }

  def unschedule(folderId: String): Unit = {
    val jobKey = new JobKey(Ids.getName(folderId), folderVersioningSettingsType)
    val retryCallback: RetryCallback[Boolean, SchedulerException] = _ => releaseSchedulerService.unschedule(jobKey)
    retryTemplate.execute(retryCallback)
    logger.debug(s"Unscheduled a job associated with folder [$folderId] to auto apply new changes with folder versioning")
  }

  private def schedule(folderVersioningSettings: FolderVersioningSettings): Unit = {
    val identity = Ids.getName(folderVersioningSettings.getFolderId)
    // If duration is ever changed, do required changes in getRandomTriggerStartDate method as well
    val duration = FiniteDuration.apply(12, TimeUnit.HOURS)

    val job: JobDetail = newJob(classOf[FolderVersioningAutoApplyQuartzJob])
      .withDescription(s"Job to auto apply new changes with folder versioning")
      .withIdentity(identity, folderVersioningSettingsType)
      .storeDurably(true)
      .setJobData(new JobDataMap(Map[String, String](AUTO_APPLY_JOB_DATA_KEY -> folderVersioningSettings.getFolderId).asJava))
      .build()

    val quartzTrigger = newTrigger()
      .withDescription(s"Trigger to auto apply new changes with folder versioning")
      .withIdentity(identity, folderVersioningSettingsType)
      .withSchedule(
        simpleSchedule()
          .withIntervalInMilliseconds(duration.toMillis)
          .repeatForever()
          .withMisfireHandlingInstructionNextWithRemainingCount() // In case of misfire, trigger scheduled time should not change
      )
      .startAt(getRandomTriggerStartDate())
      .build()

    releaseSchedulerService.schedule(job, quartzTrigger)

    logger.debug(s"Scheduled a job associated with folder [$identity] to auto apply new changes with folder versioning")
  }

  /**
   * Returns random start date for trigger between next 1 hr and 12 hrs
   * This is to avoid potential having the same start time for multiple triggers and firing at the same time
   */
  private def getRandomTriggerStartDate(): Date = {
    val now = Instant.now().plus(1, HOURS).toEpochMilli
    val end = Instant.now().plus(12, HOURS).toEpochMilli
    val randomMillisSinceEpoch = ThreadLocalRandom.current().nextLong(now, end)
    val randomInstant = Instant.ofEpochMilli(randomMillisSinceEpoch)
    Date.from(randomInstant)
  }

}

object FolderVersioningAutoApplyJobService {
  val AUTO_APPLY_JOB_DATA_KEY: String = "folderId"
}
