package com.xebialabs.xlrelease.ascode.service.referencehandler.stringhandler

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.utils.Utils
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor
import com.xebialabs.xlrelease.api.v1.filter.{ApplicationFilters, EnvironmentFilters}
import com.xebialabs.xlrelease.ascode.yaml.sugar.XLRSugar
import com.xebialabs.xlrelease.environments.service.{ApplicationService, EnvironmentService}
import com.xebialabs.xlrelease.udm.reporting.DeploymentTaskFacet
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

object DeploymentTaskFacetStringCreationHandler {
  val environmentIdField = "environmentId"
  val applicationIdField = "applicationId"
  val properties: Set[String] = Set(
    environmentIdField,
    applicationIdField
  )
}

@Autowired
@Component
class DeploymentTaskFacetStringCreationHandler(environmentService: EnvironmentService,
                                               applicationService: ApplicationService) extends StringReferenceCreationHandler {

  import DeploymentTaskFacetStringCreationHandler._

  override def isDefinedAt(config: StringReferenceCreationConfig): Boolean = config.ci.isInstanceOf[DeploymentTaskFacet]

  private def resolveAppId(app: String): Option[String] = {
    val appFilters = new ApplicationFilters
    appFilters.setTitle(app)
    applicationService.searchApplications(appFilters).asScala.find(_.getTitle.equals(app)).map(_.getId)
  }

  private def resolveEnvId(env: String): Option[String] = {
    val envFilters = new EnvironmentFilters
    envFilters.setTitle(env)
    environmentService.searchEnvironments(envFilters).asScala.find(_.getTitle.equals(env)).map(_.getId)
  }

  private def handle(config: StringReferenceCreationConfig, descriptor: PropertyDescriptor, resolveId: String => Option[String]): Option[String] = {
    val facet = config.ci.asInstanceOf[DeploymentTaskFacet]
    val property = facet.getProperty[String](descriptor.getName)
    val ciType = config.ci.getType

    property match {
      case null => None
      case "" => throw new AsCodeException(
        s"Field ${Utils.getSugaredField(descriptor, ciType, XLRSugar.config)} is empty, in CI with type ${ciType.toString} please fill in the field.")
      case _ =>
        val id = resolveId(property)
        facet.setProperty[String](descriptor.getName, id.orNull)
        id
    }
  }

  override def apply(config: StringReferenceCreationConfig): StringReferenceContainer = {
    val envIdDescriptor = config.ci.getType.getDescriptor.getPropertyDescriptor(environmentIdField)
    val appIdDescriptor = config.ci.getType.getDescriptor.getPropertyDescriptor(applicationIdField)
    val envId = handle(config, envIdDescriptor, resolveEnvId)
    val appId = handle(config, appIdDescriptor, resolveAppId)

    val envRef = envId.map(env => Map(envIdDescriptor -> Some(env))).getOrElse(Map.empty)
    val appRef = appId.map(app => Map(appIdDescriptor -> Some(app))).getOrElse(Map.empty)

    StringReferenceContainer(config.ci, envRef ++ appRef)
  }
}
