package com.xebialabs.xlrelease.ascode.service.generatestrategy

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.ascode.service.GenerateService.GeneratorConfig
import com.xebialabs.xlrelease.ascode.service.{FolderAsCodeService, FolderSearch, ReferenceSolver}
import com.xebialabs.xlrelease.ascode.utils.Utils
import com.xebialabs.xlrelease.ascode.utils.Utils.{ciSortOrderValue, getCiTitle}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.repository.FolderRepository
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.XLReleasePermissions.GENERATE_FOLDER_CONFIGURATION
import com.xebialabs.xlrelease.utils.CiHelper
import com.xebialabs.xltype.serialization.CiReference
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util
import java.util.{List => JavaList}
import scala.collection.immutable.{ListMap, ListSet}
import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._
import scala.util.Try
import scala.util.control.Breaks.{break, breakable}

object CiGenerator {
  val GENERATE_CI_LIMIT_XLR = "xl.devops-as-code.ci-limit"
  val DEFAULT_CI_LIMIT = 250L
  val MAX_PAGES = 10000
  val MaxCiCount: Long = readMaxCiCount()

  def paginate[T](fn: Long => (Seq[T], Boolean)): List[T] = {
    var page = 0L
    val combineListBuffer: ListBuffer[T] = new ListBuffer[T]
    breakable {
      while (page <= MAX_PAGES) {
        val pageItems = fn(page)
        combineListBuffer ++= pageItems._1
        if (pageItems._2 || combineListBuffer.size >= MaxCiCount) {
          break()
        }
        page += 1
      }
    }
    combineListBuffer.toList.slice(0, MaxCiCount.toInt)
  }

  private def readMaxCiCount(): Long = {
    if (XlrConfig.getInstance.rootConfig.hasPath(GENERATE_CI_LIMIT_XLR)) {
      Try(XlrConfig.getInstance.rootConfig.getLong(GENERATE_CI_LIMIT_XLR)).getOrElse(DEFAULT_CI_LIMIT)
    } else {
      DEFAULT_CI_LIMIT
    }
  }
}

@Service
class CiGenerator @Autowired()(strategies: JavaList[GenerateStrategy[_]],
                               folderGenerator: FolderGenerator,
                               folderAsCodeService: FolderAsCodeService,
                               referenceSolver: ReferenceSolver,
                               folderRepository: FolderRepository,
                               permissionChecker: PermissionChecker) extends Logging {

  import CiGenerator._

  private val sortedStrategies = strategies.asScala.sortWith(_.generateOrder < _.generateOrder)

  def generateCis(generatorConfig: GeneratorConfig): (List[ConfigurationItem], List[CiReference]) = {
    debug(s"Exporting ${generatorConfig.searchScope} and ${generatorConfig.name}. Max CI Limit: $MaxCiCount")
    val folders = folderGenerator.getFolderCis(generatorConfig)
    processGenerateFolderConfigPermissions(folders, generatorConfig)
    val generatedCis = generateCisWithRefs(generatorConfig, folders)
    val globalCis = generatedCis.getOrElse(None, CisWithRefs())
    val folderCis = generatedCis.filter(_._1.isDefined)
    val processedCis = folderCis.foldLeft(globalCis) {
      case (acc, (key, value)) =>
        val folder = key.get
        validateDuplicatedNames(value.cis, folder)
        includeCisInFolder(folder, value.cis.sortBy(ci => (ci.getType.toString, getCiTitle(ci))).sortWith(ciSortOrderValue(_) < ciSortOrderValue(_)))
        acc.merge(CisWithRefs(List(key.get), value.references))
    }

    val childrenFilledFolder = if (generatorConfig.cisConfig.generatePermissions) folders else fillChildren(folders)

    (globalCis.cis.sortBy(gci => (gci.getType.toString, getCiTitle(gci))).sortWith(ciSortOrderValue(_) < ciSortOrderValue(_)) ++ childrenFilledFolder, processedCis.references)
  }

  private def validateDuplicatedNames(cis: List[ConfigurationItem], folder: Folder): Unit = {
    var names: Set[(String, Type)] = Set.empty
    cis.foreach { ci =>
      Utils.getCiTitle(ci).foreach { ciTitle =>
        val values = (ciTitle, ci.getType)
        if (!names.contains(values)) {
          names += values
        } else {
          throw new AsCodeException(s"Found a duplicate ci with name $ciTitle while generating folder ${folder.getTitle}. Make sure there are cis with unique names.")
        }
      }
    }
  }

  private def includeCisInFolder[T <: ConfigurationItem](folder: Folder, cisInside: List[T]): Unit = {
    val childrenFilled = ListSet(folder.getChildren.asScala.toList ++ cisInside: _*).asJava
    setChildren(folder, childrenFilled)
  }

  private def getTitle(ci: ConfigurationItem): String = Try(Utils.getCiTitle(ci).getOrElse("")).getOrElse("")

  private def getHome(generatorConfig: GeneratorConfig): Option[String] = {
    generatorConfig.searchScope match {
      case FolderSearch(path, _, _) => Some(path)
      case _ => None
    }
  }

  private def generateCisWithRefs(generatorConfig: GeneratorConfig, nestedFolders: List[Folder]): ListMap[Option[Folder], CisWithRefs] = {
    val folders = CiHelper.getNestedCis(nestedFolders.asJava).asScala.toList.asInstanceOf[List[Folder]]
    sortedStrategies.foldLeft(ListMap.empty[Option[Folder], CisWithRefs]) { (strategyAccumulator, strategy) =>
      val config = CiGenerateConfig(generatorConfig.name, generatorConfig.searchScope, folders, generatorConfig.cisConfig, generatorConfig.excludedEntities,
        generatorConfig.isAdmin, generatorConfig.folderGeneratePermissions.toMap)
      val generatedCis = if (strategy.isDefinedAt(config)) strategy(config) else ListMap.empty[Option[Folder], List[ConfigurationItem]]
      val (folderCisWithRefs, _) = generatedCis.foldLeft((ListMap.empty[Option[Folder], CisWithRefs], 0)) {
        case ((generatedCisAccumulator, ciCounter), (folder, cis)) =>
          val cisWithRefs = cis.foldLeft(CisWithRefs()) { (cisAccumulator, ci) =>
            try {
              val references = CiHelper.getNestedCis(ci).asScala.toList.flatMap(referenceSolver.obtainReferencesFromCi(_, getHome(generatorConfig)))
              cisAccumulator.merge(CisWithRefs(List(ci), references))
            } catch {
              case e: Exception =>
                val dashboardString = s"Exception when exporting ${getTitle(ci)}[${ci.getId}]"
                val folderPathString = folder.fold("") { f =>
                  val id = f.getId
                  val path = Try(Utils.joinPaths(folderRepository.getPath(id).tail)).getOrElse("")
                  s": folder path [$path] [$id]"
                }
                throw new AsCodeException(s"$dashboardString $folderPathString : ${e.getMessage}")
            }
          }
          val generatedCisCounter = cisWithRefs.cis.size + ciCounter

          if (generatedCisCounter > MaxCiCount) {
            throw new AsCodeException(s"Export contains more than $MaxCiCount items. Please narrow down the export, for example by specifying a more specific folder.")
          }

          val processedCis = generatedCisAccumulator ++ ListMap(folder -> cisWithRefs)

          (processedCis, generatedCisCounter)
      }

      joinMapOfFolderCisRefs(strategyAccumulator, folderCisWithRefs)
    }
  }

  private def joinMapOfFolderCisRefs(map1: ListMap[Option[Folder], CisWithRefs],
                                     map2: ListMap[Option[Folder], CisWithRefs]): ListMap[Option[Folder], CisWithRefs] = {
    map1 ++ map2.map {
      case (key, value) => key -> map1.get(key).map(_.merge(value)).getOrElse(value)
    }
  }

  private def fillChildren(items: List[ConfigurationItem]): List[ConfigurationItem] = items.flatMap {
    case folder: Folder =>
      val children = fillChildren(folder.getChildren.asScala.toList).toSet
      if (children.nonEmpty) {
        setChildren(folder, ListSet(children.toList.sortBy(ci => (ci.getType.toString, getCiTitle(ci))).sortWith(ciSortOrderValue(_) < ciSortOrderValue(_)): _*).asJava)
        Some(folder)
      } else {
        Some(folder)
      }
    case ci => Some(ci)
  }

  // hack, big one, wasn't me
  private def setChildren(folder: Folder, children: util.Set[_ <: ConfigurationItem]): Unit = {
    val descriptor = folder.getType.getDescriptor.getPropertyDescriptor("children")
    descriptor.set(folder, children)
  }

  private def processGenerateFolderConfigPermissions(folders: List[Folder], generatorConfig: GeneratorConfig): Unit = {
    folders.map { folder =>
      processGenerateFolderConfigPermissions(folder.getChildren.asScala.toList, generatorConfig)
      generatorConfig.folderGeneratePermissions += (folder.getId -> permissionChecker.hasPermission(GENERATE_FOLDER_CONFIGURATION, folder.getId))
    }
  }
}
