package com.xebialabs.xlrelease.versioning.ascode.upgrader

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.Team
import com.xebialabs.xlrelease.domain.Team.{FOLDER_OWNER_TEAMNAME, RELEASE_ADMIN_TEAMNAME, TEMPLATE_OWNER_TEAMNAME}
import com.xebialabs.xlrelease.domain.configuration.HttpConnection
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings
import com.xebialabs.xlrelease.scm.connector.Repository
import com.xebialabs.xlrelease.security.XLReleasePermissions._
import com.xebialabs.xlrelease.service.{FolderService, SharedConfigurationService}
import com.xebialabs.xlrelease.upgrade.common.BaseInitializingUpgrade
import com.xebialabs.xlrelease.versioning.ascode.scm.strategy.MixedMultiFileStrategy.MULTIPLE_FILES_PER_CI_TYPE
import com.xebialabs.xlrelease.versioning.ascode.scm.{FolderVersionApplyHelper, FolderVersioningService}
import com.xebialabs.xlrelease.versioning.ascode.upgrader.BaseContentFolderInitializer._
import com.xebialabs.xlrelease.versioning.scheduler.FolderVersioningAutoApplyJobService
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

abstract class BaseContentFolderInitializer(folderService: FolderService,
                                            val folderVersioningService: FolderVersioningService,
                                            folderVersioningAutoApplyJobService: FolderVersioningAutoApplyJobService,
                                            sharedConfigurationService: SharedConfigurationService,
                                            xlrConfig: XlrConfig)
  extends BaseInitializingUpgrade with FolderVersionApplyHelper with Logging {

  private def scheduleAutoApplyGitVersion(folderId: String, folderName: String): Unit = {
    Try {
      val folderVersioningSettings = folderVersioningService.getSettings(folderId)
      folderVersioningAutoApplyJobService.handleAutoApplyGitVersion(folderVersioningSettings)
    } match {
      case Failure(ex) => logger.error(s"Unable to configure auto apply of folder version for folder '$folderName'", ex)
      case Success(_) => logger.trace(s"Scheduled/unscheduled auto apply of folder version for folder '$folderName")
    }
  }

  def getGitConfiguration(parentId: String): Repository = {
    val configurations = sharedConfigurationService.searchByTypeAndTitle(
      Type.valueOf("git.Repository"),
      xlrConfig.features.provision.folders.defaultContentFolder.repositoryTitle,
      parentId,
      true
    ).asScala

    configurations.headOption
      .getOrElse(throw new NotFoundException("No configuration of type 'git.Repository' found"))
      .asInstanceOf[Repository]
  }

  def createGitConfiguration(parentId: String): Repository = {
    val gitConfiguration = Type.valueOf("git.Repository").getDescriptor.newInstance[Repository]("")
    gitConfiguration.setTitle(xlrConfig.features.provision.folders.defaultContentFolder.repositoryTitle)
    gitConfiguration.setUrl(xlrConfig.features.provision.folders.defaultContentFolder.repositoryUrl)
    gitConfiguration.setAuthenticationMethod(HttpConnection.AuthenticationMethod.None)
    gitConfiguration.setFolderId(parentId)

    val createdGitConfiguration = sharedConfigurationService.create(gitConfiguration)
    createdGitConfiguration.asInstanceOf[Repository]
  }

  def createFolderVersioningSettings(parentId: String,
                                     gitRepository: Repository,
                                     scmBranch: String,
                                     scmPath: String,
                                     autoImport: Boolean): FolderVersioningSettings = {
    val versionConfiguration = Type.valueOf(classOf[FolderVersioningSettings]).getDescriptor.newInstance[FolderVersioningSettings]("")
    versionConfiguration.setFolderId(parentId)
    versionConfiguration.branch = scmBranch
    versionConfiguration.gitConnection = gitRepository
    versionConfiguration.scmPath = scmPath
    versionConfiguration.exportConfiguration = false
    versionConfiguration.exportDashboards = false
    versionConfiguration.exportNotifications = false
    versionConfiguration.exportPatterns = false
    versionConfiguration.exportTemplates = false
    versionConfiguration.exportWorkflows = false
    versionConfiguration.exportTriggers = false
    versionConfiguration.exportVariables = false
    versionConfiguration.exportSecurity = true
    versionConfiguration.versioningStyle = MULTIPLE_FILES_PER_CI_TYPE
    versionConfiguration.autoImport = autoImport

    folderVersioningService.createOrUpdateSettings(versionConfiguration)
  }

  def applyLatestContent(parentFolderId: String, parentFolderTitle: String): Unit = {
    val templatesFetched = fetchAndApplyLatestVersion(parentFolderId, parentFolderTitle)
    scheduleAutoApplyGitVersion(parentFolderId, parentFolderTitle)

    if (templatesFetched) {
      // apply versions for all child folders of templates
      val parentFolder = folderService.findViewableFoldersById(parentFolderId, enforcePermission = false)
      parentFolder.getChildren.asScala.foreach { children =>
        fetchAndApplyLatestVersion(children.getId, children.getTitle)
        scheduleAutoApplyGitVersion(children.getId, children.getTitle)
      }
    }
  }

  def getFolderViewerTeam(): Seq[Team] = {
    val viewer: Team = newTeam(VIEWER_TEAMNAME)
    viewer.getPermissions.add(VIEW_FOLDER.getPermissionName)
    viewer.addRole(GLOBAL_AUTHENTICATED_USERS_ROLE)

    getSystemTeams() :+ viewer
  }

  def getWorkflowAdminTeam(): Seq[Team] = {
    val workflowAdmin: Team = newTeam(WORKFLOW_ADMIN_TEAMNAME)
    workflowAdmin.getPermissions.add(VIEW_FOLDER.getPermissionName)
    workflowAdmin.getPermissions.add(START_WORKFLOW_EXECUTION.getPermissionName)
    workflowAdmin.addRole(GLOBAL_AUTHENTICATED_USERS_ROLE)
    getSystemTeams() :+ workflowAdmin
  }

  def newTeam(teamName: String): Team = {
    val team: Team = Type.valueOf(classOf[Team]).getDescriptor.newInstance(null)
    team.setTeamName(teamName)
    team
  }

  def getSystemTeams(): Seq[Team] = {
    val folderOwner = newTeam(FOLDER_OWNER_TEAMNAME)
    val templateOwner = newTeam(TEMPLATE_OWNER_TEAMNAME)
    val releaseAdmin = newTeam(RELEASE_ADMIN_TEAMNAME)

    Seq(folderOwner, templateOwner, releaseAdmin)
  }
}

object BaseContentFolderInitializer {
  val DEFAULT_CONTENT_FOLDER_ID = s"${Folder.ROOT_FOLDER_ID}/FolderDefaultReleaseContent"
  val TEMPLATES_FOLDER_NAME = "Templates"
  val WORKFLOWS_FOLDER_NAME = "Workflows"
  val WORKFLOW_EXECUTIONS_FOLDER_NAME = "Workflow Executions"
  val WORKFLOW_ADMIN_TEAMNAME = "Workflow Admin"
  val VIEWER_TEAMNAME = "Viewer"
}
