package com.xebialabs.xlrelease.ascode.service.spec

import com.xebialabs.ascode.service.spec.{InterpreterContext, SpecInterpreter}
import com.xebialabs.ascode.yaml.dto.AsCodeResponse
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds._
import com.xebialabs.ascode.yaml.model.Definition
import com.xebialabs.ascode.yaml.model.permission.PermissionsSpec
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import com.xebialabs.xlrelease.ascode.metadata.MetadataFields
import com.xebialabs.xlrelease.ascode.service._
import com.xebialabs.xlrelease.versioning.ascode.ImportValidator
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

@Component
@Autowired
class PermissionsSpecInterpreter(permissionsAsCodeService: PermissionsAsCodeService,
                                 metadataProcessor: MetadataProcessor) extends SpecInterpreter with DefinitionImporter {
  override def isDefinedAt(context: InterpreterContext): Boolean = context.definition.spec.isInstanceOf[PermissionsSpec]

  override def apply(context: InterpreterContext): AsCodeResponse = {
    val result = importDefinition(ImportIncludes(), context.definition, context.scmTraceabilityData, None)

    // Not in a transaction here so just run all the actions now
    result.postCommitActions.foreach(action => action.run())

    AsCodeResponse.ids(
      result.changedIds: _*
    )
  }

  override def importDefinition(importIncludes: ImportIncludes, definition: Definition, scmData: Option[SCMTraceabilityData], validator: Option[ImportValidator]): ImportResult = {
    val metadata: Map[String, String] = metadataProcessor.processMetadata(definition.kind, definition.metadata)
    val permissionsSpec = definition.spec.asInstanceOf[PermissionsSpec]
    val home = metadata.get(MetadataFields.HOMEFOLDER.toString)
    if (importIncludes.permissions) {
      permissionsSpec.permissions.foldLeft(ImportResult(List(PERMISSION.ids))) { (acc, permission) =>
        acc.merge(permissionsAsCodeService.handlePermission(permission, home))
      }
    } else {
      ImportResult.empty
    }
  }
}
