package com.xebialabs.xlrelease.versioning.ascode

import com.xebialabs.ascode.yaml.model.{CiSpec, Definition}
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.ascode.yaml.parser.ValueTagPreprocessingConstructor.VALUE_TAG
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.delivery.Delivery
import com.xebialabs.xlrelease.domain.environments.{Environment, EnvironmentStage}
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.variables.{FolderVariables, Variable}
import com.xebialabs.xlrelease.plugins.dashboard.domain.{Dashboard, Tile}
import com.xebialabs.xlrelease.risk.domain.RiskProfile
import com.xebialabs.xlrelease.service.FolderService
import com.xebialabs.xlrelease.utils.CiHelper
import com.xebialabs.xlrelease.variable.VariableHelper
import com.xebialabs.xlrelease.versioning.ascode.ExportValidator.CiDependency
import com.xebialabs.xlrelease.versioning.ascode.ValidationMessage.LEVEL_WARN
import grizzled.slf4j.Logging
import org.springframework.util.StringUtils

import java.util
import scala.beans.BeanProperty
import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._

case class FolderInfo(id: String, relativePath: String, absolutePath: String)

case class CiInfo(@BeanProperty id: String,
                  @BeanProperty ciType: String,
                  @BeanProperty ciTypeDisplayName: String,
                  @BeanProperty folderPath: String,
                  @BeanProperty folderId: String,
                  @BeanProperty title: String) {
  def withCiId(newId: String): CiInfo = {
    this.copy(newId, ciType, ciTypeDisplayName, folderPath, folderId, title)
  }
}

object CiInfo {
  def apply(ci: ConfigurationItem, folder: FolderInfo, parentCiInfo: Option[CiInfo] = None): CiInfo = {
    new CiInfo(ci.getId, s"${ci.getType.toString}", getTypeDisplayName(ci), folder.relativePath, folder.id, getTitle(ci, parentCiInfo))
  }

  // scalastyle:off cyclomatic.complexity
  private def getTypeDisplayName(ci: ConfigurationItem): String = {
    ci match {
      case _: Task => "Task"
      case _: Phase => "Phase"
      case _: Release => "Template"
      case _: Variable => "Variable"
      case _: Configuration => "Connection"
      case _: Trigger => "Trigger"
      case _: Delivery => "Pattern"
      case _: Dashboard => "Dashboard"
      case _: Tile => "Tile"
      case _ => ci.getType.getName // Won't necessarily be user friendly but better than nothing
    }
  }

  private def getTitle(ci: ConfigurationItem, parentCiInfo: Option[CiInfo]): String = {
    val title = ci match {
      case planItem: PlanItem => planItem.getTitle
      case variable: Variable => variable.getKey
      case configuration: Configuration => configuration.getTitle
      case trigger: Trigger => trigger.getTitle
      case pattern: Delivery => pattern.getTitle
      case dashboard: Dashboard => dashboard.getTitle
      case tile: Tile => tile.getTitle
      case environment: Environment => environment.getTitle
      case environmentStage: EnvironmentStage => environmentStage.getTitle
      case _ => "Unknown"
    }

    parentCiInfo match {
      case Some(parent) => s"${parent.title}/${title}"
      case None => title
    }
  }
}

sealed trait ValidationMessage {
  def ci: CiInfo

  def property: String

  def message: String

  def messageType: String

  def messageLevel: String
}

object ValidationMessage {
  val LEVEL_ERROR = "ERROR"
  val LEVEL_WARN = "WARNING"

  def updateCiId(msg: ValidationMessage, id: String): ValidationMessage = {
    msg match {
      case HardcodedPassword(ciInfo, property) => HardcodedPassword(ciInfo.withCiId(id), property)
      case MissingPassword(ciInfo, property, placeholder) => MissingPassword(ciInfo.withCiId(id), property, placeholder)
      case NonVersionedRef(ciInfo, property, ref) => NonVersionedRef(ciInfo.withCiId(id), property, ref)
    }
  }
}

case class HardcodedPassword(ci: CiInfo, property: String) extends ValidationMessage {
  val message = "Contains a hardcoded password, please use a global or folder variable in a parent folder instead."
  val messageType = "HARDCODED_PASSWORD"
  val messageLevel: String = LEVEL_WARN
}

case class MissingPassword(ci: CiInfo, property: String, placeholder: String) extends ValidationMessage {
  val message = s"Password missing (placeholder '$placeholder' empty)"
  val messageType = "MISSING_PASSWORD"
  val messageLevel: String = LEVEL_WARN
}

case class NonVersionedRef(ci: CiInfo, property: String, ref: CiInfo) extends ValidationMessage {
  val message = "Contains a reference to an external/non-versioned CI"
  val messageType = "NON_VERSIONED_REFERENCE"
  val messageLevel: String = LEVEL_WARN
}


object ExportValidator {
  // we might need to support string dependencies later
  case class CiDependency(pd: PropertyDescriptor, references: Seq[ConfigurationItem])

  def validate(definitions: Seq[Definition], rootFolder: FolderInfo, folderService: FolderService, fromApply: Boolean): Seq[ValidationMessage] = {
    new ExportValidator(definitions, rootFolder, folderService, fromApply).validate()
  }
}

class ExportValidator(definitions: Seq[Definition], rootFolder: FolderInfo, folderService: FolderService, fromApply: Boolean) extends Logging {
  private val allCisInDefinitions = definitions.collect {
    case Definition(_, _, _, CiSpec(cis, _, _)) =>
      CiHelper.getNestedCis(cis.asJava).asScala
  }.flatten.toSet

  def validate(): Seq[ValidationMessage] = {
    definitions.foldLeft(Seq.empty[ValidationMessage]) {
      case (warnings, definition) => warnings ++ validateDefinition(definition, rootFolder)
    }
  }

  private def validateDefinition(definition: Definition, rootFolder: FolderInfo): Seq[ValidationMessage] = {
    definition.spec match {
      case spec: CiSpec => validateCis(spec.cis, rootFolder)
      case _ => Seq.empty[ValidationMessage]
    }
  }

  protected def validateCis(cis: List[ConfigurationItem], parentFolder: FolderInfo): Seq[ValidationMessage] = {
    cis.foldLeft(Seq.empty[ValidationMessage]) {
      case (warnings, ci) => warnings ++ validateCi(ci, parentFolder)
    }
  }

  def validateCi(ci: ConfigurationItem, parentFolder: FolderInfo): Seq[ValidationMessage] = {
    ci match {
      case template: Release => validateTemplate(template, parentFolder)
      case folderVariables: FolderVariables => validateVariables(folderVariables.getVariables.asScala.toList, parentFolder)
      case folder: Folder =>
        val nestedCis = folder.getChildren.asInstanceOf[java.util.Set[ConfigurationItem]].asScala.toList
        val relPath = if (parentFolder.relativePath.isEmpty()) folder.getTitle else s"${parentFolder.relativePath}/${folder.getTitle}"
        val absPath = s"${parentFolder.absolutePath}/${folder.getTitle}"
        validateCis(nestedCis, FolderInfo(folder.getId(), relPath, absPath))
      case dashboard: Dashboard => validateDashboard(dashboard, parentFolder)
      case _ =>
        val info = CiInfo(ci, parentFolder)
        validateCiProperties(ci, info)
    }
  }

  protected def validateCiProperties(topLevelCi: ConfigurationItem, ciInfo: CiInfo): Seq[ValidationMessage] = {
    val nestedCisWithDeps = getExternalReferencesByDescriptorsByCis(topLevelCi)
    validateCiProperties(nestedCisWithDeps, ciInfo)
  }

  protected def validateCiProperties(nestedCisWithDeps: Iterable[(ConfigurationItem, List[CiDependency])], ciInfo: CiInfo): Seq[ValidationMessage] = {
    val (cis, dependencies) = nestedCisWithDeps.unzip
    val passValidations = validatePasswords(cis, ciInfo)
    val depsValidations = dependencies.flatMap(_.flatMap(checkDeps(_, ciInfo)))
    val crtValidation = validateCRT(cis, ciInfo)
    passValidations.concat(depsValidations).concat(crtValidation)
  }

  protected def validatePasswords(cis: Iterable[ConfigurationItem], ciInfo: CiInfo): List[ValidationMessage] = {
    cis.flatMap { nestedCi =>
      val properties = nestedCi.getType.getDescriptor.getPropertyDescriptors.asScala
      properties.flatMap(validatePasswords(nestedCi, _, ciInfo))
    }.toList
  }

  // scalastyle:off cyclomatic.complexity
  protected def validatePasswords(ci: ConfigurationItem, pd: PropertyDescriptor, ciInfo: CiInfo): List[ValidationMessage] = {
    val validationMessages = ListBuffer.empty[ValidationMessage]
    val value = pd.get(ci)
    if (pd.isPassword && value != null) {
      pd.getKind match {
        case PropertyKind.STRING =>
          val usesPlainPassword = containsPassword(pd, value.asInstanceOf[String])
          if (usesPlainPassword) {
            validationMessages.addOne(HardcodedPassword(ciInfo, pd.getFqn))
          }
        case PropertyKind.SET_OF_STRING | PropertyKind.LIST_OF_STRING =>
          val usesPlainPassword = value.asInstanceOf[java.util.Collection[String]].asScala.exists { v =>
            containsPassword(pd, v)
          }
          if (usesPlainPassword) {
            validationMessages.addOne(HardcodedPassword(ciInfo, pd.getFqn))
          }
        case PropertyKind.MAP_STRING_STRING =>
          val usesPlainPassword = value.asInstanceOf[java.util.Map[String, String]].asScala.exists { case (_, v) =>
            containsPassword(pd, v)
          }
          if (usesPlainPassword) {
            validationMessages.addOne(HardcodedPassword(ciInfo, pd.getFqn))
          }
        case _ =>
      }
    }

    validationMessages.toList
  }

  private def checkDeps(ciDependency: CiDependency, ciInfo: CiInfo) = {
    val validationMessages = ListBuffer.empty[ValidationMessage]

    ciDependency.references.foreach { ref =>
      //Using exists instead of contains,Seems like the check is done on stale hashcode or something
      if (!this.allCisInDefinitions.exists(cis => cis.equals(ref))) {
        ref match {
          // don't care about the default risk profile
          case rp: RiskProfile if rp.getId == RiskProfile.DEFAULT_RISK_PROFILE_ID => ()
          case _ =>
            // not sure if all of these types are actually relevant
            val refFolderId = ref match {
              case r: Release => r.findFolderId()
              case c: BaseConfiguration => c.getFolderId
              case t: Trigger => t.getFolderId
              case d: Delivery => d.getFolderId
              case d: Dashboard => d.getParentId
              case f: Folder => f.getId
              case e: EnvironmentStage => ciInfo.folderId
              case _ => null

            }
            // todo check if folder paths are actually needed for ref
            val refFolderInfo = FolderInfo(refFolderId, null, null)
            val refCiInfo = CiInfo(ref, refFolderInfo)
            validationMessages.addOne(NonVersionedRef(ciInfo, ciDependency.pd.getFqn, refCiInfo))
        }
      }
    }

    validationMessages.toList
  }

  private def validateCRT(cis: Iterable[ConfigurationItem], ciInfo: CiInfo): List[ValidationMessage] = {
    val validationMessages = ListBuffer.empty[ValidationMessage]

    if (ciInfo.ciType.equals("xlrelease.CreateReleaseTask")) {
      val property = cis.head.getType.getDescriptor.getPropertyDescriptors.asScala.filter(p => p.getName.equals("templateId")).head
      val createReleaseTask = cis.head.asInstanceOf[CreateReleaseTask]

      if (fromApply) {
        handleCRTValidationFromApplyVersion(createReleaseTask, ciInfo, property, validationMessages)
      } else {
        handleCRTValidationFromCreateVersion(createReleaseTask, ciInfo, property, validationMessages)
      }
    }

    validationMessages.toList
  }

  private def handleCRTValidationFromApplyVersion(task: CreateReleaseTask, ciInfo: CiInfo, prop: PropertyDescriptor, mess: ListBuffer[ValidationMessage]) {
    val folderId = task.getFolderId.split("/").take(2).mkString("/")
    if (task.getTemplateId == null && !ciInfo.getId.contains(folderId)) {
      mess.addOne(NonVersionedRef(ciInfo, prop.getFqn, null)) // applying missing template outside the scope
    }
    if (task.getTemplateId != null && !ciInfo.getId.contains(folderId)) {
      val refCiInfo: CiInfo = CiInfo(
        task.getTemplateId, "xlrelease.Release", "Template", "", task.getTemplateId.split("/Release").head, ""
      )
      mess.addOne(NonVersionedRef(ciInfo, prop.getFqn, refCiInfo))  // applying existing template outside the scope
    }
  }

  private def handleCRTValidationFromCreateVersion(task: CreateReleaseTask, ciInfo: CiInfo, prop: PropertyDescriptor, mess: ListBuffer[ValidationMessage]) {
    val folderId = task.getFolderId
    if (task.getTemplateId != null && !folderId.startsWith("./") && !folderId.startsWith("Applications/")) {
      val folder = folderService.findByPath(folderId, folderId.split("/").length - 1)
      val release = folderService.returnReleaseByFolderIdAndReleaseTitle(
        folder.getId.replace("Applications/", ""),
        task.getTemplateId.split("/").last)
      val refCiInfo: CiInfo = CiInfo(release.getId, "xlrelease.Release", "Template", "", folder.getId, release.getTitle)
      mess.addOne(NonVersionedRef(ciInfo, prop.getFqn, refCiInfo)) // creating existing template outside the scope
    }
  }

  private def containsPassword(pd: PropertyDescriptor, value: String): Boolean = {
    !(value.isEmpty ||
      VariableHelper.containsVariables(value) ||
      Option(pd.getDefaultValue).map(_.asInstanceOf[String]).contains(value))
  }

  private def validateTemplate(template: Release, parentFolder: FolderInfo): Seq[ValidationMessage] = {
    val info = CiInfo(template, parentFolder)
    // Pull the phases out of the template so we only check the template properties first
    val phases = template.getPhases
    template.setPhases(List.empty[Phase].asJava)

    // Pull the variables out of the template so we can validate those separately
    val variables = template.getVariables
    template.setVariables(List.empty[Variable].asJava)

    val templateWarnings = phases.asScala.foldLeft(validateCiProperties(template, info)) {
      case (phaseWarnings, phase) => {
        val phaseInfo = CiInfo(phase, parentFolder, Some(info))
        // Validate the tasks in the phase as separate CIs
        phaseWarnings ++ phase.getTasks.asScala.foldLeft(Seq.empty[ValidationMessage]) {
          case (taskWarnings, task) => {
            val taskInfo = CiInfo(task, parentFolder, Some(phaseInfo))
            taskWarnings ++ validateTask(task, taskInfo, parentFolder)
          }
        }
      }
    }

    val variableWarnings = validateVariables(variables.asScala.toList, parentFolder, Some(info))

    // Restore the phases in the template
    template.setPhases(phases)
    template.setVariables(variables)
    templateWarnings ++ variableWarnings
  }

  private def validateTask(task: Task, info: CiInfo, parentFolder: FolderInfo): Seq[ValidationMessage] = {
    task match {
      case group: TaskGroup =>
        // Pull the tasks out of the group so we only check the group properties first
        val tasks = group.getTasks
        group.setTasks(new util.ArrayList[Task])

        val groupWarnings = tasks.asScala.foldLeft(validateCiProperties(group, info)) {
          // Validate the subtasks in the group as separate CIs
          case (taskWarnings, subtask) => {
            val subtaskInfo = CiInfo(subtask, parentFolder, Some(info))
            taskWarnings ++ validateTask(subtask, subtaskInfo, parentFolder)
          }
        }

        // Restore the tasks in the group, sequential groups include validation on addTasks so add by index
        tasks.asScala.zipWithIndex.foreach {
          case (task, i) => group.addTask(task, i)
        }
        groupWarnings
      case _ => validateCiProperties(task, info)
    }
  }

  private def validateVariables(variables: List[Variable], parentFolder: FolderInfo, templateInfo: Option[CiInfo] = None): Seq[ValidationMessage] = {
    variables.foldLeft(Seq.empty[ValidationMessage]) {
      case (warnings, variable) => {
        val info = CiInfo(variable, parentFolder, templateInfo)
        warnings ++ validateCiProperties(variable, info)
      }
    }
  }

  private def validateDashboard(dashboard: Dashboard, parentFolder: FolderInfo): Seq[ValidationMessage] = {
    val info = CiInfo(dashboard, parentFolder)
    val tiles = dashboard.getTiles
    dashboard.setTiles(List.empty[Tile].asJava)

    val dashboardWarnings = tiles.asScala.foldLeft(validateCiProperties(dashboard, info)) {
      case (tileWarnings, tile) => {
        // Validate the tiles in the phase as separate CIs
        val tileInfo = CiInfo(tile, parentFolder, Some(info))
        tileWarnings ++ validateTile(tile, tileInfo)
      }
    }

    dashboard.setTiles(tiles)
    dashboardWarnings
  }

  private def validateTile(tile: Tile, info: CiInfo): Seq[ValidationMessage] = {
    validateCiProperties(tile, info)
  }

  private def getExternalReferencesByDescriptorsByCis(parentCi: ConfigurationItem): List[(ConfigurationItem, List[CiDependency])] = {
    CiHelper.getNestedCis(parentCi).asScala.map { ci =>
      val ciDependencies = ci.getType.getDescriptor.getPropertyDescriptors.asScala.view
        .map(pd => pd -> pd.get(ci))
        .filter { case (pd, value) => !pd.isAsContainment && value != null }
        .flatMap { case (pd, value) =>
          pd.getKind match {
            case PropertyKind.CI =>
              val valueCi = value.asInstanceOf[ConfigurationItem]
              if (!CiHelper.isChildViaOneOfChildProperties(valueCi, ci)) {
                Some(CiDependency(pd, List(valueCi)))
              } else {
                None
              }
            case PropertyKind.SET_OF_CI | PropertyKind.LIST_OF_CI =>
              val valueCiCol = value.asInstanceOf[java.util.Collection[ConfigurationItem]].asScala.toList
              if (valueCiCol.nonEmpty) {
                Some(CiDependency(pd, valueCiCol))
              } else {
                None
              }
            case _ => None
          }
        }
      ci -> ciDependencies.toList
    }.toList
  }
}

class ImportValidator(secrets: Map[String, String], definitions: Seq[Definition], rootFolder: FolderInfo, folderService: FolderService, fromApply: Boolean)
  extends ExportValidator(definitions, rootFolder, folderService, fromApply) {

  override protected def validatePasswords(ci: ConfigurationItem, pd: PropertyDescriptor, ciInfo: CiInfo): List[ValidationMessage] = {
    val validationMessages = ListBuffer.empty[ValidationMessage]

    def processSecret(str: String) = {
      getPlaceholder(str) match {
        case Some(placeholder) =>
          secrets.get(placeholder) match {
            case Some(secret) => secret
            case None =>
              logger.debug(s"Property on ${pd.getFqn} of $ciInfo has a missing secret (Missing placeholder name '$placeholder')")
              validationMessages.addOne(MissingPassword(ciInfo, pd.getFqn, placeholder))
              null
          }
        case None => str
      }
    }

    val value = pd.get(ci)
    if (pd.isPassword && value != null) {
      pd.getKind match {
        case PropertyKind.STRING =>
          val processedSecret = processSecret(value.asInstanceOf[String])
          pd.set(ci, processedSecret)
        case PropertyKind.LIST_OF_STRING =>
          val listval = value.asInstanceOf[java.util.List[String]]
          val processedSecrets = listval.asScala.map(processSecret)
          pd.set(ci, processedSecrets.asJava)
        case PropertyKind.SET_OF_STRING =>
          val setval = value.asInstanceOf[java.util.Set[String]]
          val processedSecrets = setval.asScala.map(processSecret)
          pd.set(ci, processedSecrets.asJava)
        case PropertyKind.MAP_STRING_STRING =>
          val mapval = value.asInstanceOf[java.util.Map[String, String]]
          val processedSecrets = mapval.asScala.view.mapValues(processSecret).toMap
          pd.set(ci, processedSecrets.asJava)
        case _ =>
      }
    }
    validationMessages.toList
  }


  private def getPlaceholder(value: String) = {
    if (value.startsWith(s"$VALUE_TAG ")) {
      val placeholder = value.substring(s"$VALUE_TAG ".length)
      Some(placeholder).filter(StringUtils.hasText)
    } else {
      None
    }
  }
}
