package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.ChangedIds
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds.{CI, _}
import com.xebialabs.xlrelease.api.internal.CalendarResource
import com.xebialabs.xlrelease.ascode.yaml.model.CalendarAsCode
import com.xebialabs.xlrelease.domain.calendar.SpecialDay
import grizzled.slf4j.Logging
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.text.SimpleDateFormat
import java.util.Date
import scala.collection.mutable.ListBuffer

@Service
class CalendarAsCodeService @Autowired()(calendarResource: CalendarResource) extends Logging {
  def updateCalendarEntries(calendarAsCode: CalendarAsCode): ChangedIds = {
    var createdCalendarEntries = new ListBuffer[String]
    var updatedCalendarEntries = new ListBuffer[String]

    calendarAsCode.calendar.forEach { c =>
      val now = new Date()
      val normalizedNow = new DateTime(now).withTimeAtStartOfDay().toDate

      c.blackouts.forEach { blackout =>
        if (blackout.getEndDate.before(now)) {
          throw new AsCodeException(
            s"Blackout ${blackout.getLabel} from [${blackout.getStartDate}] to [${blackout.getEndDate}] must end after the current date"
          )
        }
      }

      c.specialDays.forEach { specialDay =>
        // normalize startDate for comparison
        specialDay.setStartDate(new DateTime(specialDay.getStartDate).withTimeAtStartOfDay.toDate)
        if (specialDay.getStartDate.before(normalizedNow)) {
          throw new AsCodeException(
            s"Special day ${specialDay.getLabel} at [${specialDay.getStartDate}] must be today or future date"
          )
        }
      }

      if (!c.blackouts.isEmpty) {
        logger.debug(s"Updating calendar blackouts")

        c.blackouts.forEach { blackout =>
          calendarResource.addBlackout(blackout)
          createdCalendarEntries += blackout.getId
        }
      }

      if (!c.specialDays.isEmpty) {
        logger.debug(s"Updating calendar special days")

        val dateFormat = new SimpleDateFormat(SpecialDay.DATE_FORMAT)


        c.specialDays.forEach { specialDay =>
          val newSpecialDay = new SpecialDay(specialDay.getLabel, specialDay.getColor)
          val id = dateFormat.format(specialDay.getStartDate)
          val specialDayId = s"Configuration/Calendar/$id"

          calendarResource.setSpecialDay(specialDayId, newSpecialDay)
          updatedCalendarEntries += specialDayId
        }
      }
    }

    CI.ids.withCreated(createdCalendarEntries.toList).withUpdated(updatedCalendarEntries.toList)
  }
}
